
// __________________________________________________________
//
//                         sbs_rtl.c
//                 SBS Runtime Library V1.05
//                02-10-2003 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#define  _INLINE_SBSPRINT_
#define  _INLINE_SBSNUM_
#define  _SBS_RTL_DLL_
#include "sbs_rtl.h"

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantibility and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

09-20-2001 V1.00 Original version (SBS).

12-13-2001 V1.01 Bugfix/upgrade (SBS).

    Removed a potential infinite loop inside SbsListFind(). Fixed
    SbsListInsert() to allow automatic list sorting. If a string is
    appended to the end of the list, SBS_ADDSTRING is used instead
    of SBS_INSERTSTRING, because this message recognizes the
    LBS_SORT and CBS_SORT styles. Added SbsListSelectEx(), allowing
    to move a list selection without sending LBN_SELCHANGE or
    CBN_SELCHANGE notifications. Added the transparent ANSI/Unicode
    macro substitutes IDI, IDC, LCASE, and UCASE.

02-26-2002 V1.02 Bugfix/upgrade (SBS).

    In SbsListCreateEx(), psl->hcDefault is set to ghcDefault
    instead of GetCursor(). Before this fix, the hourglass cursor
    tended to remain visible after a simultaneous update of multiple
    lists. Added SbsListCloseEx(), allowing to close a list
    without sending LBN_SELCHANGE or CBN_SELCHANGE notifications.
    Added SbsFontFixedsys(), SbsFontTerminal(), and SbsFontSymbol().

03-11-2002 V1.03 Bugfix (SBS).

    Fixed the RegOpenKeyEx() call inside SbsRegistryOpen(),
    replacing KEY_READ by KEY_ALL_ACCESS. Fixed the RegCreateKeyEx()
    call inside SbsRegistryCreate(), replacing KEY_WRITE by
    KEY_ALL_ACCESS. SbsRegistryDeleteValue() returns success if the
    specified key/value doesn't exist.

04-05-2002 V1.04 Bugfix (SBS).

    SbsListCloseEx() and SbsListClear(): Changed the "BOOL fRedraw"
    argument of the SbsListUpdate() calls to TRUE, thus forcing
    proper refresh of multiple dependent list controls.

02-10-2003 V1.05 Upgrade (SBS).

    Replaced the CRC32 and pseudo-random functions by highly
    optimized assembly language code, comprised by the new header
    file SbsNum.h. Most of the old-fashion API functions SbsCrc32*()
    and SbsRandom*() have become wrappers around the new ones.

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

HINSTANCE ghInstance   = NULL;
HFONT     ghfDefault   = NULL;
HBRUSH    ghbDefault   = NULL;
HICON     ghiDefault   = NULL;
HCURSOR   ghcDefault   = NULL;
HCURSOR   ghcWait      = NULL;

HFONT     ghfLarge     = NULL;
HFONT     ghfSmall     = NULL;
HFONT     ghfLight     = NULL;

FILETIME  gftNull      = {0, 0};

// =================================================================
// GLOBAL STRINGS
// =================================================================

BYTE gabWindow              [] =  "Window";
WORD gawWindow              [] = L"Window";

BYTE gabCaption             [] = SA(MAIN_CAPTION);
WORD gawCaption             [] = SW(MAIN_CAPTION);

BYTE gabFilename            [] = SA(MAIN_FILENAME);
WORD gawFilename            [] = SW(MAIN_FILENAME);

BYTE gabSbsClass            [] =  "SBS_CLASS_ANSI";
WORD gawSbsClass            [] = L"SBS_CLASS_UNICODE";

BYTE gabFont                [] =  "System";
WORD gawFont                [] = L"System";

BYTE gabHex                 [] =  "0123456789ABCDEF";
WORD gawHex                 [] = L"0123456789ABCDEF";

BYTE gabVowels              [] =  "aeiou";
WORD gawVowels              [] = L"aeiou";

BYTE gabConsonants          [] =  "bcdfghjklmnpqrstvwxyz";
WORD gawConsonants          [] = L"bcdfghjklmnpqrstvwxyz";

BYTE gabIni                 [] =  ".ini";
WORD gawIni                 [] = L".ini";

BYTE gabBmp                 [] =  ".bmp";
WORD gawBmp                 [] = L".bmp";

BYTE gabPattern             [] =  "*";
WORD gawPattern             [] = L"*";

BYTE gabTrue                [] =  "TRUE";
WORD gawTrue                [] = L"TRUE";

BYTE gabFalse               [] =  "FALSE";
WORD gawFalse               [] = L"FALSE";

BYTE gabNewLine             [] =  "\r\n";
WORD gawNewLine             [] = L"\r\n";

BYTE gabString              [] =  "%s";
WORD gawString              [] = L"%s";

BYTE gabNull                [] =  "";
WORD gawNull                [] = L"";

// =================================================================
// INI SETTINGS
// =================================================================

BYTE gabSettings            [] =  "Settings";
BYTE gabSettingsColumn      [] =  "Column";
BYTE gabSettingsRow         [] =  "Row";
BYTE gabSettingsWidth       [] =  "Width";
BYTE gabSettingsHeight      [] =  "Height";
BYTE gabSettingsFont        [] =  "Font";
BYTE gabSettingsColor       [] =  "Color";
BYTE gabSettingsBrush       [] =  "Brush";
BYTE gabSettingsLanguage    [] =  "Language";

WORD gawSettings            [] = L"Settings";
WORD gawSettingsColumn      [] = L"Column";
WORD gawSettingsRow         [] = L"Row";
WORD gawSettingsWidth       [] = L"Width";
WORD gawSettingsHeight      [] = L"Height";
WORD gawSettingsFont        [] = L"Font";
WORD gawSettingsColor       [] = L"Color";
WORD gawSettingsBrush       [] = L"Brush";
WORD gawSettingsLanguage    [] = L"Language";

BYTE gabOptions             [] =  "Options";
WORD gawOptions             [] = L"Options";

// =================================================================
// REGISTRY ROOT NAMES
// =================================================================

BYTE gabHkeyClassesRoot     [] =  "HKCR\0HKEY_CLASSES_ROOT";
BYTE gabHkeyCurrentUser     [] =  "HKCU\0HKEY_CURRENT_USER";
BYTE gabHkeyLocalMachine    [] =  "HKLM\0HKEY_LOCAL_MACHINE";
BYTE gabHkeyUsers           [] =  "HKU\0HKEY_USERS";
BYTE gabHkeyPerformanceData [] =  "HKPD\0HKEY_PERFORMANCE_DATA";
BYTE gabHkeyCurrentConfig   [] =  "HKCC\0HKEY_CURRENT_CONFIG";
BYTE gabHkeyDynData         [] =  "HKDD\0HKEY_DYN_DATA";

WORD gawHkeyClassesRoot     [] = L"HKCR\0HKEY_CLASSES_ROOT";
WORD gawHkeyCurrentUser     [] = L"HKCU\0HKEY_CURRENT_USER";
WORD gawHkeyLocalMachine    [] = L"HKLM\0HKEY_LOCAL_MACHINE";
WORD gawHkeyUsers           [] = L"HKU\0HKEY_USERS";
WORD gawHkeyPerformanceData [] = L"HKPD\0HKEY_PERFORMANCE_DATA";
WORD gawHkeyCurrentConfig   [] = L"HKCC\0HKEY_CURRENT_CONFIG";
WORD gawHkeyDynData         [] = L"HKDD\0HKEY_DYN_DATA";

// =================================================================
// COOKIE INFORMATION
// =================================================================

BYTE gabCookieA    [] =  "%s\n%s\n%s\n%lu\n%lu\n%lu\n%lu\n%lu\n*\n";
BYTE gabCookieW    [] =  "%S\n%S\n%S\n%lu\n%lu\n%lu\n%lu\n%lu\n*\n";
WORD gawCookie     [] = L"%s\n%s\n%s\n%lu\n%lu\n%lu\n%lu\n%lu\n*\n";

BYTE gabCookieFile [] =  "%s@%s[%s].%s";
WORD gawCookieFile [] = L"%s@%s[%s].%s";

BYTE gabCookieExt  [] =  "txt";
WORD gawCookieExt  [] = L"txt";

BYTE gabCookieType [] =  "Cookie:";
WORD gawCookieType [] = L"Cookie:";

BYTE gabCookieName [] =  "Cookie:%s@%s";
WORD gawCookieName [] = L"Cookie:%s@%s";

BYTE gabCookieTemp [] =  "Cookie:_SBS_COOKIE_TEMPORARY_";
WORD gawCookieTemp [] = L"Cookie:_SBS_COOKIE_TEMPORARY_";

BYTE gabCookieBase [] =  "HKEY_CURRENT_USER\\Software\\"
                         "Microsoft\\Windows\\CurrentVersion\\"
                         "Explorer\\Shell Folders\\"
                         "Cookies";

WORD gawCookieBase [] = L"HKEY_CURRENT_USER\\Software\\"
                        L"Microsoft\\Windows\\CurrentVersion\\"
                        L"Explorer\\Shell Folders\\"
                        L"Cookies";

// =================================================================
// CODEC TABLES
// =================================================================

#define SBS_URL_ESCAPE_FIRST 0x20
#define SBS_URL_ESCAPE_LAST  0x7E

BYTE gabUrlEscape [] =  "\"#%&'+/:;<=>?@[\\]^`{|}~"; // cf. RFC 1738

// -----------------------------------------------------------------

#define SBS_INI_NULL   ';'
#define SBS_INI_SPACE  '+'
#define SBS_INI_ESCAPE '%'

BYTE gabIniEscape [] = {SBS_INI_NULL, SBS_INI_SPACE, SBS_INI_ESCAPE,
                        "[]\"'=&"};

// =================================================================
// COLOR PALETTES
// =================================================================

COLORREF gacrGrayShades [16] =
    {
    RGB (0x00, 0x00, 0x00), RGB (0x11, 0x11, 0x11),
    RGB (0x22, 0x22, 0x22), RGB (0x33, 0x33, 0x33),
    RGB (0x44, 0x44, 0x44), RGB (0x55, 0x55, 0x55),
    RGB (0x66, 0x66, 0x66), RGB (0x77, 0x77, 0x77),
    RGB (0x88, 0x88, 0x88), RGB (0x99, 0x99, 0x99),
    RGB (0xAA, 0xAA, 0xAA), RGB (0xBB, 0xBB, 0xBB),
    RGB (0xCC, 0xCC, 0xCC), RGB (0xDD, 0xDD, 0xDD),
    RGB (0xEE, 0xEE, 0xEE), RGB (0xFF, 0xFF, 0xFF)
    };

// =================================================================
// DATE/TIME DEFINITIONS
// =================================================================

BYTE gabTimeUS [] =  "%02hu-%02hu-%04hu %02hu:%02hu:%02hu";
WORD gawTimeUS [] = L"%02hu-%02hu-%04hu %02hu:%02hu:%02hu";

BYTE gabTimeDE [] =  "%02hu.%02hu.%04hu %02hu:%02hu:%02hu";
WORD gawTimeDE [] = L"%02hu.%02hu.%04hu %02hu:%02hu:%02hu";

DWORD adDaysPerMonth [] = {31,28,31,30,31,30,31,31,30,31,30,31};

// =================================================================
// GUID DEFINITIONS
// =================================================================

BYTE gabGuid   [] =  "{%08lX-%04lX-%04lX"
                     "-%02lX%02lX-%02lX%02lX%02lX%02lX%02lX%02lX}";

WORD gawGuid   [] = L"{%08lX-%04lX-%04lX"
                    L"-%02lX%02lX-%02lX%02lX%02lX%02lX%02lX%02lX}";

// =================================================================
// CODE PAGE DIRECTORY
// =================================================================

VS_CODE_PAGE gavcpCodePages [] =
    {
        0, "Code Page Independent",
       37, "EBCDIC",
      437, "MS-DOS United States",
      500, "EBCDIC \"500V1\"",
      708, "Arabic (ASMO 708)",
      709, "Arabic (ASMO 449+, BCON V4)",
      710, "Arabic (Transparent Arabic)",
      720, "Arabic (Transparent ASMO)",
      737, "Greek (formerly 437G)",
      775, "Baltic",
      850, "MS-DOS Multilingual (Latin I)",
      852, "MS-DOS Slavic (Latin II)",
      855, "IBM Cyrillic (primarily Russian)",
      857, "IBM Turkish",
      860, "MS-DOS Portuguese",
      861, "MS-DOS Icelandic",
      862, "Hebrew",
      863, "MS-DOS Canadian-French",
      864, "Arabic",
      865, "MS-DOS Nordic",
      866, "MS-DOS Russian",
      869, "IBM Modern Greek",
      874, "Thai",
      875, "EBCDIC",
      932, "Japan",
      936, "Chinese (PRC, Singapore)",
      949, "Korean",
      950, "Chinese (Taiwan, Hong Kong) ",
     1026, "EBCDIC",
     1200, "Unicode (BMP of ISO 10646)",
     1250, "Windows 3.1 Eastern European ",
     1251, "Windows 3.1 Cyrillic",
     1252, "Windows 3.1 US (ANSI)",
     1253, "Windows 3.1 Greek",
     1254, "Windows 3.1 Turkish",
     1255, "Hebrew",
     1256, "Arabic",
     1257, "Baltic",
     1258, "Vietnamese",
     1361, "Korean (Johab)",
    10000, "Macintosh Roman",
    10001, "Macintosh Japanese",
    10006, "Macintosh Greek I",
    10007, "Macintosh Cyrillic",
    10029, "Macintosh Latin 2",
    10079, "Macintosh Icelandic",
    10081, "Macintosh Turkish",
    10082, "Macintosh Croatian",
    20261, "Teletex",
    20866, "Cyrillic (KOI8-R)",
    28592, "Central European (ISO)",
    28595, "Cyrillic (ISO)",
    50000, "User Defined",
       -1, NULL,
    };

// =================================================================
// TEXT CONVERSION ROUTINES
// =================================================================

PWORD WINAPI SbsConvertA2W (PBYTE pbData)
    {
    PWORD pwData = _tnewA2W (pbData, -1);

    _mfree (pbData);
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsConvertW2A (PWORD pwData)
    {
    PBYTE pbData = _tnewW2A (pwData, -1);

    _mfree (pwData);
    return pbData;
    }

// =================================================================
// SIZE COMPUTATION
// =================================================================

DWORD WINAPI SbsSizeDecimal (DWORD dNumber)
    {
    DWORD dNumber1 = dNumber;
    DWORD dSize    = 0;

    do dSize++; while (dNumber1 /= 10);
    return dSize;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsSizeDivide (DWORD dShift,
                            DWORD dSize,
                            BOOL  fRoundUp,
                            BOOL  fRoundDown)
    {
    DWORD dFactor;
    DWORD dSize1 = (dShift ? 0 : dSize);

    if (dShift && (dShift < 32))
        {
        dFactor = 1     << dShift;
        dSize1  = dSize >> dShift;

        if ((dSize & (dFactor-1)) && (fRoundUp || !fRoundDown))
            {
            if ((fRoundUp && !fRoundDown) ||
                (dSize & (dFactor >> 1))) dSize1++;
            }
        }
    return dSize1;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsSizeKB (DWORD dBytes,
                        BOOL  fRoundUp,
                        BOOL  fRoundDown)
    {
    return SbsSizeDivide (10, dBytes, fRoundUp, fRoundDown);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsSizeMB (DWORD dBytes,
                        BOOL  fRoundUp,
                        BOOL  fRoundDown)
    {
    return SbsSizeDivide (20, dBytes, fRoundUp, fRoundDown);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsSizePercent (DWORD dPart,
                             DWORD dTotal)
    {
    DWORD dPart1   = min (dPart, dTotal);
    DWORD dPercent = 100;

    if (dTotal)
        {
        if (dPart1 <= MAXDWORD / 100)
            {
            dPercent = (dPart1 * 100) / dTotal;
            }
        else
            {
            dPercent = (dPart1 / (dTotal / 100));
            }
        }
    return dPercent;
    }

// =================================================================
// DECIMAL CONVERSION ROUTINES
// =================================================================

DWORD WINAPI SbsDecimalReadA (PBYTE  pbData,
                              PDWORD pdData)
    {
    BOOL  fSign;
    DWORD i, j, n;
    DWORD dData = SAFEVAL (pdData, 0);
    DWORD dSize = 0;

    if (pbData != NULL)
        {
        i = j = n = 0;
        while ((pbData [i] == ' ') || (pbData [i] == '\t')) i++;

        if (fSign = (pbData [i] == '-')) i++;

        while ((pbData [i] >= '0') && (pbData [i] <= '9'))
            {
            n *= 10;
            n += (pbData [i++] - '0');
            j++;
            }
        if (j) dData = (fSign ? 0 - n : n);
        dSize = i;
        }
    if (pdData != NULL) *pdData = dData;
    return dSize;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsDecimalReadW (PWORD  pwData,
                              PDWORD pdData)
    {
    BOOL  fSign;
    DWORD i, j, n;
    DWORD dData = SAFEVAL (pdData, 0);
    DWORD dSize = 0;

    if (pwData != NULL)
        {
        i = j = n = 0;
        while ((pwData [i] == ' ') || (pwData [i] == '\t')) i++;

        if (fSign = (pwData [i] == '-')) i++;

        while ((pwData [i] >= '0') && (pwData [i] <= '9'))
            {
            n *= 10;
            n += (pwData [i++] - '0');
            j++;
            }
        if (j) dData = (fSign ? 0 - n : n);
        dSize = i;
        }
    if (pdData != NULL) *pdData = dData;
    return dSize;
    }

// =================================================================
// HEX CONVERSION ROUTINES
// =================================================================

BOOL WINAPI SbsHexTestA (BYTE bData)
    {
    return SbsHexTestW (bData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsHexTestW (WORD wData)
    {
    return ((wData >= '0') && (wData <= '9')) ||
           ((wData >= 'A') && (wData <= 'F')) ||
           ((wData >= 'a') && (wData <= 'f'));
    }

// -----------------------------------------------------------------

BYTE WINAPI SbsHexNibbleA (BYTE bData)
    {
    return SbsHexNibbleW (bData);
    }

// -----------------------------------------------------------------

BYTE WINAPI SbsHexNibbleW (WORD wData)
    {
    return (BYTE) ((wData >= '0') && (wData <= '9')
                   ? wData - '0'
                   : ((wData >= 'A') && (wData <= 'F')
                      ? wData - 'A' + 10
                      : ((wData >= 'a') && (wData <= 'f')
                         ? wData - 'a' + 10
                         : 0)));
    }

// -----------------------------------------------------------------

BYTE WINAPI SbsHexByteA (BYTE bHigh,
                         BYTE bLow)
    {
    return (SbsHexNibbleA (bHigh) << 4) | SbsHexNibbleA (bLow);
    }

// -----------------------------------------------------------------

BYTE WINAPI SbsHexByteW (WORD wHigh,
                         WORD wLow)
    {
    return (SbsHexNibbleW (wHigh) << 4) | SbsHexNibbleW (wLow);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexBytesA (PBYTE pbHex,
                           PVOID pData,
                           DWORD dData)
    {
    BYTE  bLow, bHigh;
    DWORD i;
    DWORD n = 0;

    if (pbHex != NULL)
        {
        for (i = 0; i < dData; i++)
            {
            bHigh = bLow = '0';

            if (SbsHexTestA (pbHex [n]))
                {
                bHigh = pbHex [n++];

                if (SbsHexTestA (pbHex [n]))
                    {
                    bLow = pbHex [n++];
                    }
                }
            if (pData != NULL)
                {
                ((PBYTE) pData) [i] = SbsHexByteA (bHigh, bLow);
                }
            }
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexBytesW (PWORD pwHex,
                           PVOID pData,
                           DWORD dData)
    {
    WORD  wLow, wHigh;
    DWORD i;
    DWORD n = 0;

    if (pwHex != NULL)
        {
        for (i = 0; i < dData; i++)
            {
            wHigh = wLow = '0';

            if (SbsHexTestW (pwHex [n]))
                {
                wHigh = pwHex [n++];

                if (SbsHexTestW (pwHex [n]))
                    {
                    wLow = pwHex [n++];
                    }
                }
            if (pData != NULL)
                {
                ((PBYTE) pData) [i] = SbsHexByteW (wHigh, wLow);
                }
            }
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexWordA (PBYTE pbHex,
                          PWORD pwData)
    {
    DWORD n     = 0;
    WORD  wData = 0;

    if (pbHex != NULL)
        {
        while ((n < WORD_ * 2) && (SbsHexTestA (pbHex [n])))
            {
            wData <<= 4;
            wData  += SbsHexNibbleA (pbHex [n++]);
            }
        }
    if (pwData != NULL) *pwData = wData;
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexWordW (PWORD pwHex,
                          PWORD pwData)
    {
    DWORD n     = 0;
    WORD  wData = 0;

    if (pwHex != NULL)
        {
        while ((n < WORD_ * 2) && (SbsHexTestW (pwHex [n])))
            {
            wData <<= 4;
            wData  += SbsHexNibbleW (pwHex [n++]);
            }
        }
    if (pwData != NULL) *pwData = wData;
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexDwordA (PBYTE  pbHex,
                           PDWORD pdData)
    {
    DWORD n     = 0;
    DWORD dData = 0;

    if (pbHex != NULL)
        {
        while ((n < DWORD_ * 2) && (SbsHexTestA (pbHex [n])))
            {
            dData <<= 4;
            dData  += SbsHexNibbleA (pbHex [n++]);
            }
        }
    if (pdData != NULL) *pdData = dData;
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsHexDwordW (PWORD  pwHex,
                           PDWORD pdData)
    {
    DWORD n     = 0;
    DWORD dData = 0;

    if (pwHex != NULL)
        {
        while ((n < DWORD_ * 2) && (SbsHexTestW (pwHex [n])))
            {
            dData <<= 4;
            dData  += SbsHexNibbleW (pwHex [n++]);
            }
        }
    if (pdData != NULL) *pdData = dData;
    return n;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsHexEncodeA (PVOID  pData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, 0);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    n = dData * 2;

    if ((pbData1 = _tnewA (n+1)) != NULL)
        {
        for (i = n = 0; i < dData; i++)
            {
            bData = (pData != NULL ? ((PBYTE) pData) [i] : 0);

            pbData1 [n++] = gabHex [bData >>  4];
            pbData1 [n++] = gabHex [bData & 0xF];
            }
        pbData1 [dData1 = n] = 0;
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsHexEncodeW (PVOID  pData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, 0);
    DWORD dData1  = 0;
    PWORD pwData1 = NULL;

    n = dData * 2;

    if ((pwData1 = _tnewW (n+1)) != NULL)
        {
        for (i = n = 0; i < dData; i++)
            {
            bData = (pData != NULL ? ((PBYTE) pData) [i] : 0);

            pwData1 [n++] = gawHex [bData >>  4];
            pwData1 [n++] = gawHex [bData & 0xF];
            }
        pwData1 [dData1 = n] = 0;
        }
    if (pdData != NULL) *pdData = dData1;
    return pwData1;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsHexDecodeA (PBYTE  pbData,
                            PDWORD pdData)
    {
    BYTE  bHigh, bLow;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pbData != NULL)
        {
        dData = SbsTextDataA (pbData, dData);
        n     = dData / 2;

        if ((pbData1 = _mnew (n+2)) != NULL)
            {
            for (i = n = 0; i+1 < dData; i += 2)
                {
                bHigh = pbData [i  ];
                bLow  = pbData [i+1];

                if ((!SbsHexTestA (bHigh)) ||
                    (!SbsHexTestA (bLow ))) break;

                pbData1 [n++] = SbsHexByteA (bHigh, bLow);
                }
            if (i == dData)
                {
                pbData1 [n  ] = 0;
                pbData1 [n+1] = 0;
                dData1        = n;
                }
            else
                {
                pbData1 = _mfree (pbData1);
                }
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsHexDecodeW (PWORD  pwData,
                            PDWORD pdData)
    {
    WORD  wHigh, wLow;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pwData != NULL)
        {
        dData = SbsTextDataW (pwData, dData);
        n     = dData / 2;

        if ((pbData1 = _mnew (n+2)) != NULL)
            {
            for (i = n = 0; i+1 < dData; i += 2)
                {
                wHigh = pwData [i  ];
                wLow  = pwData [i+1];

                if ((!SbsHexTestW (wHigh)) ||
                    (!SbsHexTestW (wLow ))) break;

                pbData1 [n++] = SbsHexByteW (wHigh, wLow);
                }
            if (i == dData)
                {
                pbData1 [n  ] = 0;
                pbData1 [n+1] = 0;
                dData1        = n;
                }
            else
                {
                pbData1 = _mfree (pbData1);
                }
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// =================================================================
// TEXT PATTERN MATCHER
// =================================================================

BOOL WINAPI SbsMatchA (PBYTE pbFilter,
                       PBYTE pbData,
                       BOOL  fIgnoreCase)
    {
    DWORD i = 0;
    DWORD j = 0;

    if (pbData   == NULL) return FALSE;
    if (pbFilter == NULL) return TRUE;

    while (pbFilter [i] && pbData [j])
        {
        if (pbFilter [i] != '?')
            {
            if (pbFilter [i] == '*')
                {
                i++;

                if ((pbFilter [i] != '*') &&
                    (pbFilter [i] != '?'))
                    {
                    if (pbFilter [i])
                        {
                        while (pbData [j]
                               &&
                               (!SbsMatchA (pbFilter + i,
                                            pbData   + j,
                                            fIgnoreCase)))
                            {
                            j++;
                            }
                        }
                    return pbData [j] != 0;
                    }
                }
            if (fIgnoreCase
                ? LCASEA (pbFilter [i]) != LCASEA (pbData [j])
                :         pbFilter [i]  !=         pbData [j] )
                {
                return FALSE;
                }
            }
        i++;
        j++;
        }
    if (pbFilter [i] == '*') i++;
    return !(pbFilter [i] || pbData [j]);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsMatchW (PWORD pwFilter,
                       PWORD pwData,
                       BOOL  fIgnoreCase)
    {
    DWORD i = 0;
    DWORD j = 0;

    if (pwData   == NULL) return FALSE;
    if (pwFilter == NULL) return TRUE;

    while (pwFilter [i] && pwData [j])
        {
        if (pwFilter [i] != '?')
            {
            if (pwFilter [i] == '*')
                {
                i++;

                if ((pwFilter [i] != '*') &&
                    (pwFilter [i] != '?'))
                    {
                    if (pwFilter [i])
                        {
                        while (pwData [j]
                               &&
                               (!SbsMatchW (pwFilter + i,
                                            pwData   + j,
                                            fIgnoreCase)))
                            {
                            j++;
                            }
                        }
                    return pwData [j] != 0;
                    }
                }
            if (fIgnoreCase
                ? LCASEW (pwFilter [i]) != LCASEW (pwData [j])
                :         pwFilter [i]  !=         pwData [j] )
                {
                return FALSE;
                }
            }
        i++;
        j++;
        }
    if (pwFilter [i] == '*') i++;
    return !(pwFilter [i] || pwData [j]);
    }

// =================================================================
// SIMPLE LOCK MANAGEMENT
// =================================================================

BOOL WINAPI SbsLockSet (PSBS_LOCK psl,
                        BOOL      fBeep)
    {
    BOOL fOk = TRUE;

    if ((psl != NULL) && _mtrue (&psl->fLocked))
        {
        if (fBeep) MessageBeep (-1);
        fOk = FALSE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsLockReset (PSBS_LOCK psl)
    {
    if (psl != NULL) _mfalse (&psl->fLocked);
    return;
    }

// =================================================================
// MEMORY BUFFER MANAGEMENT
// =================================================================

DWORD WINAPI SbsBufferBlock (DWORD dData)
    {
    DWORD n;
    DWORD dBlock = SBS_PAGE - SBS_MEMORY_;

    while (dBlock < dData)
        {
        if ((n = dBlock + dBlock + SBS_MEMORY_) <= dBlock)
            {
            dBlock = dData;
            }
        else
            {
            dBlock = n;
            }
        }
    return dBlock;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferCreate (DWORD dSize)
    {
    DWORD       dData = SbsBufferBlock (dSize);
    PSBS_BUFFER psb   = NULL;

    if ((psb = _mnew (SBS_BUFFER_)) != NULL)
        {
        if ((psb->pData = _mnew (dData)) != NULL)
            {
            psb->dOffset = 0;
            psb->dData   = dData;
            }
        else
            {
            psb = _mfree (psb);
            }
        }
    return psb;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferDestroy (PSBS_BUFFER psb)
    {
    if (psb != NULL)
        {
        _mfree (psb->pData);
        _mfree (psb);
        }
    return NULL;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferResize (PSBS_BUFFER psb,
                                    DWORD       dData)
    {
    PSBS_BUFFER psb1 = psb;

    if (psb1 != NULL)
        {
        if ((psb1->pData = _mresize (psb1->pData, dData, NULL))
            != NULL)
            {
            psb1->dOffset = min (psb1->dOffset, dData);
            psb1->dData   = dData;
            }
        else
            {
            psb1 = _mfree (psb1);
            }
        }
    return psb1;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsBufferRelease (PSBS_BUFFER psb,
                               PDWORD      pdData)
    {
    PSBS_BUFFER psb1  = psb;
    DWORD       dData = 0;
    PVOID       pData = NULL;

    if ((psb1 != NULL) &&
        ((psb1 = SbsBufferResize (psb1, psb1->dOffset)) != NULL))
        {
        dData = psb1->dData;
        pData = psb1->pData;

        _mfree (psb1);
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferWrite (PSBS_BUFFER psb,
                                   PVOID       pData,
                                   DWORD       dData)
    {
    DWORD       n;
    PSBS_BUFFER psb1 = psb;

    if (psb1 != NULL)
        {
        if (((pData != NULL) || (!dData)) &&
            ((n = psb1->dOffset + dData) >= psb1->dOffset))
            {
            if (n > psb1->dData)
                {
                psb1 = SbsBufferResize (psb1, SbsBufferBlock (n));
                }
            if (psb1 != NULL)
                {
                _mcopy (psb1->pbData + psb1->dOffset, pData, dData);
                psb1->dOffset += dData;
                }
            }
        else
            {
            psb1 = SbsBufferDestroy (psb1);
            }
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferByte (PSBS_BUFFER psb,
                                  BYTE        bData,
                                  DWORD       dCount)
    {
    DWORD       i, j;
    BYTE        abData [SBS_BLOCK];
    PSBS_BUFFER psb1 = psb;

    for (i = min (dCount, SBS_BLOCK); i; i--)
        {
        abData [i-1] = bData;
        }
    for (i = dCount; (psb1 != NULL) && i; i -= j)
        {
        psb1 = SbsBufferWrite (psb1, abData,
                               (j = min (i, SBS_BLOCK)));
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferWord (PSBS_BUFFER psb,
                                  WORD        wData,
                                  DWORD       dCount)
    {
    DWORD       i, j;
    WORD        awData [SBS_BLOCK];
    PSBS_BUFFER psb1 = psb;

    for (i = min (dCount, SBS_BLOCK); i; i--)
        {
        awData [i-1] = wData;
        }
    for (i = dCount; (psb1 != NULL) && i; i -= j)
        {
        psb1 = SbsBufferWrite (psb1, awData,
                               (j = min (i, SBS_BLOCK)) * WORD_);
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsBufferDword (PSBS_BUFFER psb,
                                   DWORD       dData,
                                   DWORD       dCount)
    {
    DWORD       i, j;
    DWORD       adData [SBS_BLOCK];
    PSBS_BUFFER psb1 = psb;

    for (i = min (dCount, SBS_BLOCK); i; i--)
        {
        adData [i-1] = dData;
        }
    for (i = dCount; (psb1 != NULL) && i; i -= j)
        {
        psb1 = SbsBufferWrite (psb1, adData,
                               (j = min (i, SBS_BLOCK)) * DWORD_);
        }
    return psb1;
    }

// =================================================================
// TEXT BUFFER MANAGEMENT
// =================================================================

DWORD WINAPI SbsTextDataA (PBYTE pbData,
                           DWORD dData)
    {
    return (dData != -1 ? dData : _tsizeA (pbData));
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsTextDataW (PWORD pwData,
                           DWORD dData)
    {
    return (dData != -1 ? dData : _tsizeW (pwData));
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextCreateA (DWORD dSize)
    {
    return SbsBufferCreate (dSize);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextCreateW (DWORD dSize)
    {
    return SbsBufferCreate (dSize * WORD_);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextDestroy (PSBS_BUFFER psb)
    {
    return SbsBufferDestroy (psb);
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsTextReleaseA (PSBS_BUFFER psb,
                              PDWORD      pdData)
    {
    return SbsBufferRelease (psb, pdData);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsTextReleaseW (PSBS_BUFFER psb,
                              PDWORD      pdData)
    {
    DWORD dData  = 0;
    PWORD pwData = SbsBufferRelease (psb, &dData);

    if (pdData != NULL) *pdData = dData / WORD_;
    return pwData;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextWriteA (PSBS_BUFFER psb,
                                  PBYTE       pbData,
                                  DWORD       dData)
    {
    return SbsBufferWrite (psb, pbData,
                           SbsTextDataA (pbData, dData));
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextWriteW (PSBS_BUFFER psb,
                                  PWORD       pwData,
                                  DWORD       dData)
    {
    return SbsBufferWrite (psb, pwData,
                           SbsTextDataW (pwData, dData) * WORD_);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextAnsiA (PSBS_BUFFER psb,
                                 PBYTE       pbData,
                                 DWORD       dData)
    {
    return SbsBufferWrite (psb, pbData,
                           SbsTextDataA (pbData, dData));
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextAnsiW (PSBS_BUFFER psb,
                                 PWORD       pwData,
                                 DWORD       dData)
    {
    DWORD       i, j, n;
    BYTE        abData [SBS_BLOCK];
    DWORD       dData1 = SbsTextDataW (pwData, dData);
    PSBS_BUFFER psb1   = psb;

    if ((pwData == NULL) && dData1)
        {
        psb1 = SbsBufferDestroy (psb1);
        }
    for (i = dData1, j = 0; (psb1 != NULL) && i; i -= n, j += n)
        {
        n = min (i, SBS_BLOCK);

        _tcopyW2A (abData, pwData+j, n);

        psb1 = SbsBufferWrite (psb1, abData, n);
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextAsciiA (PSBS_BUFFER psb,
                                  PBYTE       pbData,
                                  DWORD       dData)
    {
    return SbsTextCustomA (psb, pbData, dData, _abAnsiAscii);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextAsciiW (PSBS_BUFFER psb,
                                  PWORD       pwData,
                                  DWORD       dData)
    {
    return SbsTextCustomW (psb, pwData, dData, _abAnsiAscii);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextCustomA (PSBS_BUFFER psb,
                                   PBYTE       pbData,
                                   DWORD       dData,
                                   PBYTE       pbTable)
    {
    DWORD       i, j, n;
    BYTE        abData [SBS_BLOCK];
    DWORD       dData1 = SbsTextDataA (pbData, dData);
    PSBS_BUFFER psb1   = psb;

    if ((pbData == NULL) && dData1)
        {
        psb1 = SbsBufferDestroy (psb1);
        }
    if (pbTable != NULL)
        {
        for (i = dData1, j = 0; (psb1 != NULL) && i; i -= n, j += n)
            {
            n = min (i, SBS_BLOCK);

            _tconv (abData, pbData+j, n, pbTable);

            psb1 = SbsBufferWrite (psb1, abData, n);
            }
        }
    else
        {
        psb1 = SbsTextAnsiA (psb1, pbData, dData);
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextCustomW (PSBS_BUFFER psb,
                                   PWORD       pwData,
                                   DWORD       dData,
                                   PBYTE       pbTable)
    {
    DWORD       i, j, n;
    BYTE        abData [SBS_BLOCK];
    DWORD       dData1 = SbsTextDataW (pwData, dData);
    PSBS_BUFFER psb1   = psb;

    if ((pwData == NULL) && dData1)
        {
        psb1 = SbsBufferDestroy (psb1);
        }
    if (pbTable != NULL)
        {
        for (i = dData1, j = 0; (psb1 != NULL) && i; i -= n, j += n)
            {
            n = min (i, SBS_BLOCK);

            _tcopyW2A (abData, pwData+j, n);
            _tconv    (abData, abData,   n, pbTable);

            psb1 = SbsBufferWrite (psb1, abData, n);
            }
        }
    else
        {
        psb1 = SbsTextAnsiW (psb1, pwData, dData);
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextFillA (PSBS_BUFFER psb,
                                 BYTE        bData,
                                 DWORD       dCount)
    {
    return SbsBufferByte (psb, bData, dCount);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextFillW (PSBS_BUFFER psb,
                                 WORD        wData,
                                 DWORD       dCount)
    {
    return SbsBufferWord (psb, wData, dCount);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextNumberA (PSBS_BUFFER psb,
                                   DWORD       dData,
                                   DWORD       dLimit,
                                   BOOL        fZero,
                                   BOOL        fSigned)
    {
    DWORD       i, j, k, n;
    BYTE        bData;
    BYTE        abData [SBS_UNSIGNED32-1];
    BOOL        fSign = fSigned && (dData & 0x80000000);
    PSBS_BUFFER psb1  = psb;

    i = SBS_UNSIGNED32-1;
    j = (dLimit ? dLimit : MAXDWORD);
    k = 0;
    n = (fSign ? 0 - dData : dData);

    do  {
        abData [--i] = (BYTE) (n % 10) + '0';
        k++;
        }
    while (--j && (n /= 10));

    if (dLimit)
        {
        if (fSigned && fZero && j)
            {
            bData = (fSign ? '-' : ' ');
            psb1  = SbsBufferByte (psb1, bData, 1);
            fSign = FALSE;
            j--;
            }
        if (n = (fSign && j ? j-1 : j))
            {
            bData = (fZero ? '0' : ' ');
            psb1  = SbsBufferByte (psb1, bData, n);
            }
        if (fSign && j)
            {
            psb1 = SbsBufferByte (psb1, '-', 1);
            }
        fSign = FALSE;
        }
    if (fSign)
        {
        psb1 = SbsBufferByte (psb1, '-', 1);
        }
    return SbsBufferWrite (psb1, abData+i, k);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextNumberW (PSBS_BUFFER psb,
                                   DWORD       dData,
                                   DWORD       dLimit,
                                   BOOL        fZero,
                                   BOOL        fSigned)
    {
    DWORD       i, j, k, n;
    WORD        wData;
    WORD        awData [SBS_UNSIGNED32-1];
    BOOL        fSign = fSigned && (dData & 0x80000000);
    PSBS_BUFFER psb1  = psb;

    i = SBS_UNSIGNED32-1;
    j = (dLimit ? dLimit : MAXDWORD);
    k = 0;
    n = (fSign ? 0 - dData : dData);

    do  {
        awData [--i] = (WORD) (n % 10) + '0';
        k++;
        }
    while (--j && (n /= 10));

    if (dLimit)
        {
        if (fSigned && fZero && j)
            {
            wData = (fSign ? '-' : ' ');
            psb1  = SbsBufferWord (psb1, wData, 1);
            fSign = FALSE;
            j--;
            }
        if (n = (fSign && j ? j-1 : j))
            {
            wData = (fZero ? '0' : ' ');
            psb1  = SbsBufferWord (psb1, wData, n);
            }
        if (fSign && j)
            {
            psb1 = SbsBufferWord (psb1, '-', 1);
            }
        fSign = FALSE;
        }
    if (fSign)
        {
        psb1 = SbsBufferWord (psb1, '-', 1);
        }
    return SbsBufferWrite (psb1, awData+i, k * WORD_);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextDwordA (PSBS_BUFFER psb,
                                  DWORD       dData,
                                  DWORD       dLimit,
                                  BOOL        fZero)
    {
    return SbsTextNumberA (psb, dData, dLimit, fZero, FALSE);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextDwordW (PSBS_BUFFER psb,
                                  DWORD       dData,
                                  DWORD       dLimit,
                                  BOOL        fZero)
    {
    return SbsTextNumberW (psb, dData, dLimit, fZero, FALSE);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextLongA (PSBS_BUFFER psb,
                                 LONG        lData,
                                 DWORD       dLimit,
                                 BOOL        fZero)
    {
    return SbsTextNumberA (psb, lData, dLimit, fZero, TRUE);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextLongW (PSBS_BUFFER psb,
                                 LONG        lData,
                                 DWORD       dLimit,
                                 BOOL        fZero)
    {
    return SbsTextNumberW (psb, lData, dLimit, fZero, TRUE);
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextFormatA (PSBS_BUFFER psb,
                                   PBYTE       pbFormat,
                                   ...)
    {
    DWORD       dData;
    PBYTE       pbData;
    PSBS_BUFFER psb1 = NULL;

    if (psb != NULL)
        {
        if ((pbData = _vbprintfA (&dData, pbFormat, (&pbFormat)+1))
            != NULL)
            {
            psb1 = SbsBufferWrite (psb, pbData, dData);
            _mfree (pbData);
            }
        else
            {
            psb1 = SbsTextDestroy (psb);
            }
        }
    return psb1;
    }

// -----------------------------------------------------------------

PSBS_BUFFER WINAPI SbsTextFormatW (PSBS_BUFFER psb,
                                   PWORD       pwFormat,
                                   ...)
    {
    DWORD       dData;
    PWORD       pwData;
    PSBS_BUFFER psb1 = NULL;

    if (psb != NULL)
        {
        if ((pwData = _vbprintfW (&dData, pwFormat, (&pwFormat)+1))
            != NULL)
            {
            psb1 = SbsBufferWrite (psb, pwData, dData * WORD_);
            _mfree (pwData);
            }
        else
            {
            psb1 = SbsTextDestroy (psb);
            }
        }
    return psb1;
    }

// =================================================================
// BSTR MANAGEMENT
// =================================================================

DWORD WINAPI SbsBstrSize (DWORD dBytes)
    {
    return (dBytes
            ? (((dBytes - 1) / SBS_BSTR_BLOCK) + 1) * SBS_BSTR_BLOCK
            : 0);
    }

// -----------------------------------------------------------------

PSBS_BSTR WINAPI SbsBstrBase (BSTR bs)
    {
    return (bs != NULL ? SBS_BSTR_BASE (bs) : NULL);
    }

// -----------------------------------------------------------------

BSTR WINAPI SbsBstrFree (BSTR bs)
    {
    return _mfree (SbsBstrBase (bs));
    }

// -----------------------------------------------------------------

BSTR WINAPI SbsBstrNewA (PBYTE pbData)
    {
    DWORD     dData, dSize;
    PSBS_BSTR psb;
    BSTR      bs = NULL;

    if (pbData != NULL)
        {
        dData = _tsizeA (pbData);
        dSize = SbsBstrSize (SBS_BSTR__ (dData+1));

        if ((psb = _mnew (dSize)) != NULL)
            {
            psb->dBytes = dData * WORD_;
            _tcopyA2W (bs = psb->awData, pbData, dData+1);
            }
        }
    return bs;
    }

// -----------------------------------------------------------------

BSTR WINAPI SbsBstrNewW (PWORD pwData)
    {
    DWORD     dData, dSize;
    PSBS_BSTR psb;
    BSTR      bs = NULL;

    if (pwData != NULL)
        {
        dData = _tsizeW (pwData);
        dSize = SbsBstrSize (SBS_BSTR__ (dData+1));

        if ((psb = _mnew (dSize)) != NULL)
            {
            psb->dBytes = dData * WORD_;
            _tcopyW2W (bs = psb->awData, pwData, dData+1);
            }
        }
    return bs;
    }

// -----------------------------------------------------------------

BSTR WINAPI SbsBstrClone (BSTR bs)
    {
    return SbsBstrNewW (bs);
    }

// =================================================================
// CLIPBOARD MANAGEMENT
// =================================================================

BOOL WINAPI SbsClipboardCopy (HWND  hWnd,
                              DWORD dFormat,
                              PVOID pData,
                              DWORD dData)
    {
    HGLOBAL hCopy;
    PVOID   pCopy;
    BOOL    fOk = FALSE;

    if (OpenClipboard (hWnd))
        {
        if (EmptyClipboard ())
            {
            if (dData)
                {
                if ((hCopy = GlobalAlloc (GMEM_MOVEABLE, dData))
                    != NULL)
                    {
                    if ((pCopy = GlobalLock (hCopy)) != NULL)
                        {
                        if (pData != NULL)
                            {
                            _mcopy (pCopy, pData, dData);
                            }
                        else
                            {
                            _mzero (pCopy, dData);
                            }
                        GlobalUnlock (hCopy);

                        fOk = SetClipboardData (dFormat, hCopy)
                              != NULL;
                        }
                    else
                        {
                        GlobalFree (hCopy);
                        }
                    }
                }
            else
                {
                fOk = TRUE;
                }
            }
        CloseClipboard ();
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsClipboardCopyTextA (HWND  hWnd,
                                   PBYTE pbData)
    {
    DWORD dData = (pbData != NULL ? _tsizeA (pbData) + 1 : 0);

    return SbsClipboardCopy (hWnd, CF_TEXT,
                             pbData, dData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsClipboardCopyTextW (HWND  hWnd,
                                   PWORD pwData)
    {
    DWORD dData = (pwData != NULL ? _tsizeW (pwData) + 1 : 0);

    return SbsClipboardCopy (hWnd, CF_UNICODETEXT,
                             pwData, dData * WORD_);
    }

// =================================================================
// REGISTRY MANAGEMENT
// =================================================================

BOOL WINAPI SbsRegistryTestA (PBYTE pbRoot,
                              PBYTE pbTest)
    {
    return (pbRoot != NULL) && (pbTest != NULL)
           &&
           ((!_tcompiA (pbRoot, pbTest)) ||
            (!_tcompiA (pbRoot, pbTest + _tsizeA (pbTest) + 1)));
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryTestW (PWORD pwRoot,
                              PWORD pwTest)
    {
    return (pwRoot != NULL) && (pwTest != NULL)
           &&
           ((!_tcompiW (pwRoot, pwTest)) ||
            (!_tcompiW (pwRoot, pwTest + _tsizeW (pwTest) + 1)));
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryRootA (PBYTE pbRoot,
                              PHKEY phk)
    {
    HKEY hk = HKEY_NULL;

    if      (SbsRegistryTestA (pbRoot, gabHkeyClassesRoot))
        {
        hk = HKEY_CLASSES_ROOT;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyCurrentUser))
        {
        hk = HKEY_CURRENT_USER;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyLocalMachine))
        {
        hk = HKEY_LOCAL_MACHINE;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyUsers))
        {
        hk = HKEY_USERS;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyPerformanceData))
        {
        hk = HKEY_PERFORMANCE_DATA;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyCurrentConfig))
        {
        hk = HKEY_CURRENT_CONFIG;
        }
    else if (SbsRegistryTestA (pbRoot, gabHkeyDynData))
        {
        hk = HKEY_DYN_DATA;
        }
    if (phk != NULL) *phk = hk;
    return hk != HKEY_NULL;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryRootW (PWORD pwRoot,
                              PHKEY phk)
    {
    HKEY hk = HKEY_NULL;

    if      (SbsRegistryTestW (pwRoot, gawHkeyClassesRoot))
        {
        hk = HKEY_CLASSES_ROOT;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyCurrentUser))
        {
        hk = HKEY_CURRENT_USER;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyLocalMachine))
        {
        hk = HKEY_LOCAL_MACHINE;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyUsers))
        {
        hk = HKEY_USERS;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyPerformanceData))
        {
        hk = HKEY_PERFORMANCE_DATA;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyCurrentConfig))
        {
        hk = HKEY_CURRENT_CONFIG;
        }
    else if (SbsRegistryTestW (pwRoot, gawHkeyDynData))
        {
        hk = HKEY_DYN_DATA;
        }
    if (phk != NULL) *phk = hk;
    return hk != HKEY_NULL;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryParseA (PBYTE  pbPath,
                               PHKEY  phk,
                               PPBYTE ppbRoot,
                               PPBYTE ppbKey,
                               PPBYTE ppbValue)
    {
    DWORD dKey, dValue, dStop;
    HKEY  hk      = HKEY_NULL;
    PBYTE pbRoot  = NULL;
    PBYTE pbKey   = NULL;
    PBYTE pbValue = NULL;

    if (pbPath != NULL)
        {
        for (dKey = 0;
             pbPath [dKey] && (pbPath [dKey] != '\\');
             dKey++);

        if (pbPath [dKey]) pbPath [dKey++] = 0;

        for (dValue = dStop = dKey + _tsizeA (pbPath + dKey);
             (dValue > dKey) && (pbPath [dValue-1] != '\\');
             dValue--);

        if (dValue > dKey)
            {
            pbPath [dValue-1] = 0;
            }
        else
            {
            dKey = dStop;
            }
        if (SbsRegistryRootA (pbPath, &hk))
            {
            pbRoot  = pbPath;
            pbKey   = pbPath + dKey;
            pbValue = pbPath + dValue;
            }
        }
    if (phk      != NULL) *phk      = hk;
    if (ppbRoot  != NULL) *ppbRoot  = pbRoot;
    if (ppbKey   != NULL) *ppbKey   = pbKey;
    if (ppbValue != NULL) *ppbValue = pbValue;
    return hk != HKEY_NULL;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryParseW (PWORD  pwPath,
                               PHKEY  phk,
                               PPWORD ppwRoot,
                               PPWORD ppwKey,
                               PPWORD ppwValue)
    {
    DWORD dKey, dValue, dStop;
    HKEY  hk      = HKEY_NULL;
    PWORD pwRoot  = NULL;
    PWORD pwKey   = NULL;
    PWORD pwValue = NULL;

    if (pwPath != NULL)
        {
        for (dKey = 0;
             pwPath [dKey] && (pwPath [dKey] != '\\');
             dKey++);

        if (pwPath [dKey]) pwPath [dKey++] = 0;

        for (dValue = dStop = dKey + _tsizeW (pwPath + dKey);
             (dValue > dKey) && (pwPath [dValue-1] != '\\');
             dValue--);

        if (dValue > dKey)
            {
            pwPath [dValue-1] = 0;
            }
        else
            {
            dKey = dStop;
            }
        if (SbsRegistryRootW (pwPath, &hk))
            {
            pwRoot  = pwPath;
            pwKey   = pwPath + dKey;
            pwValue = pwPath + dValue;
            }
        }
    if (phk      != NULL) *phk      = hk;
    if (ppwRoot  != NULL) *ppwRoot  = pwRoot;
    if (ppwKey   != NULL) *ppwKey   = pwKey;
    if (ppwValue != NULL) *ppwValue = pwValue;
    return hk != HKEY_NULL;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsRegistryParseExA (PBYTE  pbPath,
                                  PHKEY  phk,
                                  PPBYTE ppbRoot,
                                  PPBYTE ppbKey,
                                  PPBYTE ppbValue)
    {
    HKEY  hk      = HKEY_NULL;
    PBYTE pbRoot  = NULL;
    PBYTE pbKey   = NULL;
    PBYTE pbValue = NULL;
    PBYTE pbData  = NULL;

    if ((pbPath != NULL) &&
        ((pbData = _tnewA (_tsizeA (pbPath) + 1)) != NULL))
        {
        _tcopyA (pbData, pbPath);

        if (!SbsRegistryParseA (pbData,
                                &hk, &pbRoot, &pbKey, &pbValue))
            {
            pbData = _mfree (pbData);
            }
        }
    if (phk      != NULL) *phk      = hk;
    if (ppbRoot  != NULL) *ppbRoot  = pbRoot;
    if (ppbKey   != NULL) *ppbKey   = pbKey;
    if (ppbValue != NULL) *ppbValue = pbValue;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsRegistryParseExW (PWORD  pwPath,
                                  PHKEY  phk,
                                  PPWORD ppwRoot,
                                  PPWORD ppwKey,
                                  PPWORD ppwValue)
    {
    HKEY  hk      = HKEY_NULL;
    PWORD pwRoot  = NULL;
    PWORD pwKey   = NULL;
    PWORD pwValue = NULL;
    PWORD pwData  = NULL;

    if ((pwPath != NULL) &&
        ((pwData = _tnewW (_tsizeW (pwPath) + 1)) != NULL))
        {
        _tcopyW (pwData, pwPath);

        if (!SbsRegistryParseW (pwData,
                                &hk, &pwRoot, &pwKey, &pwValue))
            {
            pwData = _mfree (pwData);
            }
        }
    if (phk      != NULL) *phk      = hk;
    if (ppwRoot  != NULL) *ppwRoot  = pwRoot;
    if (ppwKey   != NULL) *ppwKey   = pwKey;
    if (ppwValue != NULL) *ppwValue = pwValue;
    return pwData;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryClose (HKEY hk)
    {
    if ((hk != HKEY_NULL            ) &&
        (hk != HKEY_CLASSES_ROOT    ) &&
        (hk != HKEY_CURRENT_USER    ) &&
        (hk != HKEY_LOCAL_MACHINE   ) &&
        (hk != HKEY_USERS           ) &&
        (hk != HKEY_PERFORMANCE_DATA) &&
        (hk != HKEY_CURRENT_CONFIG  ) &&
        (hk != HKEY_DYN_DATA        ))
        {
        RegCloseKey (hk);
        }
    return HKEY_NULL;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryOpenA (HKEY  hkBase,
                              PBYTE pbKey)
    {
    HKEY hk = HKEY_NULL;

    if (hkBase != HKEY_NULL)
        {
        if ((pbKey != NULL) && pbKey [0])
            {
            if (RegOpenKeyExA (hkBase, pbKey, 0,
                               KEY_ALL_ACCESS, &hk)
                != ERROR_SUCCESS)
                {
                hk = HKEY_NULL;
                }
            }
        else
            {
            hk = hkBase;
            }
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryOpenW (HKEY  hkBase,
                              PWORD pwKey)
    {
    HKEY hk = HKEY_NULL;

    if (hkBase != HKEY_NULL)
        {
        if ((pwKey != NULL) && pwKey [0])
            {
            if (RegOpenKeyExW (hkBase, pwKey, 0,
                               KEY_ALL_ACCESS, &hk)
                != ERROR_SUCCESS)
                {
                hk = HKEY_NULL;
                }
            }
        else
            {
            hk = hkBase;
            }
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryOpenExA (HKEY  hkBase,
                                DWORD dKeys,
                                PBYTE pbKey,
                                ...)
    {
    HKEY   hk1;
    DWORD  i;
    PPBYTE ppbKeys = &pbKey;
    HKEY   hk      = hkBase;

    for (i = 0; (hk != HKEY_NULL) && (i < dKeys); i++)
        {
        hk = SbsRegistryOpenA (hk1 = hk, ppbKeys [i]);
        if (hk != hk1) SbsRegistryClose (hk1);
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryOpenExW (HKEY  hkBase,
                                DWORD dKeys,
                                PWORD pwKey,
                                ...)
    {
    HKEY   hk1;
    DWORD  i;
    PPWORD ppwKeys = &pwKey;
    HKEY   hk      = hkBase;

    for (i = 0; (hk != HKEY_NULL) && (i < dKeys); i++)
        {
        hk = SbsRegistryOpenW (hk1 = hk, ppwKeys [i]);
        if (hk != hk1) SbsRegistryClose (hk1);
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryCreateA (HKEY  hkBase,
                                PBYTE pbKey)
    {
    HKEY hk = HKEY_NULL;

    if (hkBase != HKEY_NULL)
        {
        if ((pbKey != NULL) && pbKey [0])
            {
            if (RegCreateKeyExA (hkBase, pbKey, 0, gabNull,
                                 REG_OPTION_NON_VOLATILE,
                                 KEY_ALL_ACCESS, NULL, &hk, NULL)
                != ERROR_SUCCESS)
                {
                hk = HKEY_NULL;
                }
            }
        else
            {
            hk = hkBase;
            }
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryCreateW (HKEY  hkBase,
                                PWORD pwKey)
    {
    HKEY hk = HKEY_NULL;

    if (hkBase != HKEY_NULL)
        {
        if ((pwKey != NULL) && pwKey [0])
            {
            if (RegCreateKeyExW (hkBase, pwKey, 0, gawNull,
                                 REG_OPTION_NON_VOLATILE,
                                 KEY_ALL_ACCESS, NULL, &hk, NULL)
                != ERROR_SUCCESS)
                {
                hk = HKEY_NULL;
                }
            }
        else
            {
            hk = hkBase;
            }
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryCreateExA (HKEY  hkBase,
                                  DWORD dKeys,
                                  PBYTE pbKey,
                                  ...)
    {
    HKEY   hk1;
    DWORD  i;
    PPBYTE ppbKeys = &pbKey;
    HKEY   hk      = hkBase;

    for (i = 0; (hk != HKEY_NULL) && (i < dKeys); i++)
        {
        hk = SbsRegistryCreateA (hk1 = hk, ppbKeys [i]);
        if (hk != hk1) SbsRegistryClose (hk1);
        }
    return hk;
    }

// -----------------------------------------------------------------

HKEY WINAPI SbsRegistryCreateExW (HKEY  hkBase,
                                  DWORD dKeys,
                                  PWORD pwKey,
                                  ...)
    {
    HKEY   hk1;
    DWORD  i;
    PPWORD ppwKeys = &pwKey;
    HKEY   hk      = hkBase;

    for (i = 0; (hk != HKEY_NULL) && (i < dKeys); i++)
        {
        hk = SbsRegistryCreateW (hk1 = hk, ppwKeys [i]);
        if (hk != hk1) SbsRegistryClose (hk1);
        }
    return hk;
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathA (PBYTE pbPath,
                                   BOOL  fCreate)
    {
    PSBS_PATH psp = NULL;

    if ((pbPath != NULL) &&
        ((psp = _mnew (SBS_PATH_)) != NULL))
        {
        psp->fUnicode = FALSE;

        psp->pbParts = SbsRegistryParseExA (pbPath,
                                            &psp->hkBase,
                                            &psp->pbRoot,
                                            &psp->pbKey,
                                            &psp->pbValue);
        psp->hk = (fCreate
                   ? SbsRegistryCreateA
                   : SbsRegistryOpenA) (psp->hkBase, psp->pbKey);

        if (psp->hk == HKEY_NULL)
            {
            psp = SbsRegistryPathClose (psp);
            }
        }
    return psp;
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathW (PWORD pwPath,
                                   BOOL  fCreate)
    {
    PSBS_PATH psp = NULL;

    if ((pwPath != NULL) &&
        ((psp = _mnew (SBS_PATH_)) != NULL))
        {
        psp->fUnicode = TRUE;

        psp->pwParts = SbsRegistryParseExW (pwPath,
                                            &psp->hkBase,
                                            &psp->pwRoot,
                                            &psp->pwKey,
                                            &psp->pwValue);
        psp->hk = (fCreate
                   ? SbsRegistryCreateW
                   : SbsRegistryOpenW) (psp->hkBase, psp->pwKey);

        if (psp->hk == HKEY_NULL)
            {
            psp = SbsRegistryPathClose (psp);
            }
        }
    return psp;
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathOpenA (PBYTE pbPath)
    {
    return SbsRegistryPathA (pbPath, FALSE);
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathOpenW (PWORD pwPath)
    {
    return SbsRegistryPathW (pwPath, FALSE);
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathCreateA (PBYTE pbPath)
    {
    return SbsRegistryPathA (pbPath, TRUE);
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathCreateW (PWORD pwPath)
    {
    return SbsRegistryPathW (pwPath, TRUE);
    }

// -----------------------------------------------------------------

PSBS_PATH WINAPI SbsRegistryPathClose (PSBS_PATH psp)
    {
    if (psp != NULL)
        {
        SbsRegistryClose (psp->hk);
        _mfree           (psp->pParts);
        _mfree           (psp);
        }
    return NULL;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryPathReadA (PBYTE  pbPath,
                                   DWORD  dMode,
                                   PDWORD pdSize)
    {
    return SbsRegistryReadA (HKEY_NULL, pbPath, dMode, pdSize);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryPathReadW (PWORD  pwPath,
                                   DWORD  dMode,
                                   PDWORD pdSize)
    {
    return SbsRegistryReadW (HKEY_NULL, pwPath, dMode, pdSize);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryPathWriteA (PBYTE pbPath,
                                   DWORD dMode,
                                   PVOID pData)
    {
    return SbsRegistryWriteA (HKEY_NULL, pbPath, dMode, pData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryPathWriteW (PWORD pwPath,
                                   DWORD dMode,
                                   PVOID pData)
    {
    return SbsRegistryWriteW (HKEY_NULL, pwPath, dMode, pData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryPathDeleteA (PBYTE pbPath)
    {
    DWORD     dIndex;
    BYTE      abKey [SBS_KEY];
    PBYTE     pbPath1;
    PSBS_PATH psp = SbsRegistryPathOpenA (pbPath);
    BOOL      fOk = (pbPath != NULL) && (psp == NULL);

    if (psp != NULL)
        {
        if (psp->pbKey [0])
            {
            if (RegQueryInfoKeyA (psp->hk, NULL, NULL, NULL,
                                  &dIndex, NULL, NULL, NULL,
                                  NULL,    NULL, NULL, NULL)
                == ERROR_SUCCESS)
                {
                while (dIndex)
                    {
                    if (RegEnumKeyA (psp->hk, --dIndex,
                                     abKey, SBS_KEY)
                        == ERROR_SUCCESS)
                        {
                        if ((pbPath1 = _bprintfA (NULL,
                                                  "%s\\%s\\%s\\",
                                                  psp->pbRoot,
                                                  psp->pbKey,
                                                  abKey))
                            != NULL)
                            {
                            SbsRegistryPathDeleteA (pbPath1);
                            _mfree (pbPath1);
                            }
                        }
                    }
                }
            fOk = (RegDeleteKeyA (psp->hkBase, psp->pbKey)
                   == ERROR_SUCCESS);
            }
        SbsRegistryPathClose (psp);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryPathDeleteW (PWORD pwPath)
    {
    DWORD     dIndex;
    WORD      awKey [SBS_KEY];
    PWORD     pwPath1;
    PSBS_PATH psp = SbsRegistryPathOpenW (pwPath);
    BOOL      fOk = (pwPath != NULL) && (psp == NULL);

    if (psp != NULL)
        {
        if (psp->pwKey [0])
            {
            if (RegQueryInfoKeyW (psp->hk, NULL, NULL, NULL,
                                  &dIndex, NULL, NULL, NULL,
                                  NULL,    NULL, NULL, NULL)
                == ERROR_SUCCESS)
                {
                while (dIndex)
                    {
                    if (RegEnumKeyW (psp->hk, --dIndex,
                                     awKey, SBS_KEY)
                        == ERROR_SUCCESS)
                        {
                        if ((pwPath1 = _bprintfW (NULL,
                                                  L"%s\\%s\\%s\\",
                                                  psp->pwRoot,
                                                  psp->pwKey,
                                                  awKey))
                            != NULL)
                            {
                            SbsRegistryPathDeleteW (pwPath1);
                            _mfree (pwPath1);
                            }
                        }
                    }
                }
            fOk = (RegDeleteKeyW (psp->hkBase, psp->pwKey)
                   == ERROR_SUCCESS);
            }
        SbsRegistryPathClose (psp);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryDataA (HKEY   hk,
                               PBYTE  pbValue,
                               PDWORD pdType,
                               PDWORD pdSize)
    {
    DWORD     dType, dSize;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PBYTE     pbValue1 = pbValue;
    PVOID     pData    = NULL;

    if ((hk1 == HKEY_NULL) &&
        ((psp = SbsRegistryPathOpenA (pbValue1)) != NULL))
        {
        hk1      = psp->hk;
        pbValue1 = psp->pbValue;
        }
    if ((hk1 != HKEY_NULL)
        &&
        (RegQueryValueExA (hk1, pbValue1, NULL,
                           NULL, NULL, &dSize)
         == ERROR_SUCCESS)
        &&
        ((pData = _mnew (dSize)) != NULL)
        &&
        (RegQueryValueExA (hk1, pbValue1, NULL,
                           &dType, pData, &dSize)
         != ERROR_SUCCESS))
        {
        pData = _mfree (pData);
        }
    SbsRegistryPathClose (psp);

    if (pdType != NULL) *pdType = (pData != NULL? dType : REG_NONE);
    if (pdSize != NULL) *pdSize = (pData != NULL? dSize : 0       );
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryDataW (HKEY   hk,
                               PWORD  pwValue,
                               PDWORD pdType,
                               PDWORD pdSize)
    {
    DWORD     dType, dSize;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PWORD     pwValue1 = pwValue;
    PVOID     pData    = NULL;

    if ((hk1 == HKEY_NULL) &&
        ((psp = SbsRegistryPathOpenW (pwValue1)) != NULL))
        {
        hk1      = psp->hk;
        pwValue1 = psp->pwValue;
        }
    if ((hk1 != HKEY_NULL)
        &&
        (RegQueryValueExW (hk1, pwValue1, NULL,
                           NULL, NULL, &dSize)
         == ERROR_SUCCESS)
        &&
        ((pData = _mnew (dSize)) != NULL)
        &&
        (RegQueryValueExW (hk1, pwValue1, NULL,
                           &dType, pData, &dSize)
         != ERROR_SUCCESS))
        {
        pData = _mfree (pData);
        }
    SbsRegistryPathClose (psp);

    if (pdType != NULL) *pdType = (pData != NULL? dType : REG_NONE);
    if (pdSize != NULL) *pdSize = (pData != NULL? dSize : 0       );
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryReadA (HKEY   hk,
                               PBYTE  pbValue,
                               DWORD  dMode,
                               PDWORD pdSize)
    {
    DWORD dType, dSize;
    PVOID pData = NULL;

    if ((pData = SbsRegistryDataA (hk, pbValue, &dType, &dSize))
        != NULL)
        {
        if (((dMode == SBS_MODE_DWORD) &&
             ((dType != REG_DWORD) || (dSize != DWORD_)))
            ||
            ((dMode == SBS_MODE_TEXT) &&
             ((dType != REG_SZ) || (!dSize)))
            ||
            (SBS_MODE_BINARY (dMode) &&
             (dMode != dSize)))
            {
            pData = _mfree (pData);
            }
        }
    if (pdSize != NULL) *pdSize = (pData != NULL? dSize : 0);
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryReadW (HKEY   hk,
                               PWORD  pwValue,
                               DWORD  dMode,
                               PDWORD pdSize)
    {
    DWORD dType, dSize;
    PVOID pData = NULL;

    if ((pData = SbsRegistryDataW (hk, pwValue, &dType, &dSize))
        != NULL)
        {
        if (((dMode == SBS_MODE_DWORD) &&
             ((dType != REG_DWORD) || (dSize != DWORD_)))
            ||
            ((dMode == SBS_MODE_TEXT) &&
             ((dType != REG_SZ) || (!(dSize /= WORD_))))
            ||
            (SBS_MODE_BINARY (dMode) &&
             (dMode != dSize)))
            {
            pData = _mfree (pData);
            }
        }
    if (pdSize != NULL) *pdSize = (pData != NULL? dSize : 0);
    return pData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryReadDwordA (HKEY   hk,
                                   PBYTE  pbValue,
                                   PDWORD pdData)
    {
    PVOID pData;
    DWORD dData = 0;
    BOOL  fOk   = FALSE;

    if ((pData = SbsRegistryReadA (hk, pbValue, SBS_MODE_DWORD,
                                   NULL))
        != NULL)
        {
        dData = *(PDWORD) pData;
        _mfree (pData);

        fOk = TRUE;
        }
    if (pdData != NULL) *pdData = dData;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryReadDwordW (HKEY   hk,
                                   PWORD  pwValue,
                                   PDWORD pdData)
    {
    PVOID pData;
    DWORD dData = 0;
    BOOL  fOk   = FALSE;

    if ((pData = SbsRegistryReadW (hk, pwValue, SBS_MODE_DWORD,
                                   NULL))
        != NULL)
        {
        dData = *(PDWORD) pData;
        _mfree (pData);

        fOk = TRUE;
        }
    if (pdData != NULL) *pdData = dData;
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsRegistryReadTextA (HKEY   hk,
                                   PBYTE  pbValue,
                                   PDWORD pdSize)
    {
    return SbsRegistryReadA (hk, pbValue, SBS_MODE_TEXT, pdSize);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsRegistryReadTextW (HKEY   hk,
                                   PWORD  pwValue,
                                   PDWORD pdSize)
    {
    return SbsRegistryReadW (hk, pwValue, SBS_MODE_TEXT, pdSize);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryReadBinaryA (HKEY  hk,
                                     PBYTE pbValue,
                                     DWORD dSize)
    {
    PVOID pData = NULL;

    if (SBS_MODE_BINARY (dSize))
        {
        pData = SbsRegistryReadA (hk, pbValue, dSize, NULL);
        }
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsRegistryReadBinaryW (HKEY  hk,
                                     PWORD pwValue,
                                     DWORD dSize)
    {
    PVOID pData = NULL;

    if (SBS_MODE_BINARY (dSize))
        {
        pData = SbsRegistryReadW (hk, pwValue, dSize, NULL);
        }
    return pData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteA (HKEY  hk,
                               PBYTE pbValue,
                               DWORD dMode,
                               PVOID pData)
    {
    PVOID     pData1;
    DWORD     dSize;
    DWORD     dNull    = 0;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PBYTE     pbValue1 = pbValue;
    BOOL      fOk      = FALSE;

    if ((hk1 == HKEY_NULL) &&
        ((psp = SbsRegistryPathCreateA (pbValue1)) != NULL))
        {
        hk1      = psp->hk;
        pbValue1 = psp->pbValue;
        }
    if (hk1 != HKEY_NULL)
        {
        switch (dMode)
            {
            case SBS_MODE_DWORD:
                {
                pData1 = SAFEPTR (pData, &dNull);
                dSize  = DWORD_;

                fOk = (RegSetValueExA (hk1, pbValue1, 0,
                                       REG_DWORD,
                                       pData1, dSize)
                       == ERROR_SUCCESS);
                break;
                }
            case SBS_MODE_TEXT:
                {
                pData1 = SAFEPTR (pData, gabNull);
                dSize  = (_tsizeA (pData1) + 1);

                fOk = (RegSetValueExA (hk1, pbValue1, 0,
                                       REG_SZ,
                                       pData1, dSize)
                       == ERROR_SUCCESS);
                break;
                }
            default:
                {
                dSize = dMode;

                if (pData != NULL)
                    {
                    pData1 = pData;
                    }
                else
                    {
                    if ((pData1 = _mnew (dSize)) != NULL)
                        {
                        _mzero (pData1, dSize);
                        }
                    }
                if (pData1 != NULL)
                    {
                    fOk = (RegSetValueExA (hk1, pbValue1, 0,
                                           REG_BINARY,
                                           pData1, dSize)
                           == ERROR_SUCCESS);

                    if (pData1 != pData)
                        {
                        _mfree (pData1);
                        }
                    }
                break;
                }
            }
        }
    SbsRegistryPathClose (psp);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteW (HKEY  hk,
                               PWORD pwValue,
                               DWORD dMode,
                               PVOID pData)
    {
    PVOID     pData1;
    DWORD     dSize;
    DWORD     dNull    = 0;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PWORD     pwValue1 = pwValue;
    BOOL      fOk      = FALSE;

    if ((hk1 == HKEY_NULL) &&
        ((psp = SbsRegistryPathCreateW (pwValue1)) != NULL))
        {
        hk1      = psp->hk;
        pwValue1 = psp->pwValue;
        }
    if (hk1 != HKEY_NULL)
        {
        switch (dMode)
            {
            case SBS_MODE_DWORD:
                {
                pData1 = SAFEPTR (pData, &dNull);
                dSize  = DWORD_;

                fOk = (RegSetValueExW (hk1, pwValue1, 0,
                                       REG_DWORD,
                                       pData1, dSize)
                       == ERROR_SUCCESS);
                break;
                }
            case SBS_MODE_TEXT:
                {
                pData1 = SAFEPTR (pData, gawNull);
                dSize  = (_tsizeW (pData1) + 1) * WORD_;

                fOk = (RegSetValueExW (hk1, pwValue1, 0,
                                       REG_SZ,
                                       pData1, dSize)
                       == ERROR_SUCCESS);
                break;
                }
            default:
                {
                dSize = dMode;

                if (pData != NULL)
                    {
                    pData1 = pData;
                    }
                else
                    {
                    if ((pData1 = _mnew (dSize)) != NULL)
                        {
                        _mzero (pData1, dSize);
                        }
                    }
                if (pData1 != NULL)
                    {
                    fOk = (RegSetValueExW (hk1, pwValue1, 0,
                                           REG_BINARY,
                                           pData1, dSize)
                           == ERROR_SUCCESS);

                    if (pData1 != pData)
                        {
                        _mfree (pData1);
                        }
                    }
                break;
                }
            }
        }
    SbsRegistryPathClose (psp);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteDwordA (HKEY  hk,
                                    PBYTE pbValue,
                                    DWORD dData)
    {
    return SbsRegistryWriteA (hk, pbValue, SBS_MODE_DWORD, &dData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteDwordW (HKEY  hk,
                                    PWORD pwValue,
                                    DWORD dData)
    {
    return SbsRegistryWriteW (hk, pwValue, SBS_MODE_DWORD, &dData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteTextA (HKEY  hk,
                                   PBYTE pbValue,
                                   PBYTE pbData)
    {
    return SbsRegistryWriteA (hk, pbValue, SBS_MODE_TEXT, pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteTextW (HKEY  hk,
                                   PWORD pwValue,
                                   PWORD pwData)
    {
    return SbsRegistryWriteW (hk, pwValue, SBS_MODE_TEXT, pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteBinaryA (HKEY  hk,
                                     PBYTE pbValue,
                                     PVOID pData,
                                     DWORD dSize)
    {
    return SbsRegistryWriteA (hk, pbValue, dSize, pData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryWriteBinaryW (HKEY  hk,
                                     PWORD pwValue,
                                     PVOID pData,
                                     DWORD dSize)
    {
    return SbsRegistryWriteW (hk, pwValue, dSize, pData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryDeleteValueA (HKEY  hk,
                                     PBYTE pbValue)
    {
    DWORD     dStatus;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PBYTE     pbValue1 = pbValue;
    BOOL      fOk      = FALSE;

    if (hk1 == HKEY_NULL)
        {
        if ((psp = SbsRegistryPathOpenA (pbValue1)) != NULL)
            {
            hk1      = psp->hk;
            pbValue1 = psp->pbValue;
            }
        else
            {
            fOk = TRUE;
            }
        }
    if ((!fOk) && (hk1 != HKEY_NULL))
        {
        dStatus = RegDeleteValueA (hk1, pbValue1);

        fOk = (dStatus == ERROR_SUCCESS       ) ||
              (dStatus == ERROR_FILE_NOT_FOUND);
        }
    SbsRegistryPathClose (psp);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsRegistryDeleteValueW (HKEY  hk,
                                     PWORD pwValue)
    {
    DWORD     dStatus;
    PSBS_PATH psp      = NULL;
    HKEY      hk1      = hk;
    PWORD     pwValue1 = pwValue;
    BOOL      fOk      = FALSE;

    if (hk1 == HKEY_NULL)
        {
        if ((psp = SbsRegistryPathOpenW (pwValue1)) != NULL)
            {
            hk1      = psp->hk;
            pwValue1 = psp->pwValue;
            }
        else
            {
            fOk = TRUE;
            }
        }
    if ((!fOk) && (hk1 != HKEY_NULL))
        {
        dStatus = RegDeleteValueW (hk1, pwValue1);

        fOk = (dStatus == ERROR_SUCCESS       ) ||
              (dStatus == ERROR_FILE_NOT_FOUND);
        }
    SbsRegistryPathClose (psp);
    return fOk;
    }

// =================================================================
// SOFTWARE DATA MANAGEMENT
// =================================================================

PBYTE WINAPI SbsSoftwareValueA (PBYTE pbFormat,
                                PBYTE pbCompany,
                                PBYTE pbProduct,
                                PBYTE pbVersion,
                                PBYTE pbItem)
    {
    PBYTE apbArguments [4];
    PBYTE pbFormat1;
    DWORD n = 0;

    if (pbFormat != NULL)
        {
        pbFormat1 = pbFormat;

        if (pbCompany != NULL) apbArguments [n++] = pbCompany;
        if (pbProduct != NULL) apbArguments [n++] = pbProduct;
        if (pbVersion != NULL) apbArguments [n++] = pbVersion;
        if (pbItem    != NULL) apbArguments [n++] = pbItem;
        }
    else
        {
        pbFormat1 = "HKCU\\Software\\%s\\%s\\%s\\%s";

        apbArguments [n++] = SAFESTR (pbCompany, "All Companies" );
        apbArguments [n++] = SAFESTR (pbProduct, "All Products"  );
        apbArguments [n++] = SAFESTR (pbVersion, "CurrentVersion");
        apbArguments [n++] = SAFESTR (pbItem,    ""              );
        }
    return _vbprintfA (NULL, pbFormat1, apbArguments);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsSoftwareValueW (PWORD pwFormat,
                                PWORD pwCompany,
                                PWORD pwProduct,
                                PWORD pwVersion,
                                PWORD pwItem)
    {
    PWORD apwArguments [4];
    PWORD pwFormat1;
    DWORD n = 0;

    if (pwFormat != NULL)
        {
        pwFormat1 = pwFormat;

        if (pwCompany != NULL) apwArguments [n++] = pwCompany;
        if (pwProduct != NULL) apwArguments [n++] = pwProduct;
        if (pwVersion != NULL) apwArguments [n++] = pwVersion;
        if (pwItem    != NULL) apwArguments [n++] = pwItem;
        }
    else
        {
        pwFormat1 = L"HKCU\\Software\\%s\\%s\\%s\\%s";

        apwArguments [n++] = SAFESTR (pwCompany, L"All Companies" );
        apwArguments [n++] = SAFESTR (pwProduct, L"All Products"  );
        apwArguments [n++] = SAFESTR (pwVersion, L"CurrentVersion");
        apwArguments [n++] = SAFESTR (pwItem,    L""              );
        }
    return _vbprintfW (NULL, pwFormat1, apwArguments);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareReadDwordA (PBYTE  pbFormat,
                                   PBYTE  pbCompany,
                                   PBYTE  pbProduct,
                                   PBYTE  pbVersion,
                                   PBYTE  pbItem,
                                   PDWORD pdData)
    {
    PBYTE pbValue;
    DWORD dData = 0;
    BOOL  fOk   = FALSE;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryReadDwordA (HKEY_NULL, pbValue, &dData);
        _mfree (pbValue);
        }
    if (pdData != NULL) *pdData = dData;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareReadDwordW (PWORD  pwFormat,
                                   PWORD  pwCompany,
                                   PWORD  pwProduct,
                                   PWORD  pwVersion,
                                   PWORD  pwItem,
                                   PDWORD pdData)
    {
    PWORD pwValue;
    DWORD dData = 0;
    BOOL  fOk   = FALSE;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryReadDwordW (HKEY_NULL, pwValue, &dData);
        _mfree (pwValue);
        }
    if (pdData != NULL) *pdData = dData;
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsSoftwareReadTextA (PBYTE  pbFormat,
                                   PBYTE  pbCompany,
                                   PBYTE  pbProduct,
                                   PBYTE  pbVersion,
                                   PBYTE  pbItem,
                                   PDWORD pdSize)
    {
    PBYTE pbValue;
    DWORD dSize  = 0;
    PBYTE pbData = NULL;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        pbData = SbsRegistryReadTextA (HKEY_NULL, pbValue, &dSize);
        _mfree (pbValue);
        }
    if (pdSize != NULL) *pdSize = dSize;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsSoftwareReadTextW (PWORD  pwFormat,
                                   PWORD  pwCompany,
                                   PWORD  pwProduct,
                                   PWORD  pwVersion,
                                   PWORD  pwItem,
                                   PDWORD pdSize)
    {
    PWORD pwValue;
    DWORD dSize  = 0;
    PWORD pwData = NULL;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        pwData = SbsRegistryReadTextW (HKEY_NULL, pwValue, &dSize);
        _mfree (pwValue);
        }
    if (pdSize != NULL) *pdSize = dSize;
    return pwData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsSoftwareReadBinaryA (PBYTE pbFormat,
                                     PBYTE pbCompany,
                                     PBYTE pbProduct,
                                     PBYTE pbVersion,
                                     PBYTE pbItem,
                                     DWORD dSize)
    {
    PBYTE pbValue;
    PVOID pData = NULL;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        pData = SbsRegistryReadBinaryA (HKEY_NULL, pbValue, dSize);
        _mfree (pbValue);
        }
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsSoftwareReadBinaryW (PWORD pwFormat,
                                     PWORD pwCompany,
                                     PWORD pwProduct,
                                     PWORD pwVersion,
                                     PWORD pwItem,
                                     DWORD dSize)
    {
    PWORD pwValue;
    PVOID pData = NULL;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        pData = SbsRegistryReadBinaryW (HKEY_NULL, pwValue, dSize);
        _mfree (pwValue);
        }
    return pData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteDwordA (PBYTE pbFormat,
                                    PBYTE pbCompany,
                                    PBYTE pbProduct,
                                    PBYTE pbVersion,
                                    PBYTE pbItem,
                                    DWORD dData)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryWriteDwordA (HKEY_NULL, pbValue, dData);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteDwordW (PWORD pwFormat,
                                    PWORD pwCompany,
                                    PWORD pwProduct,
                                    PWORD pwVersion,
                                    PWORD pwItem,
                                    DWORD dData)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryWriteDwordW (HKEY_NULL, pwValue, dData);
        _mfree (pwValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteTextA (PBYTE pbFormat,
                                   PBYTE pbCompany,
                                   PBYTE pbProduct,
                                   PBYTE pbVersion,
                                   PBYTE pbItem,
                                   PBYTE pbData)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryWriteTextA (HKEY_NULL, pbValue, pbData);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteTextW (PWORD pwFormat,
                                   PWORD pwCompany,
                                   PWORD pwProduct,
                                   PWORD pwVersion,
                                   PWORD pwItem,
                                   PWORD pwData)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryWriteTextW (HKEY_NULL, pwValue, pwData);
        _mfree (pwValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteBinaryA (PBYTE pbFormat,
                                     PBYTE pbCompany,
                                     PBYTE pbProduct,
                                     PBYTE pbVersion,
                                     PBYTE pbItem,
                                     PVOID pData,
                                     DWORD dSize)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryWriteBinaryA (HKEY_NULL, pbValue,
                                       pData, dSize);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareWriteBinaryW (PWORD pwFormat,
                                     PWORD pwCompany,
                                     PWORD pwProduct,
                                     PWORD pwVersion,
                                     PWORD pwItem,
                                     PVOID pData,
                                     DWORD dSize)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryWriteBinaryW (HKEY_NULL, pwValue,
                                       pData, dSize);
        _mfree (pwValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareDeleteKeyA (PBYTE pbFormat,
                                   PBYTE pbCompany,
                                   PBYTE pbProduct,
                                   PBYTE pbVersion,
                                   PBYTE pbItem)
    {
    PBYTE pbPath;
    BOOL  fOk = FALSE;

    if ((pbPath = SbsSoftwareValueA (pbFormat,  pbCompany,
                                     pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryPathDeleteA (pbPath);
        _mfree (pbPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareDeleteKeyW (PWORD pwFormat,
                                   PWORD pwCompany,
                                   PWORD pwProduct,
                                   PWORD pwVersion,
                                   PWORD pwItem)
    {
    PWORD pwPath;
    BOOL  fOk = FALSE;

    if ((pwPath = SbsSoftwareValueW (pwFormat,  pwCompany,
                                     pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryPathDeleteW (pwPath);
        _mfree (pwPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareDeleteValueA (PBYTE pbFormat,
                                     PBYTE pbCompany,
                                     PBYTE pbProduct,
                                     PBYTE pbVersion,
                                     PBYTE pbItem)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbValue = SbsSoftwareValueA (pbFormat,  pbCompany,
                                      pbProduct, pbVersion, pbItem))
        != NULL)
        {
        fOk = SbsRegistryDeleteValueA (HKEY_NULL, pbValue);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsSoftwareDeleteValueW (PWORD pwFormat,
                                     PWORD pwCompany,
                                     PWORD pwProduct,
                                     PWORD pwVersion,
                                     PWORD pwItem)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwValue = SbsSoftwareValueW (pwFormat,  pwCompany,
                                      pwProduct, pwVersion, pwItem))
        != NULL)
        {
        fOk = SbsRegistryDeleteValueW (HKEY_NULL, pwValue);
        _mfree (pwValue);
        }
    return fOk;
    }

// =================================================================
// VERSION INFO MANAGEMENT
// =================================================================

PVOID WINAPI SbsVersionRelease (PVOID pData)
    {
    return _mfree (pData);
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionTokenA (PBYTE  pbVersion,
                               PDWORD pdOffset)
    {
    DWORD dOffset = (pdOffset != NULL ? *pdOffset : 0);
    QWORD qToken  = 0;

    if (dOffset < _tsizeA (pbVersion))
        {
        while ((pbVersion [dOffset]       ) &&
               (pbVersion [dOffset] != '.') &&
               (pbVersion [dOffset] != ',')
               &&
               ((pbVersion [dOffset] < '0') ||
                (pbVersion [dOffset] > '9')))
            {
            dOffset++;
            }
        while ((pbVersion [dOffset] >= '0') &&
               (pbVersion [dOffset] <= '9'))
            {
            qToken *= 10;
            qToken += (pbVersion [dOffset++] - '0');
            }
        while ((pbVersion [dOffset]       ) &&
               (pbVersion [dOffset] != '.') &&
               (pbVersion [dOffset] != ','))
            {
            dOffset++;
            }
        }
    if (pdOffset != NULL) *pdOffset = dOffset;
    return qToken;
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionTokenW (PWORD  pwVersion,
                               PDWORD pdOffset)
    {
    DWORD dOffset = (pdOffset != NULL ? *pdOffset : 0);
    QWORD qToken  = 0;

    if (dOffset < _tsizeW (pwVersion))
        {
        while ((pwVersion [dOffset]       ) &&
               (pwVersion [dOffset] != '.') &&
               (pwVersion [dOffset] != ',')
               &&
               ((pwVersion [dOffset] < '0') ||
                (pwVersion [dOffset] > '9')))
            {
            dOffset++;
            }
        while ((pwVersion [dOffset] >= '0') &&
               (pwVersion [dOffset] <= '9'))
            {
            qToken *= 10;
            qToken += (pwVersion [dOffset++] - '0');
            }
        while ((pwVersion [dOffset]       ) &&
               (pwVersion [dOffset] != '.') &&
               (pwVersion [dOffset] != ','))
            {
            dOffset++;
            }
        }
    if (pdOffset != NULL) *pdOffset = dOffset;
    return qToken;
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionBinaryA (PBYTE pbVersion,
                                BOOL  fFlushLeft)
    {
    DWORD dOffset, dShift;
    QWORD qToken;
    QWORD qVersion = VERSION_NULL;

    if (pbVersion != NULL)
        {
        dOffset  =  0;
        dShift   = 64;
        qVersion =  0;

        do  {
            qToken  = SbsVersionTokenA (pbVersion, &dOffset);
            dShift -= 16;

            if (!pbVersion [dOffset++])
                {
                dOffset--;
                if (!fFlushLeft) dShift = 0;
                }
            qVersion += (qToken << dShift);
            }
        while (dShift);
        }
    return qVersion;
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionBinaryW (PWORD pwVersion,
                                BOOL  fFlushLeft)
    {
    DWORD dOffset, dShift;
    QWORD qToken;
    QWORD qVersion = VERSION_NULL;

    if (pwVersion != NULL)
        {
        dOffset  =  0;
        dShift   = 64;
        qVersion =  0;

        do  {
            qToken  = SbsVersionTokenW (pwVersion, &dOffset);
            dShift -= 16;

            if (!pwVersion [dOffset++])
                {
                dOffset--;
                if (!fFlushLeft) dShift = 0;
                }
            qVersion += (qToken << dShift);
            }
        while (dShift);
        }
    return qVersion;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsVersionTextA (QWORD qVersion)
    {
    PBYTE pbVersion = NULL;

    if (qVersion != VERSION_NULL)
        {
        pbVersion = _bprintfA (NULL, "%hu.%hu.%hu.%hu",
                               SBS_VERSION1 (qVersion),
                               SBS_VERSION2 (qVersion),
                               SBS_VERSION3 (qVersion),
                               SBS_VERSION4 (qVersion));
        }
    return pbVersion;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsVersionTextW (QWORD qVersion)
    {
    PWORD pwVersion = NULL;

    if (qVersion != VERSION_NULL)
        {
        pwVersion = _bprintfW (NULL, L"%hu.%hu.%hu.%hu",
                               SBS_VERSION1 (qVersion),
                               SBS_VERSION2 (qVersion),
                               SBS_VERSION3 (qVersion),
                               SBS_VERSION4 (qVersion));
        }
    return pwVersion;
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionFileA (PBYTE pbPath)
    {
    PVS_VERSIONINFO pvvi;
    QWORD           qVersion = VERSION_NULL;

    if ((pvvi = SbsVersionInfoA (pbPath, NULL)) != NULL)
        {
        qVersion   = pvvi->FixedFileInfo.dwFileVersionMS;
        qVersion <<= 32;
        qVersion  += pvvi->FixedFileInfo.dwFileVersionLS;

        _mfree (pvvi);
        }
    return qVersion;
    }

// -----------------------------------------------------------------

QWORD WINAPI SbsVersionFileW (PWORD pwPath)
    {
    PVS_VERSIONINFO pvvi;
    QWORD           qVersion = VERSION_NULL;

    if ((pvvi = SbsVersionInfoW (pwPath, NULL)) != NULL)
        {
        qVersion   = pvvi->FixedFileInfo.dwFileVersionMS;
        qVersion <<= 32;
        qVersion  += pvvi->FixedFileInfo.dwFileVersionLS;

        _mfree (pvvi);
        }
    return qVersion;
    }

// -----------------------------------------------------------------

PVS_VERSIONDATA WINAPI SbsVersionDataA (PBYTE pbPath,
                                        DWORD dIndex)
    {
    PVS_VERSIONINFO pvvi;
    PVS_VERSIONDATA pvvd = NULL;

    if ((pvvi = SbsVersionInfoA (pbPath, NULL)) != NULL)
        {
        pvvd = SbsVersionInfoDataA (pvvi, dIndex);
        _mfree (pvvi);
        }
    return pvvd;
    }

// -----------------------------------------------------------------

PVS_VERSIONDATA WINAPI SbsVersionDataW (PWORD pwPath,
                                        DWORD dIndex)
    {
    PVS_VERSIONINFO pvvi;
    PVS_VERSIONDATA pvvd = NULL;

    if ((pvvi = SbsVersionInfoW (pwPath, NULL)) != NULL)
        {
        pvvd = SbsVersionInfoDataW (pvvi, dIndex);
        _mfree (pvvi);
        }
    return pvvd;
    }

// -----------------------------------------------------------------

PVS_VERSIONINFO WINAPI SbsVersionInfoA (PBYTE  pbPath,
                                        PDWORD pdData)
    {
    PBYTE           pbPath1;
    BYTE            abPath [MAX_PATH];
    DWORD           dHandle;
    DWORD           dData = 0;
    PVS_VERSIONINFO pvvi  = NULL;

    pbPath1 = ((pbPath != NULL) && pbPath [0]
               ? pbPath
               : (GetModuleFileNameA (NULL, abPath, MAX_PATH)
                  ? abPath
                  : NULL));

    if ((pbPath1 != NULL)                                     &&
        (dData = GetFileVersionInfoSizeA (pbPath1, &dHandle)) &&
        ((pvvi = _mnew (dData)) != NULL))
        {
        if ((!GetFileVersionInfoA (pbPath1, 0, dData, pvvi)) ||
            (!SbsVersionVerify (pvvi)))
            {
            pvvi = _mfree (pvvi);
            }
        }
    if (pdData != NULL) *pdData = (pvvi != NULL ? dData : 0);
    return pvvi;
    }

// -----------------------------------------------------------------

PVS_VERSIONINFO WINAPI SbsVersionInfoW (PWORD  pwPath,
                                        PDWORD pdData)
    {
    PWORD           pwPath1;
    WORD            awPath [MAX_PATH];
    DWORD           dHandle;
    DWORD           dData = 0;
    PVS_VERSIONINFO pvvi  = NULL;

    pwPath1 = ((pwPath != NULL) && pwPath [0]
               ? pwPath
               : (GetModuleFileNameW (NULL, awPath, MAX_PATH)
                  ? awPath
                  : NULL));

    if ((pwPath1 != NULL)                                     &&
        (dData = GetFileVersionInfoSizeW (pwPath1, &dHandle)) &&
        ((pvvi = _mnew (dData)) != NULL))
        {
        if ((!GetFileVersionInfoW (pwPath1, 0, dData, pvvi)) ||
            (!SbsVersionVerify (pvvi)))
            {
            pvvi = _mfree (pvvi);
            }
        }
    if (pdData != NULL) *pdData = (pvvi != NULL ? dData : 0);
    return pvvi;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsVersionAlign (DWORD dSize)
    {
    return (((dSize - 1) >> 2) + 1) << 2;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsVersionVerify (PVS_VERSIONINFO pvvi)
    {
    return (pvvi != NULL)                            &&
           (pvvi->Header.wLength >= VS_VERSIONINFO_) &&
           (!_tcompW (pvvi->Header.awKey, L"VS_VERSION_INFO"));
    }

// -----------------------------------------------------------------

PVS_STRINGFILEINFO WINAPI SbsVersionStrings (PVS_VERSIONINFO pvvi)
    {
    DWORD              dOffset = 0;
    PVS_STRINGFILEINFO pvsfi   = NULL;

    if (SbsVersionVerify (pvvi))
        {
        while (VS_VERSIONINFO_ + dOffset + VS_STRINGFILEINFO_
               <= pvvi->Header.wLength)
            {
            pvsfi = (PVS_STRINGFILEINFO) (pvvi->abData + dOffset);

            if (!_tcompW (pvsfi->Header.awKey, L"StringFileInfo"))
                break;

            dOffset += SbsVersionAlign (pvsfi->Header.wLength);
            pvsfi    = NULL;
            }
        }
    return pvsfi;
    }

// -----------------------------------------------------------------

PVS_VARFILEINFO WINAPI SbsVersionVars (PVS_VERSIONINFO pvvi)
    {
    DWORD           dOffset = 0;
    PVS_VARFILEINFO pvvfi   = NULL;

    if (SbsVersionVerify (pvvi))
        {
        while (VS_VERSIONINFO_ + dOffset + VS_VARFILEINFO_
               <= pvvi->Header.wLength)
            {
            pvvfi = (PVS_VARFILEINFO) (pvvi->abData + dOffset);

            if (!_tcompW (pvvfi->Header.awKey, L"VarFileInfo"))
                break;

            dOffset += SbsVersionAlign (pvvfi->Header.wLength);
            pvvfi    = NULL;
            }
        }
    return pvvfi;
    }

// -----------------------------------------------------------------

PVS_TRANSLATION WINAPI SbsVersionTranslation (PVS_VERSIONINFO pvvi,
                                              PDWORD       pdCount)
    {
    PVS_VARFILEINFO pvvfi;
    DWORD           dCount = 0;
    PVS_TRANSLATION pvt    = NULL;

    if (((pvvfi = SbsVersionVars (pvvi)) != NULL) &&
        (!_tcompW (pvvfi->Var.Header.awKey, L"Translation")))
        {
        if (dCount = ((DWORD) &pvvfi->Var                +
                      (DWORD)  pvvfi->Var.Header.wLength -
                      (DWORD)  pvvfi->Var.Translation)
                     / VS_TRANSLATION_)
            {
            pvt = pvvfi->Var.Translation;
            }
        }
    if (pdCount != NULL) *pdCount = dCount;
    return pvt;
    }

// -----------------------------------------------------------------

PVS_STRINGTABLE WINAPI SbsVersionTableA (PVS_VERSIONINFO pvvi,
                                         PBYTE           pbTable)
    {
    PWORD           pwTable = NULL;
    PVS_STRINGTABLE pvst    = NULL;

    if ((pbTable == NULL) ||
        ((pwTable = _tnewA2W (pbTable, -1)) != NULL))
        {
        pvst = SbsVersionTableW (pvvi, pwTable);
        _mfree (pwTable);
        }
    return pvst;
    }

// -----------------------------------------------------------------

PVS_STRINGTABLE WINAPI SbsVersionTableW (PVS_VERSIONINFO pvvi,
                                         PWORD           pwTable)
    {
    DWORD              dOffset;
    PVS_STRINGFILEINFO pvsfi;
    PVS_STRINGTABLE    pvst = NULL;

    if ((pvsfi = SbsVersionStrings (pvvi)) != NULL)
        {
        dOffset = (DWORD_PTR) &pvsfi->StringTable -
                  (DWORD_PTR) pvsfi;

        while (dOffset + VS_STRINGTABLE_ <= pvsfi->Header.wLength)
            {
            pvst = (PVS_STRINGTABLE) ((PBYTE) pvsfi + dOffset);

            if ((pwTable == NULL) ||
                (!_tcompW (pvst->Header.awKey, pwTable))) break;

            dOffset += SbsVersionAlign (pvst->Header.wLength);
            pvst     = NULL;
            }
        }
    return pvst;
    }

// -----------------------------------------------------------------

PVS_STRINGTABLE WINAPI SbsVersionTableEx (PVS_VERSIONINFO pvvi,
                                          DWORD           dIndex,
                                          PDWORD          pdCount)
    {
    WORD            awTable [8+1] = L"";
    DWORD           dCount;
    PVS_TRANSLATION pvt  = SbsVersionTranslation (pvvi, &dCount);
    PVS_STRINGTABLE pvst = NULL;

    if (dIndex == -1)
        {
        pvst = SbsVersionTableW (pvvi, NULL);
        }
    else
        {
        if ((pvt != NULL) && (dIndex < dCount))
            {
            _sprintfW (awTable, L"%04hX%04hX",
                       pvt [dIndex].wLanguage,
                       pvt [dIndex].wCodePage);

            pvst = SbsVersionTableW (pvvi, awTable);
            }
        }
    if (pdCount != NULL) *pdCount = dCount;
    return pvst;
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionTableStringA (PVS_STRINGTABLE pvst,
                                          PBYTE           pbName)
    {
    PWORD      pwName = NULL;
    PVS_STRING pvs    = NULL;

    if ((pbName == NULL) ||
        ((pwName = _tnewA2W (pbName, -1)) != NULL))
        {
        pvs = SbsVersionTableStringW (pvst, pwName);
        _mfree (pwName);
        }
    return pvs;
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionTableStringW (PVS_STRINGTABLE pvst,
                                          PWORD           pwName)
    {
    DWORD      dOffset;
    PVS_STRING pvs = NULL;

    if ((pvst != NULL) && (pwName != NULL))
        {
        dOffset = (DWORD_PTR) &pvst->String - (DWORD_PTR) pvst;

        while (dOffset + VS_STRING_ <= pvst->Header.wLength)
            {
            pvs = (PVS_STRING) ((PBYTE) pvst + dOffset);

            if (!_tcompW (pvs->Header.awKey, pwName)) break;

            dOffset += SbsVersionAlign (pvs->Header.wLength);
            pvs      = NULL;
            }
        }
    return pvs;
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionStringA (PVS_VERSIONINFO pvvi,
                                     PBYTE           pbName,
                                     PBYTE           pbTable)
    {
    return SbsVersionTableStringA
               (SbsVersionTableA (pvvi, pbTable),
                pbName);
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionStringW (PVS_VERSIONINFO pvvi,
                                     PWORD           pwName,
                                     PWORD           pwTable)
    {
    return SbsVersionTableStringW
               (SbsVersionTableW (pvvi, pwTable),
                pwName);
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionStringExA (PVS_VERSIONINFO pvvi,
                                       PBYTE           pbName,
                                       DWORD           dIndex,
                                       PDWORD          pdCount)
    {
    return SbsVersionTableStringA
               (SbsVersionTableEx (pvvi, dIndex, pdCount),
                pbName);
    }

// -----------------------------------------------------------------

PVS_STRING WINAPI SbsVersionStringExW (PVS_VERSIONINFO pvvi,
                                       PWORD           pwName,
                                       DWORD           dIndex,
                                       PDWORD          pdCount)
    {
    return SbsVersionTableStringW
               (SbsVersionTableEx (pvvi, dIndex, pdCount),
                pwName);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsVersionStringValue (PVS_STRING pvs,
                                    PDWORD     pdData)
    {
    DWORD n;
    DWORD dData  = 0;
    PWORD pwData = NULL;

    if (pvs != NULL)
        {
        n = SbsVersionAlign ((_tsizeW (pvs->awData) + 1) * WORD_);

        pwData = (PWORD) ((PBYTE) pvs->awData + n);
        dData  = _tsizeW (pwData) + 1;
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsVersionStringData (PVS_STRING pvs)
    {
    DWORD dData  = 0;
    PWORD pwData = SbsVersionStringValue (pvs, &dData);

    return dData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsVersionStringCopyA (PVS_STRING pvs)
    {
    DWORD dData  = 0;
    PWORD pwData = SbsVersionStringValue (pvs, &dData);

    return _tnewW2A (pwData, dData);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsVersionStringCopyW (PVS_STRING pvs)
    {
    DWORD dData  = 0;
    PWORD pwData = SbsVersionStringValue (pvs, &dData);

    return _tnewW2W (pwData, dData);
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsVersionLanguageA (DWORD  dId,
                                  PDWORD pdData)
    {
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    if ((pbData = _tnewA (SBS_STRING)) != NULL)
        {
        if (VerLanguageNameA (dId, pbData, SBS_STRING))
            {
            dData = _tsizeA (pbData) + 1;
            }
        else
            {
            pbData = _mfree (pbData);
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsVersionLanguageW (DWORD  dId,
                                  PDWORD pdData)
    {
    DWORD dData  = 0;
    PWORD pwData = NULL;

    if ((pwData = _tnewW (SBS_STRING)) != NULL)
        {
        if (VerLanguageNameW (dId, pwData, SBS_STRING))
            {
            dData = _tsizeW (pwData) + 1;
            }
        else
            {
            pwData = _mfree (pwData);
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsVersionCodePageA (DWORD  dId,
                                  PDWORD pdData)
    {
    DWORD i;
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    for (i = 0; gavcpCodePages [i].pbName != NULL; i++)
        {
        if (gavcpCodePages [i].dId == dId)
            {
            if ((pbData = _tnewA2A (gavcpCodePages [i].pbName, -1))
                != NULL)
                {
                dData = _tsizeA (pbData) + 1;
                }
            break;
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsVersionCodePageW (DWORD  dId,
                                  PDWORD pdData)
    {
    DWORD i;
    DWORD dData  = 0;
    PWORD pwData = NULL;

    for (i = 0; gavcpCodePages [i].pbName != NULL; i++)
        {
        if (gavcpCodePages [i].dId == dId)
            {
            if ((pwData = _tnewA2W (gavcpCodePages [i].pbName, -1))
                != NULL)
                {
                dData = _tsizeW (pwData) + 1;
                }
            break;
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PVS_VERSIONDATA WINAPI SbsVersionInfoDataA (PVS_VERSIONINFO pvvi,
                                            DWORD           dIndex)
    {
    DWORD           dCount, dIndex1, dLanguage, dCodePage;
    DWORD           dData,  dBuffer;
    PBYTE           pbData, pbBuffer;
    PWORD           pwData;
    VS_VERSIONDATA  vvd;
    PVS_TRANSLATION pvt;
    PVS_VERSIONDATA pvvd = NULL;

    if (((pvt = SbsVersionTranslation (pvvi, &dCount)) != NULL) &&
        ((dIndex1 = (dIndex == -1 ? 0 : dIndex)) < dCount))
        {
        vvd.vvqFile.dVersionL =
            pvvi->FixedFileInfo.dwFileVersionLS;

        vvd.vvqFile.dVersionH =
            pvvi->FixedFileInfo.dwFileVersionMS;

        vvd.vvqProduct.dVersionL =
            pvvi->FixedFileInfo.dwProductVersionLS;

        vvd.vvqProduct.dVersionH =
            pvvi->FixedFileInfo.dwProductVersionMS;

        vvd.dLanguage   = pvt [dIndex1].wLanguage;
        vvd.dCodePage   = pvt [dIndex1].wCodePage;
        vvd.dTableIndex = dIndex;
        vvd.dTableCount = dCount;

        vvd.pbLanguage = SbsVersionLanguageA (vvd.dLanguage,
                                              &dLanguage);

        vvd.pbCodePage = SbsVersionCodePageA (vvd.dCodePage,
                                              &dCodePage);

        dBuffer = dLanguage + dCodePage
            +
            SbsVersionStringData
                (vvd.pvsComments =
                 SbsVersionStringExW (pvvi, L"Comments",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsCompanyName =
                 SbsVersionStringExW (pvvi, L"CompanyName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsFileDescription =
                 SbsVersionStringExW (pvvi, L"FileDescription",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsFileVersion =
                 SbsVersionStringExW (pvvi, L"FileVersion",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsInternalName =
                 SbsVersionStringExW (pvvi, L"InternalName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsLegalCopyright =
                 SbsVersionStringExW (pvvi, L"LegalCopyright",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsLegalTrademarks =
                 SbsVersionStringExW (pvvi, L"LegalTrademarks",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsOriginalFilename =
                 SbsVersionStringExW (pvvi, L"OriginalFilename",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsPrivateBuild =
                 SbsVersionStringExW (pvvi, L"PrivateBuild",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsProductName =
                 SbsVersionStringExW (pvvi, L"ProductName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsProductVersion =
                 SbsVersionStringExW (pvvi, L"ProductVersion",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsSpecialBuild =
                 SbsVersionStringExW (pvvi, L"SpecialBuild",
                                      dIndex, NULL));

        if ((pvvd = _mnew (VS_VERSIONDATA_A (dBuffer))) != NULL)
            {
            *pvvd    = vvd;
            pbBuffer = (PBYTE) pvvd->awBuffer;

            if ((pbData = pvvd->pbLanguage) != NULL)
                {
                pvvd->pbLanguage = pbBuffer;
                pbBuffer = _tcopyA2A (pbBuffer, pbData, dLanguage);
                }
            if ((pbData = pvvd->pbCodePage) != NULL)
                {
                pvvd->pbCodePage = pbBuffer;
                pbBuffer = _tcopyA2A (pbBuffer, pbData, dCodePage);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsComments, &dData))
                != NULL)
                {
                pvvd->pbComments = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsCompanyName, &dData))
                != NULL)
                {
                pvvd->pbCompanyName = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsFileDescription, &dData))
                != NULL)
                {
                pvvd->pbFileDescription = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsFileVersion, &dData))
                != NULL)
                {
                pvvd->pbFileVersion = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsInternalName, &dData))
                != NULL)
                {
                pvvd->pbInternalName = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsLegalCopyright, &dData))
                != NULL)
                {
                pvvd->pbLegalCopyright = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsLegalTrademarks, &dData))
                != NULL)
                {
                pvvd->pbLegalTrademarks = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsOriginalFilename, &dData))
                != NULL)
                {
                pvvd->pbOriginalFilename = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsPrivateBuild, &dData))
                != NULL)
                {
                pvvd->pbPrivateBuild = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsProductName, &dData))
                != NULL)
                {
                pvvd->pbProductName = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsProductVersion, &dData))
                != NULL)
                {
                pvvd->pbProductVersion = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsSpecialBuild, &dData))
                != NULL)
                {
                pvvd->pbSpecialBuild = pbBuffer;
                pbBuffer = _tcopyW2A (pbBuffer, pwData, dData);
                }
            }
        _mfree (vvd.pbLanguage);
        _mfree (vvd.pbCodePage);
        }
    return pvvd;
    }

// -----------------------------------------------------------------

PVS_VERSIONDATA WINAPI SbsVersionInfoDataW (PVS_VERSIONINFO pvvi,
                                            DWORD           dIndex)
    {
    DWORD           dCount, dIndex1, dLanguage, dCodePage;
    DWORD           dData,  dBuffer;
    PWORD           pwData, pwBuffer;
    VS_VERSIONDATA  vvd;
    PVS_TRANSLATION pvt;
    PVS_VERSIONDATA pvvd = NULL;

    if (((pvt = SbsVersionTranslation (pvvi, &dCount)) != NULL) &&
        ((dIndex1 = (dIndex == -1 ? 0 : dIndex)) < dCount))
        {
        vvd.vvqFile.dVersionL =
            pvvi->FixedFileInfo.dwFileVersionLS;

        vvd.vvqFile.dVersionH =
            pvvi->FixedFileInfo.dwFileVersionMS;

        vvd.vvqProduct.dVersionL =
            pvvi->FixedFileInfo.dwProductVersionLS;

        vvd.vvqProduct.dVersionH =
            pvvi->FixedFileInfo.dwProductVersionMS;

        vvd.dLanguage   = pvt [dIndex1].wLanguage;
        vvd.dCodePage   = pvt [dIndex1].wCodePage;
        vvd.dTableIndex = dIndex;
        vvd.dTableCount = dCount;

        vvd.pwLanguage = SbsVersionLanguageW (vvd.dLanguage,
                                              &dLanguage);

        vvd.pwCodePage = SbsVersionCodePageW (vvd.dCodePage,
                                              &dCodePage);

        dBuffer = dLanguage + dCodePage
            +
            SbsVersionStringData
                (vvd.pvsComments =
                 SbsVersionStringExW (pvvi, L"Comments",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsCompanyName =
                 SbsVersionStringExW (pvvi, L"CompanyName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsFileDescription =
                 SbsVersionStringExW (pvvi, L"FileDescription",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsFileVersion =
                 SbsVersionStringExW (pvvi, L"FileVersion",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsInternalName =
                 SbsVersionStringExW (pvvi, L"InternalName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsLegalCopyright =
                 SbsVersionStringExW (pvvi, L"LegalCopyright",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsLegalTrademarks =
                 SbsVersionStringExW (pvvi, L"LegalTrademarks",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsOriginalFilename =
                 SbsVersionStringExW (pvvi, L"OriginalFilename",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsPrivateBuild =
                 SbsVersionStringExW (pvvi, L"PrivateBuild",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsProductName =
                 SbsVersionStringExW (pvvi, L"ProductName",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsProductVersion =
                 SbsVersionStringExW (pvvi, L"ProductVersion",
                                      dIndex, NULL))
            +
            SbsVersionStringData
                (vvd.pvsSpecialBuild =
                 SbsVersionStringExW (pvvi, L"SpecialBuild",
                                      dIndex, NULL));

        if ((pvvd = _mnew (VS_VERSIONDATA_W (dBuffer))) != NULL)
            {
            *pvvd    = vvd;
            pwBuffer = pvvd->awBuffer;

            if ((pwData = pvvd->pwLanguage) != NULL)
                {
                pvvd->pwLanguage = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dLanguage);
                }
            if ((pwData = pvvd->pwCodePage) != NULL)
                {
                pvvd->pwCodePage = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dCodePage);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsComments, &dData))
                != NULL)
                {
                pvvd->pwComments = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsCompanyName, &dData))
                != NULL)
                {
                pvvd->pwCompanyName = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsFileDescription, &dData))
                != NULL)
                {
                pvvd->pwFileDescription = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsFileVersion, &dData))
                != NULL)
                {
                pvvd->pwFileVersion = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsInternalName, &dData))
                != NULL)
                {
                pvvd->pwInternalName = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsLegalCopyright, &dData))
                != NULL)
                {
                pvvd->pwLegalCopyright = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsLegalTrademarks, &dData))
                != NULL)
                {
                pvvd->pwLegalTrademarks = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsOriginalFilename, &dData))
                != NULL)
                {
                pvvd->pwOriginalFilename = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsPrivateBuild, &dData))
                != NULL)
                {
                pvvd->pwPrivateBuild = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsProductName, &dData))
                != NULL)
                {
                pvvd->pwProductName = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsProductVersion, &dData))
                != NULL)
                {
                pvvd->pwProductVersion = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            if ((pwData = SbsVersionStringValue
                              (pvvd->pvsSpecialBuild, &dData))
                != NULL)
                {
                pvvd->pwSpecialBuild = pwBuffer;
                pwBuffer = _tcopyW2W (pwBuffer, pwData, dData);
                }
            }
        _mfree (vvd.pwLanguage);
        _mfree (vvd.pwCodePage);
        }
    return pvvd;
    }

// -----------------------------------------------------------------

INT WINAPI SbsVersionBoxA (HWND   hWnd,
                           UINT   uiType,
                           PBYTE  pbCaption,
                           PBYTE  pbPath,
                           DWORD  dIndex,
                           PDWORD pdCount)
    {
    PVS_VERSIONDATA pvvd;
    DWORD           dCount = 0;
    INT             iId    = IDABORT;

    if ((pvvd = SbsVersionDataA (pbPath, dIndex)) != NULL)
        {
        dCount = pvvd->dTableCount;

        iId = _mprintfA (hWnd, uiType, pbCaption,

             "File Name\t<%s>\n"
             "File Version\t%hu.%hu.%hu.%hu\n"
             "Product Version\t%hu.%hu.%hu.%hu\n"
             "Translation Code\t%04lX-%04lX  (#%lu of %lu)\n"
             "Language Name\t%s%s%s\n"
             "Code Page Name\t%s%s%s\n"
             "\n"
             "Comments\t%s%s%s\n"
             "CompanyName\t%s%s%s\n"
             "FileDescription\t%s%s%s\n"
             "FileVersion\t%s%s%s\n"
             "InternalName\t%s%s%s\n"
             "LegalCopyright\t%s%s%s\n"
             "LegalTrademarks\t%s%s%s\n"
             "OriginalFilename\t%s%s%s\n"
             "PrivateBuild\t%s%s%s\n"
             "ProductName\t%s%s%s\n"
             "ProductVersion\t%s%s%s\n"
             "SpecialBuild\t%s%s%s",
             _ffileA (pbPath),
             pvvd->vvqFile.wVersionHH,
             pvvd->vvqFile.wVersionH,
             pvvd->vvqFile.wVersionL,
             pvvd->vvqFile.wVersionLL,
             pvvd->vvqProduct.wVersionHH,
             pvvd->vvqProduct.wVersionH,
             pvvd->vvqProduct.wVersionL,
             pvvd->vvqProduct.wVersionLL,
             pvvd->dLanguage,
             pvvd->dCodePage,
             pvvd->dTableIndex + 1,
             pvvd->dTableCount,
             STRTOKEN (pvvd->pbLanguage,         "<", ">", NULL),
             STRTOKEN (pvvd->pbCodePage,         "<", ">", NULL),
             STRTOKEN (pvvd->pbComments,         "<", ">", NULL),
             STRTOKEN (pvvd->pbCompanyName,      "<", ">", NULL),
             STRTOKEN (pvvd->pbFileDescription,  "<", ">", NULL),
             STRTOKEN (pvvd->pbFileVersion,      "<", ">", NULL),
             STRTOKEN (pvvd->pbInternalName,     "<", ">", NULL),
             STRTOKEN (pvvd->pbLegalCopyright,   "<", ">", NULL),
             STRTOKEN (pvvd->pbLegalTrademarks,  "<", ">", NULL),
             STRTOKEN (pvvd->pbOriginalFilename, "<", ">", NULL),
             STRTOKEN (pvvd->pbPrivateBuild,     "<", ">", NULL),
             STRTOKEN (pvvd->pbProductName,      "<", ">", NULL),
             STRTOKEN (pvvd->pbProductVersion,   "<", ">", NULL),
             STRTOKEN (pvvd->pbSpecialBuild,     "<", ">", NULL));

        _mfree (pvvd);
        }
    if (pdCount != NULL) *pdCount = dCount;
    return iId;
    }

// -----------------------------------------------------------------

INT WINAPI SbsVersionBoxW (HWND   hWnd,
                           UINT   uiType,
                           PWORD  pwCaption,
                           PWORD  pwPath,
                           DWORD  dIndex,
                           PDWORD pdCount)
    {
    PVS_VERSIONDATA pvvd;
    DWORD           dCount = 0;
    INT             iId    = IDABORT;

    if ((pvvd = SbsVersionDataW (pwPath, dIndex)) != NULL)
        {
        dCount = pvvd->dTableCount;

        iId = _mprintfW (hWnd, uiType, pwCaption,

             L"File Name\t<%s>\n"
             L"File Version\t%hu.%hu.%hu.%hu\n"
             L"Product Version\t%hu.%hu.%hu.%hu\n"
             L"Translation Code\t%04lX-%04lX  (#%lu of %lu)\n"
             L"Language Name\t%s%s%s\n"
             L"Code Page Name\t%s%s%s\n"
             L"\n"
             L"Comments\t%s%s%s\n"
             L"CompanyName\t%s%s%s\n"
             L"FileDescription\t%s%s%s\n"
             L"FileVersion\t%s%s%s\n"
             L"InternalName\t%s%s%s\n"
             L"LegalCopyright\t%s%s%s\n"
             L"LegalTrademarks\t%s%s%s\n"
             L"OriginalFilename\t%s%s%s\n"
             L"PrivateBuild\t%s%s%s\n"
             L"ProductName\t%s%s%s\n"
             L"ProductVersion\t%s%s%s\n"
             L"SpecialBuild\t%s%s%s",
             _ffileW (pwPath),
             pvvd->vvqFile.wVersionHH,
             pvvd->vvqFile.wVersionH,
             pvvd->vvqFile.wVersionL,
             pvvd->vvqFile.wVersionLL,
             pvvd->vvqProduct.wVersionHH,
             pvvd->vvqProduct.wVersionH,
             pvvd->vvqProduct.wVersionL,
             pvvd->vvqProduct.wVersionLL,
             pvvd->dLanguage,
             pvvd->dCodePage,
             pvvd->dTableIndex + 1,
             pvvd->dTableCount,
             STRTOKEN (pvvd->pwLanguage,         L"<", L">", NULL),
             STRTOKEN (pvvd->pwCodePage,         L"<", L">", NULL),
             STRTOKEN (pvvd->pwComments,         L"<", L">", NULL),
             STRTOKEN (pvvd->pwCompanyName,      L"<", L">", NULL),
             STRTOKEN (pvvd->pwFileDescription,  L"<", L">", NULL),
             STRTOKEN (pvvd->pwFileVersion,      L"<", L">", NULL),
             STRTOKEN (pvvd->pwInternalName,     L"<", L">", NULL),
             STRTOKEN (pvvd->pwLegalCopyright,   L"<", L">", NULL),
             STRTOKEN (pvvd->pwLegalTrademarks,  L"<", L">", NULL),
             STRTOKEN (pvvd->pwOriginalFilename, L"<", L">", NULL),
             STRTOKEN (pvvd->pwPrivateBuild,     L"<", L">", NULL),
             STRTOKEN (pvvd->pwProductName,      L"<", L">", NULL),
             STRTOKEN (pvvd->pwProductVersion,   L"<", L">", NULL),
             STRTOKEN (pvvd->pwSpecialBuild,     L"<", L">", NULL));

        _mfree (pvvd);
        }
    if (pdCount != NULL) *pdCount = dCount;
    return iId;
    }

// =================================================================
// BITMAP MANAGEMENT
// =================================================================

BOOL WINAPI SbsBitmapInitialize (PSBS_BITMAP psb,
                                 DWORD       dRows,
                                 DWORD       dColumns,
                                 DWORD       dColors)
    {
    DWORD i;
    BOOL  fOk = FALSE;

    if ((psb != NULL) && dRows && dColumns &&
        ((dColors == 0x00000002) || (dColors == 0x00000010) ||
         (dColors == 0x00000100) || (dColors == 0x00010000) ||
         (dColors == 0x01000000) || (dColors == 0x00000000)))
        {
        psb->dRows    = dRows;
        psb->dColumns = dColumns;
        psb->dColors  = dColors;
        psb->dPalette = (dColors <= 256 ? dColors : 0);

        for (psb->dBitsPerPixel = 0, i   = 1; i && (!(i & dColors));
             psb->dBitsPerPixel++,   i <<= 1);

        psb->dBytesPerPixel      = psb->dBitsPerPixel / BYTE__;
        psb->dPixelsPerByte      = BYTE__ / psb->dBitsPerPixel;

        psb->dTotal              = SBS_BITMAP_ +
                                   (psb->dPalette * RGBQUAD_);

        psb->dBitmap             = SBS_BITMAP_HEADER +
                                   (psb->dPalette * RGBQUAD_);

        psb->pBitmap             = (PBYTE) &psb->bfh + psb->dBitmap;
        psb->pbi                 = (PBITMAPINFO) &psb->bih;
        psb->hPalette            = NULL;
        psb->wReserved           = 0;

        psb->bfh.bfType          = MAGIC_BMP;
        psb->bfh.bfSize          = 0;
        psb->bfh.bfReserved1     = 0;
        psb->bfh.bfReserved2     = 0;
        psb->bfh.bfOffBits       = psb->dBitmap;

        psb->bih.biSize          = BITMAPINFOHEADER_;
        psb->bih.biWidth         = dColumns;
        psb->bih.biHeight        = dRows;
        psb->bih.biPlanes        = 1;
        psb->bih.biBitCount      = (WORD) psb->dBitsPerPixel;
        psb->bih.biCompression   = BI_RGB;
        psb->bih.biSizeImage     = 0;
        psb->bih.biXPelsPerMeter = 0;
        psb->bih.biYPelsPerMeter = 0;
        psb->bih.biClrUsed       = psb->dPalette;
        psb->bih.biClrImportant  = psb->dPalette;

        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsBitmapScan (PSBS_BITMAP psb,
                            DWORD       dColumns)
    {
    DWORD n = 0;

    if ((psb != NULL) && (n = dColumns))
        {
        if (psb->dPixelsPerByte)
            {
            n = ((n - 1) / psb->dPixelsPerByte) + 1;
            }
        else
            {
            n *= psb->dBytesPerPixel;
            }
        n = (((n - 1) / DWORD_) + 1) * DWORD_;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsBitmapColors (DWORD dBitsPerPixel)
    {
    DWORD i;
    DWORD dColors = 1;

    for (i = 0; i < dBitsPerPixel; i++) dColors <<= 1;
    return dColors;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsBitmapData (PSBS_BITMAP psb,
                            PDWORD      pdData)
    {
    DWORD dData = 0;
    PVOID pData = NULL;

    if (psb != NULL)
        {
        pData = &psb->bfh;
        dData =  psb->bfh.bfSize;
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PLOGPALETTE WINAPI SbsBitmapPaletteData (PSBS_BITMAP psb)
    {
    DWORD         dBytes, i;
    PPALETTEENTRY ppe;
    PRGBQUAD      prq;
    PLOGPALETTE   plp = NULL;

    if ((psb != NULL) && psb->dColors && (psb->dColors <= 256))
        {
        dBytes = LOGPALETTE_ + ((psb->dColors - 1) * PALETTEENTRY_);

        if ((plp = _mnew (dBytes)) != NULL)
            {
            plp->palVersion    = VERSION_PALETTE;
            plp->palNumEntries = (WORD) psb->dColors;

            for (i = 0; i < psb->dColors; i++)
                {
                ppe = plp->palPalEntry + i;
                prq = psb->Palette     + i;

                ppe->peRed   = prq->rgbRed;
                ppe->peGreen = prq->rgbGreen;
                ppe->peBlue  = prq->rgbBlue;
                ppe->peFlags = 0;
                }
            }
        }
    return plp;
    }

// -----------------------------------------------------------------

HPALETTE WINAPI SbsBitmapPaletteHandle (PSBS_BITMAP psb)
    {
    PLOGPALETTE plp;
    HPALETTE    hPalette = NULL;

    if ((plp = SbsBitmapPaletteData (psb)) != NULL)
        {
        hPalette = CreatePalette (plp);
        _mfree (plp);
        }
    return hPalette;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapCreate (PBITMAPFILEHEADER pbfh,
                                    DWORD             dSize)
    {
    DWORD             dData;
    PBITMAPINFOHEADER pbih;
    PRGBQUAD          Palette;
    SBS_BITMAP        sb;
    PSBS_BITMAP       psb = NULL;

    if (pbfh != NULL)
        {
        pbih    = (PBITMAPINFOHEADER)
                  ((PBYTE) pbfh + BITMAPFILEHEADER_);

        Palette = (PRGBQUAD)
                  ((PBYTE) pbih + BITMAPINFOHEADER_);

        dData   = dSize - ((DWORD) Palette - (DWORD) pbfh);

        if ((dSize > BITMAPFILEHEADER_ + BITMAPINFOHEADER_)
            &&
            SbsBitmapInitialize (&sb, pbih->biHeight, pbih->biWidth,
                                 SbsBitmapColors (pbih->biBitCount))
            &&
            ((psb = _mnew (SBS_BITMAP_ + dData)) != NULL))
            {
            _mcopy (psb, &sb, SBS_BITMAP_OFFSET);

            psb->bfh = *pbfh;
            psb->bih = *pbih;

            _mcopy (psb->Palette, Palette, dData);

            psb->pBitmap  = (PBYTE) &psb->bfh + psb->dBitmap;
            psb->pbi      = (PBITMAPINFO) &psb->bih;
            psb->hPalette = SbsBitmapPaletteHandle (psb);
            }
        }
    return psb;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapSaveA (PSBS_BITMAP psb,
                            PBYTE       pbPath)
    {
    PVOID pData;
    DWORD dData;
    BOOL  fOk = FALSE;

    if ((pData = SbsBitmapData (psb, &dData)) != NULL)
        {
        fOk = _fsaveA (pbPath, pData, dData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapSaveW (PSBS_BITMAP psb,
                            PWORD       pwPath)
    {
    PVOID pData;
    DWORD dData;
    BOOL  fOk = FALSE;

    if ((pData = SbsBitmapData (psb, &dData)) != NULL)
        {
        fOk = _fsaveW (pwPath, pData, dData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapLoadA (PBYTE pbPath)
    {
    DWORD             dSize;
    PBITMAPFILEHEADER pbfh;
    PSBS_BITMAP       psb = NULL;

    if ((pbfh = _floadA (pbPath, &dSize)) != NULL)
        {
        psb = SbsBitmapCreate (pbfh, dSize);
        _mfree (pbfh);
        }
    return psb;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapLoadW (PWORD pwPath)
    {
    DWORD             dSize;
    PBITMAPFILEHEADER pbfh;
    PSBS_BITMAP       psb = NULL;

    if ((pbfh = _floadW (pwPath, &dSize)) != NULL)
        {
        psb = SbsBitmapCreate (pbfh, dSize);
        _mfree (pbfh);
        }
    return psb;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapUnload (PSBS_BITMAP psb)
    {
    if (psb != NULL)
        {
        if (psb->hPalette != NULL) DeleteObject (psb->hPalette);
        _mfree (psb);
        }
    return NULL;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapDefaultA (HINSTANCE hInstance,
                                      PBYTE     pbItem)
    {
    BYTE        abPath [MAX_PATH];
    PSBS_BITMAP psb = NULL;

    if (_fpathA (hInstance, SAFEPTR (pbItem, gabBmp),
                 abPath, MAX_PATH))
        {
        psb = SbsBitmapLoadA (abPath);
        }
    return psb;
    }

// -----------------------------------------------------------------

PSBS_BITMAP WINAPI SbsBitmapDefaultW (HINSTANCE hInstance,
                                      PWORD     pwItem)
    {
    WORD        awPath [MAX_PATH];
    PSBS_BITMAP psb = NULL;

    if (_fpathW (hInstance, SAFEPTR (pwItem, gawBmp),
                 awPath, MAX_PATH))
        {
        psb = SbsBitmapLoadW (awPath);
        }
    return psb;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapDraw (PSBS_BITMAP psb,
                           HDC         hDC,
                           PRECT       pr)
    {
    INT      iMode;
    HPALETTE hPalette = NULL;
    BOOL     fOk      = FALSE;

    if ((psb != NULL) && (hDC != NULL) && (pr != NULL))
        {
        if (psb->hPalette != NULL)
            {
            hPalette = SelectPalette (hDC, psb->hPalette, FALSE);
            }
        RealizePalette (hDC);

        iMode = SetStretchBltMode (hDC, COLORONCOLOR);

        fOk = (StretchDIBits (hDC, SBS_PRECT (pr),
                              0, 0, psb->dColumns, psb->dRows,
                              psb->pBitmap, psb->pbi,
                              DIB_RGB_COLORS, SRCCOPY)
               != GDI_ERROR);

        if (iMode) SetStretchBltMode (hDC, iMode);
        if (hPalette != NULL) SelectPalette (hDC, hPalette, FALSE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapRestore (PSBS_BITMAP psb,
                              HWND        hWnd)
    {
    return (psb != NULL) &&
           SbsWindowZoom (hWnd, psb->dColumns, psb->dRows, TRUE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapStretch (PSBS_BITMAP psb,
                              HWND        hWnd,
                              DWORD       dFactor)
    {
    return (psb != NULL) &&
           SbsWindowZoom (hWnd, psb->dColumns * dFactor,
                                psb->dRows    * dFactor, TRUE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsBitmapShrink (PSBS_BITMAP psb,
                             HWND        hWnd,
                             DWORD       dFactor)
    {
    return (psb != NULL) &&
           SbsWindowZoom (hWnd, psb->dColumns / dFactor,
                                psb->dRows    / dFactor, TRUE);
    }

// =================================================================
// URL CODEC
// =================================================================

BOOL WINAPI SbsUrlSpaceA (BYTE bData)
    {
    return (bData == ' ' ) || (bData == '\t') ||
           (bData == '\r') || (bData == '\n');
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsUrlSpaceW (WORD wData)
    {
    return SbsUrlSpaceA (UNICODE_ANSI (wData));
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsUrlEscapeA (BYTE bData)
    {
    DWORD i;
    BOOL  fEscape = TRUE;

    if ((bData >= SBS_URL_ESCAPE_FIRST) &&
        (bData <= SBS_URL_ESCAPE_LAST))
        {
        i = 0;
        while (gabUrlEscape [i] && (gabUrlEscape [i] != bData)) i++;

        if (!gabUrlEscape [i]) fEscape = FALSE;
        }
    return fEscape;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsUrlEscapeW (WORD wData)
    {
    return SbsUrlEscapeA (UNICODE_ANSI (wData));
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsUrlCountA (PBYTE pbData,
                           DWORD dData,
                           BOOL  fEncode)
    {
    DWORD i;
    DWORD dData1 = SbsTextDataA (pbData, dData);
    DWORD dCount = 0;

    if (pbData != NULL)
        {
        if (fEncode)
            {
            for (i = 0; i < dData1; i++)
                {
                dCount += (SbsUrlEscapeA (pbData [i]) ? 3 : 1);
                }
            }
        else
            {
            for (i = 0; i < dData1; i++)
                {
                if (!SbsUrlSpaceA (pbData [i]))
                    {
                    if ((pbData [i] == '%') && (i+2 < dData1) &&
                        SbsHexTestA (pbData [i+1]) &&
                        SbsHexTestA (pbData [i+2]))
                        {
                        i += 2;
                        }
                    dCount++;
                    }
                }
            }
        }
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsUrlCountW (PWORD pwData,
                           DWORD dData,
                           BOOL  fEncode)
    {
    DWORD i;
    DWORD dData1 = SbsTextDataW (pwData, dData);
    DWORD dCount = 0;

    if (pwData != NULL)
        {
        if (fEncode)
            {
            for (i = 0; i < dData1; i++)
                {
                dCount += (SbsUrlEscapeW (pwData [i]) ? 3 : 1);
                }
            }
        else
            {
            for (i = 0; i < dData1; i++)
                {
                if (!SbsUrlSpaceW (pwData [i]))
                    {
                    if ((pwData [i] == '%') && (i+2 < dData1) &&
                        SbsHexTestW (pwData [i+1]) &&
                        SbsHexTestW (pwData [i+2]))
                        {
                        i += 2;
                        }
                    dCount++;
                    }
                }
            }
        }
    return dCount;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsUrlEncodeA (PBYTE  pbData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pbData != NULL)
        {
        dData = SbsTextDataA (pbData, dData);
        n     = SbsUrlCountA (pbData, dData, TRUE);

        if ((pbData1 = _tnewA (n+1)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                bData = pbData [i];

                if (SbsUrlEscapeA (bData))
                    {
                    pbData1 [n++] = '%';
                    pbData1 [n++] = gabHex [bData >>  4];
                    pbData1 [n++] = gabHex [bData & 0xF];
                    }
                else
                    {
                    pbData1 [n++] = (bData != ' ' ? bData : '+');
                    }
                }
            pbData1 [dData1 = n] = 0;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsUrlEncodeW (PWORD  pwData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PWORD pwData1 = NULL;

    if (pwData != NULL)
        {
        dData = SbsTextDataW (pwData, dData);
        n     = SbsUrlCountW (pwData, dData, TRUE);

        if ((pwData1 = _tnewW (n+1)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                bData = UNICODE_ANSI (pwData [i]);

                if (SbsUrlEscapeA (bData))
                    {
                    pwData1 [n++] = '%';
                    pwData1 [n++] = gawHex [bData >>  4];
                    pwData1 [n++] = gawHex [bData & 0xF];
                    }
                else
                    {
                    pwData1 [n++] = (bData != ' ' ? bData : '+');
                    }
                }
            pwData1 [dData1 = n] = 0;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pwData1;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsUrlDecodeA (PBYTE  pbData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pbData != NULL)
        {
        dData = SbsTextDataA (pbData, dData);
        n     = SbsUrlCountA (pbData, dData, FALSE);

        if ((pbData1 = _tnewA (n+1)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                bData = pbData [i];

                if (!SbsUrlSpaceA (bData))
                    {
                    if ((bData == '%') && (i+2 < dData) &&
                        SbsHexTestA (pbData [i+1]) &&
                        SbsHexTestA (pbData [i+2]))
                        {
                        pbData1 [n++] = SbsHexByteA (pbData [i+1],
                                                     pbData [i+2]);
                        i += 2;
                        }
                    else
                        {
                        pbData1 [n++] = (bData != '+' ? bData
                                                      : ' ');
                        }
                    }
                }
            pbData1 [dData1 = n] = 0;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsUrlDecodeW (PWORD  pwData,
                            PDWORD pdData)
    {
    WORD  wData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PWORD pwData1 = NULL;

    if (pwData != NULL)
        {
        dData = SbsTextDataW (pwData, dData);
        n     = SbsUrlCountW (pwData, dData, FALSE);

        if ((pwData1 = _tnewW (n+1)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                wData = pwData [i];

                if (!SbsUrlSpaceW (wData))
                    {
                    if ((wData == '%') && (i+2 < dData) &&
                        SbsHexTestW (pwData [i+1]) &&
                        SbsHexTestW (pwData [i+2]))
                        {
                        pwData1 [n++] = SbsHexByteW (pwData [i+1],
                                                     pwData [i+2]);
                        i += 2;
                        }
                    else
                        {
                        pwData1 [n++] = (wData != '+' ? wData
                                                      : ' ');
                        }
                    }
                }
            pwData1 [dData1 = n] = 0;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pwData1;
    }

// =================================================================
// URL CACHE MANAGEMENT
// =================================================================

PSBS_URL WINAPI SbsUrlTrim (PSBS_URL psu)
    {
    LONG     lBiasA, lBiasW;
    PSBS_URL psu1 = NULL;

    if (psu != NULL)
        {
        if ((psu1 = _mnew (SBS_URL__ (psu->dBuffer))) != NULL)
            {
            _mcopy (psu1, psu, SBS_URL__ (psu->dBuffer));

            lBiasA = (PBYTE) psu1 - (PBYTE) psu;
            lBiasW = (PWORD) psu1 - (PWORD) psu;

            if (psu1->fUnicode)
                {
                psu1->InfoW.lpHeaderInfo      += lBiasA;
                psu1->InfoW.lpszSourceUrlName += lBiasW;
                psu1->InfoW.lpszLocalFileName += lBiasW;
                psu1->InfoW.lpszFileExtension += lBiasW;

                psu1->pwName = psu1->InfoW.lpszSourceUrlName;
                psu1->pwPath = psu1->InfoW.lpszLocalFileName;
                psu1->pwFile = _ffileW (psu1->pwPath);
                }
            else
                {
                psu1->InfoA.lpHeaderInfo      += lBiasA;
                psu1->InfoA.lpszSourceUrlName += lBiasA;
                psu1->InfoA.lpszLocalFileName += lBiasA;
                psu1->InfoA.lpszFileExtension += lBiasA;

                psu1->pbName = psu1->InfoA.lpszSourceUrlName;
                psu1->pbPath = psu1->InfoA.lpszLocalFileName;
                psu1->pbFile = _ffileA (psu1->pbPath);
                }
            _mfree (psu);
            }
        else
            {
            psu1 = psu;
            }
        }
    return psu1;
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsUrlFirstA (PBYTE pbPattern)
    {
    PSBS_URL psu = NULL;

    if ((psu = _mnew (SBS_URL_)) != NULL)
        {
        psu->fUnicode = FALSE;
        psu->dBuffer  = SBS_URL_BUFFER;

        if ((psu->hInfo = FindFirstUrlCacheEntryA (pbPattern,
                                                   &psu->InfoA,
                                                   &psu->dBuffer))
            != NULL)
            {
            psu->ftExpire = psu->InfoA.ExpireTime;
            psu->ftWrite  = psu->InfoA.LastModifiedTime;
            psu->pbName   = psu->InfoA.lpszSourceUrlName;
            psu->pbPath   = psu->InfoA.lpszLocalFileName;
            psu->pbFile   = _ffileA (psu->pbPath);
            }
        else
            {
            psu = _mfree (psu);
            }
        }
    return SbsUrlTrim (psu);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsUrlFirstW (PWORD pwPattern)
    {
    PSBS_URL psu = NULL;

    if ((psu = _mnew (SBS_URL_)) != NULL)
        {
        psu->fUnicode = TRUE;
        psu->dBuffer  = SBS_URL_BUFFER;

        if ((psu->hInfo = FindFirstUrlCacheEntryW (pwPattern,
                                                   &psu->InfoW,
                                                   &psu->dBuffer))
            != NULL)
            {
            psu->ftExpire = psu->InfoW.ExpireTime;
            psu->ftWrite  = psu->InfoW.LastModifiedTime;
            psu->pwName   = psu->InfoW.lpszSourceUrlName;
            psu->pwPath   = psu->InfoW.lpszLocalFileName;
            psu->pwFile   = _ffileW (psu->pwPath);
            }
        else
            {
            psu = _mfree (psu);
            }
        }
    return SbsUrlTrim (psu);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsUrlNext (PSBS_URL psu)
    {
    PSBS_URL psu1 = NULL;

    if (psu != NULL) 
        {
        if ((psu1 = _mnew (SBS_URL_)) != NULL)
            {
            psu1->fUnicode = psu->fUnicode;
            psu1->dBuffer  = SBS_URL_BUFFER;
            psu1->hInfo    = psu->hInfo;

            if (psu1->fUnicode)
                {
                if (FindNextUrlCacheEntryW (psu1->hInfo,
                                            &psu1->InfoW,
                                            &psu1->dBuffer))
                    {
                    psu1->ftExpire = psu1->InfoW.ExpireTime;
                    psu1->ftWrite  = psu1->InfoW.LastModifiedTime;
                    psu1->pwName   = psu1->InfoW.lpszSourceUrlName;
                    psu1->pwPath   = psu1->InfoW.lpszLocalFileName;
                    psu1->pwFile   = _ffileW (psu1->pwPath);
                    }
                else
                    {
                    psu1 = SbsUrlExit (psu1, TRUE);
                    }
                }
            else
                {
                if (FindNextUrlCacheEntryA (psu1->hInfo,
                                            &psu1->InfoA,
                                            &psu1->dBuffer))
                    {
                    psu1->ftExpire = psu1->InfoA.ExpireTime;
                    psu1->ftWrite  = psu1->InfoA.LastModifiedTime;
                    psu1->pbName   = psu1->InfoA.lpszSourceUrlName;
                    psu1->pbPath   = psu1->InfoA.lpszLocalFileName;
                    psu1->pbFile   = _ffileA (psu1->pbPath);
                    }
                else
                    {
                    psu1 = SbsUrlExit (psu1, TRUE);
                    }
                }
            }
        _mfree (psu);
        }
    return SbsUrlTrim (psu1);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsUrlExit (PSBS_URL psu,
                            BOOL     fDestroy)
    {
    if (psu != NULL)
        {
        if (psu->hInfo != NULL)
            {
            FindCloseUrlCache (psu->hInfo);
            psu->hInfo = NULL;
            }
        if (fDestroy)
            {
            _mfree (psu);
            }
        }
    return (fDestroy ? NULL : psu);
    }

// =================================================================
// COOKIE MANAGEMENT
// =================================================================

PSBS_URL WINAPI SbsCookieFindA (PBYTE pbDomain)
    {
    PBYTE    pbName;
    PSBS_URL psu = NULL;

    if ((pbName = SbsCookieNameA (pbDomain)) != NULL)
        {
        for (psu = SbsCookieFirstA (); psu != NULL;
             psu = SbsCookieNext (psu))
            {
            if (!_tcompA (psu->pbName, pbName))
                {
                SbsCookieExit (psu, FALSE);
                break;
                }
            }
        _mfree (pbName);
        }
    return psu;
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsCookieFindW (PWORD pwDomain)
    {
    PWORD    pwName;
    PSBS_URL psu = NULL;

    if ((pwName = SbsCookieNameW (pwDomain)) != NULL)
        {
        for (psu = SbsCookieFirstW (); psu != NULL;
             psu = SbsCookieNext (psu))
            {
            if (!_tcompW (psu->pwName, pwName))
                {
                SbsCookieExit (psu, FALSE);
                break;
                }
            }
        _mfree (pwName);
        }
    return psu;
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsCookieFirstA (VOID)
    {
    return SbsUrlFirstA (gabCookieType);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsCookieFirstW (VOID)
    {
    return SbsUrlFirstW (gawCookieType);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsCookieNext (PSBS_URL psu)
    {
    return SbsUrlNext (psu);
    }

// -----------------------------------------------------------------

PSBS_URL WINAPI SbsCookieExit (PSBS_URL psu,
                               BOOL     fDestroy)
    {
    return SbsUrlExit (psu, fDestroy);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieDeleteA (PBYTE pbDomain)
    {
    PSBS_URL psu;
    BOOL     fOk = TRUE;

    while ((psu = SbsCookieFindA (pbDomain)) != NULL)
        {
        if (!DeleteUrlCacheEntryA (psu->pbName)) fOk = FALSE;
        DeleteFileA (psu->pbPath);

        _mfree (psu);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieDeleteW (PWORD pwDomain)
    {
    PSBS_URL psu;
    BOOL     fOk = TRUE;

    while ((psu = SbsCookieFindW (pwDomain)) != NULL)
        {
        if (!DeleteUrlCacheEntryW (psu->pwName)) fOk = FALSE;
        DeleteFileW (psu->pwPath);

        _mfree (psu);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieCreateA (PBYTE pbDomain)
    {
    PBYTE pbName;
    PBYTE pbPath = NULL;

    SbsCookieDeleteA (pbDomain);

    if ((pbName = SbsCookieNameA (pbDomain)) != NULL)
        {
        if ((pbPath = _tnewA (MAX_PATH)) != NULL)
            {
            if (CreateUrlCacheEntryA (pbName, 0, gabCookieExt,
                                      pbPath, 0))
                {
                DeleteFileA (pbPath);
                }
            else
                {
                pbPath = _mfree (pbPath);
                }
            }
        _mfree (pbName);
        }
    return pbPath;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsCookieCreateW (PWORD pwDomain)
    {
    PWORD pwName;
    PWORD pwPath = NULL;

    SbsCookieDeleteW (pwDomain);

    if ((pwName = SbsCookieNameW (pwDomain)) != NULL)
        {
        if ((pwPath = _tnewW (MAX_PATH)) != NULL)
            {
            if (CreateUrlCacheEntryW (pwName, 0, gawCookieExt,
                                      pwPath, 0))
                {
                DeleteFileW (pwPath);
                }
            else
                {
                pwPath = _mfree (pwPath);
                }
            }
        _mfree (pwName);
        }
    return pwPath;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieCommitA (PBYTE    pbDomain,
                              PBYTE    pbPath,
                              FILETIME ftExpire,
                              FILETIME ftWrite)
    {
    PBYTE pbName;
    BOOL  fOk = FALSE;

    if ((pbPath != NULL) &&
        ((pbName = SbsCookieNameA (pbDomain)) != NULL))
        {
        fOk = CommitUrlCacheEntryA (pbName, pbPath,
                                    ftExpire, ftWrite,
                                    COOKIE_CACHE_ENTRY,
                                    NULL, 0, NULL, NULL);
        _mfree (pbName);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieCommitW (PWORD    pwDomain,
                              PWORD    pwPath,
                              FILETIME ftExpire,
                              FILETIME ftWrite)
    {
    PWORD pwName;
    BOOL  fOk = FALSE;

    if ((pwPath != NULL) &&
        ((pwName = SbsCookieNameW (pwDomain)) != NULL))
        {
        fOk = CommitUrlCacheEntryW (pwName, pwPath,
                                    ftExpire, ftWrite,
                                    COOKIE_CACHE_ENTRY,
                                    NULL, 0, NULL, NULL);
        _mfree (pwName);
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCookieTestA (PBYTE pbData,
                             DWORD dData)
    {
    DWORD i, n;
    DWORD dValues = 0;

    if (pbData != NULL)
        {
        for (i = 0; i < dData; i += n, dValues++)
            {
            if (!(n = SbsCookieParseA (pbData+i, dData-i, NULL)))
                {
                dValues = 0;
                break;
                }
            }
        }
    return dValues;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCookieTestW (PWORD pwData,
                             DWORD dData)
    {
    DWORD i, n;
    DWORD dValues = 0;

    if (pwData != NULL)
        {
        for (i = 0; i < dData; i += n, dValues++)
            {
            if (!(n = SbsCookieParseW (pwData+i, dData-i, NULL)))
                {
                dValues = 0;
                break;
                }
            }
        }
    return dValues;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCookieParseA (PBYTE             pbData,
                              DWORD             dData,
                              PSBS_COOKIE_VALUE pscv)
    {
    BOOL  fNumeric;
    BYTE  b;
    DWORD i, j;
    DWORD n = 0;

    if (pbData != NULL)
        {
        for (i = 0; i < 9; i++)
            {
            fNumeric = FALSE;
            j        = n;

            while ((n < dData) && (pbData [n] != '\n'))
                {
                if (j == n) fNumeric = TRUE;
                b = pbData [n++];
                if ((b < '0') || (b > '9')) fNumeric = FALSE;
                }
            if ((n == dData)                          ||
                ((i >= 3) && (i <= 7) && (!fNumeric)) ||
                ((i == 8) && ((n-j != 1) || (b != '*')))) break;
            n++;
            }
        if (i < 9) n = 0;
        }
    if (n && (pscv != NULL))
        {
        j = 0;

        for (i = j; pbData [j] != '\n'; j++);
        pscv->pbKey    = pbData+i;
        pscv->dKey     = j-i;
        pbData [j++]   = 0;

        for (i = j; pbData [j] != '\n'; j++);
        pscv->pbValue  = pbData+i;
        pscv->dValue   = j-i;
        pbData [j++]   = 0;

        for (i = j; pbData [j] != '\n'; j++);
        pscv->pbDomain = pbData+i;
        pscv->dDomain  = j-i;
        pbData [j++]   = 0;

        for (i = j; pbData [j] != '\n'; j++);
        SbsDecimalReadA (pbData+i, &pscv->fSecure);
        pbData [j++] = 0;

        for (i = j; pbData [j] != '\n'; j++);
        SbsDecimalReadA (pbData+i, &pscv->ftExpire.dwLowDateTime);
        pbData [j++] = 0;

        for (i = j; pbData [j] != '\n'; j++);
        SbsDecimalReadA (pbData+i, &pscv->ftExpire.dwHighDateTime);
        pbData [j++] = 0;

        for (i = j; pbData [j] != '\n'; j++);
        SbsDecimalReadA (pbData+i, &pscv->ftWrite.dwLowDateTime);
        pbData [j++] = 0;

        for (i = j; pbData [j] != '\n'; j++);
        SbsDecimalReadA (pbData+i, &pscv->ftWrite.dwHighDateTime);
        pbData [j++] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCookieParseW (PWORD             pwData,
                              DWORD             dData,
                              PSBS_COOKIE_VALUE pscv)
    {
    BOOL  fNumeric;
    WORD  w;
    DWORD i, j;
    DWORD n = 0;

    if (pwData != NULL)
        {
        for (i = 0; i < 9; i++)
            {
            fNumeric = FALSE;
            j        = n;

            while ((n < dData) && (pwData [n] != '\n'))
                {
                if (j == n) fNumeric = TRUE;
                w = pwData [n++];
                if ((w < '0') || (w > '9')) fNumeric = FALSE;
                }
            if ((n == dData)                          ||
                ((i >= 3) && (i <= 7) && (!fNumeric)) ||
                ((i == 8) && ((n-j != 1) || (w != '*')))) break;
            n++;
            }
        if (i < 9) n = 0;
        }
    if (n && (pscv != NULL))
        {
        j = 0;

        for (i = j; pwData [j] != '\n'; j++);
        pscv->pwKey    = pwData+i;
        pscv->dKey     = j-i;
        pwData [j++]   = 0;

        for (i = j; pwData [j] != '\n'; j++);
        pscv->pwValue  = pwData+i;
        pscv->dValue   = j-i;
        pwData [j++]   = 0;

        for (i = j; pwData [j] != '\n'; j++);
        pscv->pwDomain = pwData+i;
        pscv->dDomain  = j-i;
        pwData [j++]   = 0;

        for (i = j; pwData [j] != '\n'; j++);
        SbsDecimalReadW (pwData+i, &pscv->fSecure);
        pwData [j++] = 0;

        for (i = j; pwData [j] != '\n'; j++);
        SbsDecimalReadW (pwData+i, &pscv->ftExpire.dwLowDateTime);
        pwData [j++] = 0;

        for (i = j; pwData [j] != '\n'; j++);
        SbsDecimalReadW (pwData+i, &pscv->ftExpire.dwHighDateTime);
        pwData [j++] = 0;

        for (i = j; pwData [j] != '\n'; j++);
        SbsDecimalReadW (pwData+i, &pscv->ftWrite.dwLowDateTime);
        pwData [j++] = 0;

        for (i = j; pwData [j] != '\n'; j++);
        SbsDecimalReadW (pwData+i, &pscv->ftWrite.dwHighDateTime);
        pwData [j++] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieCompileA (PBYTE     pbData,
                                      DWORD     dData,
                                      PFILETIME pftCreate,
                                      PFILETIME pftAccess,
                                      PFILETIME pftWrite)
    {
    DWORD       dValues, i, n;
    SYSTEMTIME  st;
    FILETIME    ft;
    PSBS_COOKIE psc = NULL;

    GetSystemTime (&st);

    if (SystemTimeToFileTime (&st, &ft)            &&
        (dValues = SbsCookieTestA (pbData, dData)) &&
        ((psc = _mnew (SBS_COOKIE__ (dValues))) != NULL))
        {
        if ((pftCreate != NULL) || (pftWrite != NULL))
            {
            psc->ftCreate = SAFEVAL (pftCreate, *pftWrite);
            psc->ftWrite  = SAFEVAL (pftWrite,  *pftCreate);
            }
        else
            {
            psc->ftCreate =
            psc->ftWrite  = SAFEVAL (pftAccess, ft);
            }
        psc->ftAccess = SAFEVAL (pftAccess, *pftWrite);
        psc->pbData   = pbData;
        psc->dData    = dData;
        psc->dValues  = dValues;

        for (i = n = 0; i < psc->dValues; i++)
            {
            n += SbsCookieParseA (psc->pbData  + n,
                                  psc->dData   - n,
                                  psc->aValues + i);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieCompileW (PWORD     pwData,
                                      DWORD     dData,
                                      PFILETIME pftCreate,
                                      PFILETIME pftAccess,
                                      PFILETIME pftWrite)
    {
    DWORD       dValues, i, n;
    SYSTEMTIME  st;
    FILETIME    ft;
    PSBS_COOKIE psc = NULL;

    GetSystemTime (&st);

    if (SystemTimeToFileTime (&st, &ft)            &&
        (dValues = SbsCookieTestW (pwData, dData)) &&
        ((psc = _mnew (SBS_COOKIE__ (dValues))) != NULL))
        {
        if ((pftCreate != NULL) || (pftWrite != NULL))
            {
            psc->ftCreate = SAFEVAL (pftCreate, *pftWrite);
            psc->ftWrite  = SAFEVAL (pftWrite,  *pftCreate);
            }
        else
            {
            psc->ftCreate =
            psc->ftWrite  = SAFEVAL (pftAccess, ft);
            }
        psc->ftAccess = SAFEVAL (pftAccess, *pftWrite);
        psc->pwData   = pwData;
        psc->dData    = dData;
        psc->dValues  = dValues;

        for (i = n = 0; i < psc->dValues; i++)
            {
            n += SbsCookieParseW (psc->pwData  + n,
                                  psc->dData   - n,
                                  psc->aValues + i);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieRelease (PSBS_COOKIE psc)
    {
    if (psc != NULL)
        {
        _mfree (psc->pData);
        _mfree (psc);
        }
    return NULL;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieLoadA (PBYTE pbPath)
    {
    PBYTE       pbData;
    DWORD       dData;
    FILETIME    ftCreate, ftAccess, ftWrite;
    PSBS_COOKIE psc = NULL;

    if (((pbData = _ftextxA (pbPath, &dData,
                             &ftCreate, &ftAccess, &ftWrite))
         != NULL)
        &&
        ((psc = SbsCookieCompileA (pbData, dData,
                                   &ftCreate, &ftAccess, &ftWrite))
         == NULL))
        {
        _mfree (pbData);
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieLoadW (PWORD pwPath)
    {
    PWORD       pwData;
    DWORD       dData;
    FILETIME    ftCreate, ftAccess, ftWrite;
    PSBS_COOKIE psc = NULL;

    if (((pwData = _ftextxW (pwPath, &dData,
                             &ftCreate, &ftAccess, &ftWrite))
         != NULL)
        &&
        ((psc = SbsCookieCompileW (pwData, dData,
                                   &ftCreate, &ftAccess, &ftWrite))
         == NULL))
        {
        _mfree (pwData);
        }
    return psc;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieSaveA (PSBS_COOKIE psc,
                            PBYTE       pbPath)
    {
    HANDLE hf;
    PBYTE  pbData;
    DWORD  dData, i;
    BOOL   fOk = FALSE;

    if ((psc != NULL)
        &&
        ((hf = _fopenA (pbPath, SBS_FILE_NEW))
         != INVALID_HANDLE_VALUE))
        {
        for (i = 0, fOk = TRUE; fOk && (i < psc->dValues); i++)
            {
            if ((pbData = SbsCookieDataA (psc->aValues + i,
                                          &dData))
                != NULL)
                {
                fOk = _fwrite (hf, pbData, dData);
                _mfree (pbData);
                }
            else
                {
                fOk = FALSE;
                }
            }
        fOk = fOk && SetEndOfFile (hf);
        _fclose (hf);

        if (!fOk) DeleteFileA (pbPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieSaveW (PSBS_COOKIE psc,
                            PWORD       pwPath)
    {
    HANDLE hf;
    PBYTE  pbData;
    DWORD  dData, i;
    BOOL   fOk = FALSE;

    if ((psc != NULL)
        &&
        ((hf = _fopenW (pwPath, SBS_FILE_NEW))
         != INVALID_HANDLE_VALUE))
        {
        for (i = 0, fOk = TRUE; fOk && (i < psc->dValues); i++)
            {
            if ((pbData = SbsCookieDataW2A (psc->aValues + i,
                                            &dData))
                != NULL)
                {
                fOk = _fwrite (hf, pbData, dData);
                _mfree (pbData);
                }
            else
                {
                fOk = FALSE;
                }
            }
        fOk = fOk && SetEndOfFile (hf);
        _fclose (hf);

        if (!fOk) DeleteFileW (pwPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieAssignA (PBYTE     pbKey,
                                     PBYTE     pbValue,
                                     PBYTE     pbDomain,
                                     BOOL      fSecure,
                                     PFILETIME pftExpire,
                                     PFILETIME pftWrite)
    {
    PSBS_URL    psu;
    PSBS_COOKIE psc = NULL;

    if (pbDomain != NULL)
        {
        if ((psu = SbsCookieFindA (pbDomain)) != NULL)
            {
            psc = SbsCookieLoadA (psu->pbPath);
            _mfree (psu);
            }
        if ((psc != NULL) || (psu == NULL))
            {
            psc = SbsCookieUpdateA (psc, pbKey, pbValue, pbDomain,
                                    fSecure, pftExpire, pftWrite);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieAssignW (PWORD     pwKey,
                                     PWORD     pwValue,
                                     PWORD     pwDomain,
                                     BOOL      fSecure,
                                     PFILETIME pftExpire,
                                     PFILETIME pftWrite)
    {
    PSBS_URL    psu;
    PSBS_COOKIE psc = NULL;

    if (pwDomain != NULL)
        {
        if ((psu = SbsCookieFindW (pwDomain)) != NULL)
            {
            psc = SbsCookieLoadW (psu->pwPath);
            _mfree (psu);
            }
        if ((psc != NULL) || (psu == NULL))
            {
            psc = SbsCookieUpdateW (psc, pwKey, pwValue, pwDomain,
                                    fSecure, pftExpire, pftWrite);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieUpdateA (PSBS_COOKIE psc,
                                     PBYTE       pbKey,
                                     PBYTE       pbValue,
                                     PBYTE       pbDomain,
                                     BOOL        fSecure,
                                     PFILETIME   pftExpire,
                                     PFILETIME   pftWrite)
    {
    DWORD             i;
    BOOL              fFound;
    PSBS_COOKIE_VALUE pscv;
    PSBS_COOKIE       psc1 = NULL;

    if (psc == NULL)
        {
        psc1 = SbsCookieNewA (pbKey, pbValue, pbDomain,
                              fSecure, pftExpire);
        }
    else
        {
        pscv = psc->aValues;

        if (pbKey != NULL)
            {
            fFound = FALSE;

            for (i = 0; i < psc->dValues; i++)
                {
                if (_tcompA (psc->aValues [i].pbKey, pbKey))
                    {
                    if (pscv != psc->aValues + i)
                        {
                        *pscv = psc->aValues [i];
                        }
                    pscv++;
                    }
                else
                    {
                    if (!fFound)
                        {
                        fFound = TRUE;

                        if (pbValue != NULL)
                            {
                            pscv->pbValue  = pbValue;
                            pscv->dValue   = _tsizeA (pbValue);
                            }
                        if (pbDomain != NULL)
                            {
                            pscv->pbDomain = pbDomain;
                            pscv->dDomain  = _tsizeA (pbDomain);
                            }
                        if (pftExpire != NULL)
                            {
                            pscv->ftExpire = *pftExpire;
                            }
                        if (pftWrite != NULL)
                            {
                            pscv->ftWrite  = *pftWrite;
                            }
                        pscv->fSecure = fSecure;
                        pscv++;
                        }
                    }
                }
            }
        psc->dValues = pscv - psc->aValues;

        psc1 = (fFound
                ? SbsCookieRebuildA (psc)
                : SbsCookieAppendA  (psc, pbKey, pbValue, pbDomain,
                                     fSecure, pftExpire));
        }
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieUpdateW (PSBS_COOKIE psc,
                                     PWORD       pwKey,
                                     PWORD       pwValue,
                                     PWORD       pwDomain,
                                     BOOL        fSecure,
                                     PFILETIME   pftExpire,
                                     PFILETIME   pftWrite)
    {
    DWORD             i;
    BOOL              fFound;
    PSBS_COOKIE_VALUE pscv;
    PSBS_COOKIE       psc1 = NULL;

    if (psc == NULL)
        {
        psc1 = SbsCookieNewW (pwKey, pwValue, pwDomain,
                              fSecure, pftExpire);
        }
    else
        {
        pscv = psc->aValues;

        if (pwKey != NULL)
            {
            fFound = FALSE;

            for (i = 0; i < psc->dValues; i++)
                {
                if (_tcompW (psc->aValues [i].pwKey, pwKey))
                    {
                    if (pscv != psc->aValues + i)
                        {
                        *pscv = psc->aValues [i];
                        }
                    pscv++;
                    }
                else
                    {
                    if (!fFound)
                        {
                        fFound = TRUE;

                        if (pwValue != NULL)
                            {
                            pscv->pwValue  = pwValue;
                            pscv->dValue   = _tsizeW (pwValue);
                            }
                        if (pwDomain != NULL)
                            {
                            pscv->pwDomain = pwDomain;
                            pscv->dDomain  = _tsizeW (pwDomain);
                            }
                        if (pftExpire != NULL)
                            {
                            pscv->ftExpire = *pftExpire;
                            }
                        if (pftWrite != NULL)
                            {
                            pscv->ftWrite  = *pftWrite;
                            }
                        pscv->fSecure = fSecure;
                        pscv++;
                        }
                    }
                }
            }
        psc->dValues = pscv - psc->aValues;

        psc1 = (fFound
                ? SbsCookieRebuildW (psc)
                : SbsCookieAppendW  (psc, pwKey, pwValue, pwDomain,
                                     fSecure, pftExpire));
        }
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieRemoveA (PSBS_COOKIE psc,
                                     PBYTE       pbKey)
    {
    DWORD       i, j;
    PSBS_COOKIE psc1 = psc;

    if ((psc != NULL) && (pbKey != NULL))
        {
        for (i = j = 0; i < psc->dValues; i++)
            {
            if (_tcompA (psc->aValues [i].pbKey, pbKey))
                {
                if (j != i) psc->aValues [j] = psc->aValues [i];
                j++;
                }
            }
        if (psc->dValues != j)
            {
            psc->dValues = j;
            psc1 = SbsCookieRebuildA (psc);
            }
        }
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieRemoveW (PSBS_COOKIE psc,
                                     PWORD       pwKey)
    {
    DWORD       i, j;
    PSBS_COOKIE psc1 = psc;

    if ((psc != NULL) && (pwKey != NULL))
        {
        for (i = j = 0; i < psc->dValues; i++)
            {
            if (_tcompW (psc->aValues [i].pwKey, pwKey))
                {
                if (j != i) psc->aValues [j] = psc->aValues [i];
                j++;
                }
            }
        psc->dValues = j;
        psc1 = SbsCookieRebuildW (psc);
        }
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieNewA (PBYTE     pbKey,
                                  PBYTE     pbValue,
                                  PBYTE     pbDomain,
                                  BOOL      fSecure,
                                  PFILETIME pftExpire)
    {
    SYSTEMTIME  stExpire, stWrite;
    FILETIME    ftExpire, ftWrite;
    PBYTE       pbData;
    DWORD       dData;
    PSBS_COOKIE psc = NULL;

    GetSystemTime (&stWrite);

    stExpire = stWrite;
    stExpire.wYear++;

    if (SystemTimeToFileTime (&stExpire, &ftExpire) &&
        SystemTimeToFileTime (&stWrite,  &ftWrite ))
        {
        if (pftExpire != NULL) ftExpire = *pftExpire;

        if (((pbData = SbsCookieDataExA (pbKey, pbValue, pbDomain,
                                         fSecure, ftExpire, ftWrite,
                                         &dData))
             != NULL)
            &&
            ((psc = SbsCookieCompileA (pbData, dData,
                                       NULL, NULL, &ftWrite))
             == NULL))
            {
            _mfree (pbData);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieNewW (PWORD     pwKey,
                                  PWORD     pwValue,
                                  PWORD     pwDomain,
                                  BOOL      fSecure,
                                  PFILETIME pftExpire)
    {
    SYSTEMTIME  stExpire, stWrite;
    FILETIME    ftExpire, ftWrite;
    PWORD       pwData;
    DWORD       dData;
    PSBS_COOKIE psc = NULL;

    GetSystemTime (&stWrite);

    stExpire = stWrite;
    stExpire.wYear++;

    if (SystemTimeToFileTime (&stExpire, &ftExpire) &&
        SystemTimeToFileTime (&stWrite,  &ftWrite ))
        {
        if (pftExpire != NULL) ftExpire = *pftExpire;

        if (((pwData = SbsCookieDataExW (pwKey, pwValue, pwDomain,
                                         fSecure, ftExpire, ftWrite,
                                         &dData))
             != NULL)
            &&
            ((psc = SbsCookieCompileW (pwData, dData,
                                       NULL, NULL, &ftWrite))
             == NULL))
            {
            _mfree (pwData);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieAppendA (PSBS_COOKIE psc,
                                     PBYTE       pbKey,
                                     PBYTE       pbValue,
                                     PBYTE       pbDomain,
                                     BOOL        fSecure,
                                     PFILETIME   pftExpire)
    {
    PSBS_COOKIE psc1;
    PSBS_COOKIE psc2 = NULL;

    if ((psc1 = SbsCookieNewA (pbKey, pbValue, pbDomain,
                               fSecure, pftExpire))
        != NULL)
        {
        psc2 = SbsCookieMergeA (2, psc, psc1);
        SbsCookieRelease (psc1);
        }
    SbsCookieRelease (psc);
    return psc2;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieAppendW (PSBS_COOKIE psc,
                                     PWORD       pwKey,
                                     PWORD       pwValue,
                                     PWORD       pwDomain,
                                     BOOL        fSecure,
                                     PFILETIME   pftExpire)
    {
    PSBS_COOKIE psc1;
    PSBS_COOKIE psc2 = NULL;

    if ((psc1 = SbsCookieNewW (pwKey, pwValue, pwDomain,
                               fSecure, pftExpire))
        != NULL)
        {
        psc2 = SbsCookieMergeW (2, psc, psc1);
        SbsCookieRelease (psc1);
        }
    SbsCookieRelease (psc);
    return psc2;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieRebuildA (PSBS_COOKIE psc)
    {
    PSBS_COOKIE psc1 = SbsCookieMergeA (1, psc);

    SbsCookieRelease (psc);
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieRebuildW (PSBS_COOKIE psc)
    {
    PSBS_COOKIE psc1 = SbsCookieMergeW (1, psc);

    SbsCookieRelease (psc);
    return psc1;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieMergeA (DWORD dCookies,
                                    ...)
    {
    SYSTEMTIME   st;
    FILETIME     ft;
    DWORD        dData, i, j;
    PBYTE        pbData;
    PSBS_BUFFER  psb;
    PPSBS_COOKIE ppsc = (PPSBS_COOKIE) ((&dCookies)+1);
    PSBS_COOKIE  psc  = NULL;

    GetSystemTime (&st);

    if (dCookies && SystemTimeToFileTime (&st, &ft) &&
        ((psb = SbsTextCreateA (0)) != NULL))
        {
        for (i = 0; i < dCookies; i++)
            {
            if (ppsc [i] != NULL)
                {
                for (j = 0; j < ppsc [i]->dValues; j++)
                    {
                    if ((pbData = SbsCookieDataA
                                      (ppsc [i]->aValues + j,
                                       &dData))
                        != NULL)
                        {
                        psb = SbsTextWriteA (psb, pbData, dData);
                        _mfree (pbData);
                        }
                    else
                        {
                        psb = SbsTextDestroy (psb);
                        }
                    if (psb == NULL) break;
                    }
                }
            if (psb == NULL) break;
            }
        if ((pbData = SbsTextReleaseA (psb, &dData)) != NULL)
            {
            psc = SbsCookieCompileA (pbData, dData,
                                     NULL, NULL, &ft);
            _mfree (pbData);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

PSBS_COOKIE WINAPI SbsCookieMergeW (DWORD dCookies,
                                    ...)
    {
    SYSTEMTIME   st;
    FILETIME     ft;
    DWORD        dData, i, j;
    PWORD        pwData;
    PSBS_BUFFER  psb;
    PPSBS_COOKIE ppsc = (PPSBS_COOKIE) ((&dCookies)+1);
    PSBS_COOKIE  psc  = NULL;

    GetSystemTime (&st);

    if (dCookies && SystemTimeToFileTime (&st, &ft) &&
        ((psb = SbsTextCreateW (0)) != NULL))
        {
        for (i = 0; i < dCookies; i++)
            {
            if (ppsc [i] != NULL)
                {
                for (j = 0; j < ppsc [i]->dValues; j++)
                    {
                    if ((pwData = SbsCookieDataW
                                      (ppsc [i]->aValues + j,
                                       &dData))
                        != NULL)
                        {
                        psb = SbsTextWriteW (psb, pwData, dData);
                        _mfree (pwData);
                        }
                    else
                        {
                        psb = SbsTextDestroy (psb);
                        }
                    if (psb == NULL) break;
                    }
                }
            if (psb == NULL) break;
            }
        if ((pwData = SbsTextReleaseW (psb, &dData)) != NULL)
            {
            psc = SbsCookieCompileW (pwData, dData,
                                     NULL, NULL, &ft);
            _mfree (pwData);
            }
        }
    return psc;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieRegisterA (PSBS_COOKIE psc)
    {
    PBYTE    pbPath, pbDomain;
    FILETIME ftExpire, ftWrite;
    BOOL     fOk = FALSE;

    if ((psc != NULL) && psc->dValues)
        {
        pbDomain = psc->aValues [0].pbDomain;

        if ((pbPath = SbsCookieCreateA (pbDomain)) != NULL)
            {
            SbsCookieSaveA (psc, pbPath);
            SbsCookieTime  (psc, &ftExpire, &ftWrite);

            fOk = SbsCookieCommitA (pbDomain, pbPath,
                                    ftExpire, ftWrite);
            _mfree (pbPath);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieRegisterW (PSBS_COOKIE psc)
    {
    PWORD    pwPath, pwDomain;
    FILETIME ftExpire, ftWrite;
    BOOL     fOk = FALSE;

    if ((psc != NULL) && psc->dValues)
        {
        pwDomain = psc->aValues [0].pwDomain;

        if ((pwPath = SbsCookieCreateW (pwDomain)) != NULL)
            {
            SbsCookieSaveW (psc, pwPath);
            SbsCookieTime  (psc, &ftExpire, &ftWrite);

            fOk = SbsCookieCommitW (pwDomain, pwPath,
                                    ftExpire, ftWrite);
            _mfree (pwPath);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCookieTime (PSBS_COOKIE psc,
                           PFILETIME   pftExpire,
                           PFILETIME   pftWrite)
    {
    DWORD    i;
    FILETIME ftExpire = gftNull;
    FILETIME ftWrite  = gftNull;
    BOOL     fOk      = FALSE;

    if (psc != NULL)
        {
        for (i = 0; i < psc->dValues; i++)
            {
            if (_fnewer (psc->aValues [i].ftExpire, ftExpire))
                {
                ftExpire = psc->aValues [i].ftExpire;
                }
            if (_fnewer (psc->aValues [i].ftWrite,  ftWrite ))
                {
                ftWrite  = psc->aValues [i].ftWrite;
                }
            }
        fOk = TRUE;
        }
    if (pftExpire != NULL) *pftExpire = ftExpire;
    if (pftWrite  != NULL) *pftWrite  = ftWrite;
    return fOk;
    }

// -----------------------------------------------------------------

PSBS_COOKIE_VALUE WINAPI SbsCookieValueA (PSBS_COOKIE psc,
                                          PBYTE       pbKey)
    {
    DWORD             i;
    PSBS_COOKIE_VALUE pscv = NULL;

    if ((psc != NULL) && (pbKey != NULL))
        {
        for (i = 0; i < psc->dValues; i++)
            {
            if (!_tcompA (psc->aValues [i].pbKey, pbKey))
                {
                pscv = psc->aValues + i;
                break;
                }
            }
        }
    return pscv;
    }

// -----------------------------------------------------------------

PSBS_COOKIE_VALUE WINAPI SbsCookieValueW (PSBS_COOKIE psc,
                                          PWORD       pwKey)
    {
    DWORD             i;
    PSBS_COOKIE_VALUE pscv = NULL;

    if ((psc != NULL) && (pwKey != NULL))
        {
        for (i = 0; i < psc->dValues; i++)
            {
            if (!_tcompW (psc->aValues [i].pwKey, pwKey))
                {
                pscv = psc->aValues + i;
                break;
                }
            }
        }
    return pscv;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieDataA (PSBS_COOKIE_VALUE pscv,
                             PDWORD            pdData)
    {
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    if (pscv != NULL)
        {
        pbData = SbsCookieDataExA (pscv->pbKey,    pscv->pbValue,
                                   pscv->pbDomain, pscv->fSecure,
                                   pscv->ftExpire, pscv->ftWrite,
                                   &dData);
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsCookieDataW (PSBS_COOKIE_VALUE pscv,
                             PDWORD            pdData)
    {
    DWORD dData  = 0;
    PWORD pwData = NULL;

    if (pscv != NULL)
        {
        pwData = SbsCookieDataExW (pscv->pwKey,    pscv->pwValue,
                                   pscv->pwDomain, pscv->fSecure,
                                   pscv->ftExpire, pscv->ftWrite,
                                   &dData);
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieDataW2A (PSBS_COOKIE_VALUE pscv,
                               PDWORD            pdData)
    {
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    if (pscv != NULL)
        {
        pbData = SbsCookieDataExW2A (pscv->pwKey,    pscv->pwValue,
                                     pscv->pwDomain, pscv->fSecure,
                                     pscv->ftExpire, pscv->ftWrite,
                                     &dData);
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieDataExA (PBYTE    pbKey,
                               PBYTE    pbValue,
                               PBYTE    pbDomain,
                               BOOL     fSecure,
                               FILETIME ftExpire,
                               FILETIME ftWrite,
                               PDWORD   pdData)
    {
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    if ((pbKey != NULL) && (pbValue != NULL) && (pbDomain != NULL))
        {
        pbData = _bprintfA (&dData, gabCookieA,
                            pbKey, pbValue, pbDomain, fSecure,
                            ftExpire.dwLowDateTime,
                            ftExpire.dwHighDateTime,
                            ftWrite .dwLowDateTime,
                            ftWrite .dwHighDateTime);
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsCookieDataExW (PWORD    pwKey,
                               PWORD    pwValue,
                               PWORD    pwDomain,
                               BOOL     fSecure,
                               FILETIME ftExpire,
                               FILETIME ftWrite,
                               PDWORD   pdData)
    {
    DWORD dData  = 0;
    PWORD pwData = NULL;

    if ((pwKey != NULL) && (pwValue != NULL) && (pwDomain != NULL))
        {
        pwData = _bprintfW (&dData, gawCookie,
                            pwKey, pwValue, pwDomain, fSecure,
                            ftExpire.dwLowDateTime,
                            ftExpire.dwHighDateTime,
                            ftWrite .dwLowDateTime,
                            ftWrite .dwHighDateTime);
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieDataExW2A (PWORD    pwKey,
                                 PWORD    pwValue,
                                 PWORD    pwDomain,
                                 BOOL     fSecure,
                                 FILETIME ftExpire,
                                 FILETIME ftWrite,
                                 PDWORD   pdData)
    {
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    if ((pwKey != NULL) && (pwValue != NULL) && (pwDomain != NULL))
        {
        pbData = _bprintfA (&dData, gabCookieW,
                            pwKey, pwValue, pwDomain, fSecure,
                            ftExpire.dwLowDateTime,
                            ftExpire.dwHighDateTime,
                            ftWrite .dwLowDateTime,
                            ftWrite .dwHighDateTime);
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieUserA (VOID)
    {
    DWORD i, j;
    PBYTE pbPath = NULL;

    if ((pbPath = _tnewA (MAX_PATH)) != NULL)
        {
        if (CreateUrlCacheEntryA (gabCookieTemp, 0, gabCookieExt,
                                  pbPath, 0))
            {
            CommitUrlCacheEntryA (gabCookieTemp, pbPath,
                                  gftNull, gftNull,
                                  COOKIE_CACHE_ENTRY,
                                  NULL, 0, gabNull, NULL);

            DeleteUrlCacheEntryA (gabCookieTemp);

            for (i = 0, j = _fleafA (pbPath);
                 pbPath [j] != '@'; i++, j++)
                {
                pbPath [i] = pbPath [j];
                }
            pbPath [i] = 0;
            }
        else
            {
            pbPath = _mfree (pbPath);
            }
        }
    return pbPath;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsCookieUserW (VOID)
    {
    DWORD i, j;
    PWORD pwPath = NULL;

    if ((pwPath = _tnewW (MAX_PATH)) != NULL)
        {
        if (CreateUrlCacheEntryW (gawCookieTemp, 0, gawCookieExt,
                                  pwPath, 0))
            {
            CommitUrlCacheEntryW (gawCookieTemp, pwPath,
                                  gftNull, gftNull,
                                  COOKIE_CACHE_ENTRY,
                                  NULL, 0, gawNull, NULL);

            DeleteUrlCacheEntryW (gawCookieTemp);

            for (i = 0, j = _fleafW (pwPath);
                 pwPath [j] != '@'; i++, j++)
                {
                pwPath [i] = pwPath [j];
                }
            pwPath [i] = 0;
            }
        else
            {
            pwPath = _mfree (pwPath);
            }
        }
    return pwPath;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsCookieNameA (PBYTE pbDomain)
    {
    PBYTE pbUser;
    PBYTE pbName = NULL;

    if ((pbDomain != NULL) &&
        ((pbUser = SbsCookieUserA ()) != NULL))
        {
        pbName = _bprintfA (NULL, gabCookieName, pbUser, pbDomain);
        _mfree (pbUser);
        }
    return pbName;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsCookieNameW (PWORD pwDomain)
    {
    PWORD pwUser;
    PWORD pwName = NULL;

    if ((pwDomain != NULL) &&
        ((pwUser = SbsCookieUserW ()) != NULL))
        {
        pwName = _bprintfW (NULL, gawCookieName, pwUser, pwDomain);
        _mfree (pwUser);
        }
    return pwName;
    }

// =================================================================
// PROFILE ENCODER
// =================================================================

BOOL WINAPI SbsIniEscape (BYTE bData)
    {
    DWORD i;
    BOOL  fEscape = FALSE;

    if (bData)
        {
        fEscape = TRUE;

        if ((bData >= 0x20) && (bData != 0x7F))
            {
            for (i = 0;
                 gabIniEscape [i] && (gabIniEscape [i] != bData);
                 i++);

            if (!gabIniEscape [i]) fEscape = FALSE;
            }
        }
    return fEscape;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniEncoded (PVOID pData,
                            DWORD dData)
    {
    BYTE  bData;
    DWORD i;
    DWORD dCount = 0;

    for (i = 0; i < dData; i++)
        {
        bData   = (pData != NULL ? ((PBYTE) pData) [i] : 0);
        dCount += (SbsIniEscape (bData) ? 3 : 1);
        }
    return dCount;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsIniEncodeA (PVOID  pData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, 0);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    n = SbsIniEncoded (pData, dData);

    if ((pbData1 = _tnewA (n+1)) != NULL)
        {
        for (i = n = 0; i < dData; i++)
            {
            bData = (pData != NULL ? ((PBYTE) pData) [i] : 0);

            if (SbsIniEscape (bData))
                {
                pbData1 [n++] = SBS_INI_ESCAPE;
                pbData1 [n++] = gabHex [bData >>  4];
                pbData1 [n++] = gabHex [bData & 0xF];
                }
            else
                {
                switch (bData)
                    {
                    case  0 : bData = SBS_INI_NULL ; break;
                    case ' ': bData = SBS_INI_SPACE; break;
                    }
                pbData1 [n++] = bData;
                }
            }
        pbData1 [dData1 = n] = 0;
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsIniEncodeW (PVOID  pData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, 0);
    DWORD dData1  = 0;
    PWORD pwData1 = NULL;

    n = SbsIniEncoded (pData, dData);

    if ((pwData1 = _tnewW (n+1)) != NULL)
        {
        for (i = n = 0; i < dData; i++)
            {
            bData = (pData != NULL ? ((PBYTE) pData) [i] : 0);

            if (SbsIniEscape (bData))
                {
                pwData1 [n++] = SBS_INI_ESCAPE;
                pwData1 [n++] = gawHex [bData >>  4];
                pwData1 [n++] = gawHex [bData & 0xF];
                }
            else
                {
                switch (bData)
                    {
                    case  0 : bData = SBS_INI_NULL ; break;
                    case ' ': bData = SBS_INI_SPACE; break;
                    }
                pwData1 [n++] = bData;
                }
            }
        pwData1 [dData1 = n] = 0;
        }
    if (pdData != NULL) *pdData = dData1;
    return pwData1;
    }

// =================================================================
// PROFILE DECODER
// =================================================================

BOOL WINAPI SbsIniSpaceA (BYTE bData)
    {
    return (bData == ' ' ) || (bData == '\t');
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniSpaceW (WORD wData)
    {
    return (wData == ' ' ) || (wData == '\t');
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniDecodedA (PBYTE pbData,
                             DWORD dData)
    {
    DWORD i;
    DWORD dData1 = SbsTextDataA (pbData, dData);
    DWORD dCount = 0;

    if (pbData != NULL)
        {
        for (i = 0; i < dData1; i++)
            {
            if (!SbsIniSpaceA (pbData [i]))
                {
                if ((pbData [i] == SBS_INI_ESCAPE) &&
                    (i+2 < dData1) && SbsHexTestA (pbData [i+1])
                                   && SbsHexTestA (pbData [i+2]))
                    {
                    i += 2;
                    }
                dCount++;
                }
            }
        }
    return dCount;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniDecodedW (PWORD pwData,
                             DWORD dData)
    {
    DWORD i;
    DWORD dData1 = SbsTextDataW (pwData, dData);
    DWORD dCount = 0;

    if (pwData != NULL)
        {
        for (i = 0; i < dData1; i++)
            {
            if (!SbsIniSpaceW (pwData [i]))
                {
                if ((pwData [i] == SBS_INI_ESCAPE) &&
                    (i+2 < dData1) && SbsHexTestW (pwData [i+1])
                                   && SbsHexTestW (pwData [i+2]))
                    {
                    i += 2;
                    }
                dCount++;
                }
            }
        }
    return dCount;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniDecodeA (PBYTE  pbData,
                            PDWORD pdData)
    {
    BYTE  bData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pbData != NULL)
        {
        dData = SbsTextDataA   (pbData, dData);
        n     = SbsIniDecodedA (pbData, dData);

        if ((pbData1 = _mnew (n+2)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                bData = pbData [i];

                if (!SbsIniSpaceA (bData))
                    {
                    if ((bData == SBS_INI_ESCAPE) &&
                        (i+2 < dData) && SbsHexTestA (pbData [i+1])
                                      && SbsHexTestA (pbData [i+2]))
                        {
                        pbData1 [n++] = SbsHexByteA (pbData [i+1],
                                                     pbData [i+2]);
                        i += 2;
                        }
                    else
                        {
                        switch (bData)
                            {
                            case SBS_INI_NULL : bData =  0 ; break;
                            case SBS_INI_SPACE: bData = ' '; break;
                            }
                        pbData1 [n++] = bData;
                        }
                    }
                }
            pbData1 [n  ] = 0;
            pbData1 [n+1] = 0;
            dData1        = n;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniDecodeW (PWORD  pwData,
                            PDWORD pdData)
    {
    WORD  wData;
    DWORD i, n;
    DWORD dData   = SAFEVAL (pdData, -1);
    DWORD dData1  = 0;
    PBYTE pbData1 = NULL;

    if (pwData != NULL)
        {
        dData = SbsTextDataW   (pwData, dData);
        n     = SbsIniDecodedW (pwData, dData);

        if ((pbData1 = _mnew (n+2)) != NULL)
            {
            for (i = n = 0; i < dData; i++)
                {
                wData = pwData [i];

                if (!SbsIniSpaceW (wData))
                    {
                    if ((wData == SBS_INI_ESCAPE) &&
                        (i+2 < dData) && SbsHexTestW (pwData [i+1])
                                      && SbsHexTestW (pwData [i+2]))
                        {
                        pbData1 [n++] = SbsHexByteW (pwData [i+1],
                                                     pwData [i+2]);
                        i += 2;
                        }
                    else
                        {
                        switch (wData)
                            {
                            case SBS_INI_NULL : wData =  0 ; break;
                            case SBS_INI_SPACE: wData = ' '; break;
                            }
                        pbData1 [n++] = UNICODE_ANSI (wData);
                        }
                    }
                }
            pbData1 [n  ] = 0;
            pbData1 [n+1] = 0;
            dData1        = n;
            }
        }
    if (pdData != NULL) *pdData = dData1;
    return pbData1;
    }

// =================================================================
// READ FROM PROFILE
// =================================================================

PBYTE WINAPI SbsIniReadA (PBYTE  pbPath,
                          PBYTE  pbSection,
                          PBYTE  pbKey,
                          PDWORD pdData)
    {
    BOOL  fOk;
    DWORD dSize, n;
    DWORD dBytes = SBS_PAGE - SBS_MEMORY_;
    DWORD dData  = 0;
    PBYTE pbData = NULL;

    while ((pbData = _tnewA (dSize = dBytes)) != NULL)
        {
        if ((pbPath != NULL) && pbPath [0])
            {
            n = GetPrivateProfileStringA (pbSection, pbKey, gabNull,
                                          pbData, dSize, pbPath);
            }
        else
            {
            n = GetProfileStringA (pbSection, pbKey, gabNull,
                                   pbData, dSize);
            }
        if (n < dSize-2)
            {
            dBytes = (n+2);

            if ((pbData = _mresize (pbData, dBytes, &fOk)) != NULL)
                {
                pbData [n+1] = 0;
                dData        = n;
                }
            break;
            }
        pbData = _mfree (pbData);

        if ((n = dBytes + dBytes + SBS_MEMORY_) <= dBytes)
            {
            break;
            }
        dBytes = n;
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsIniReadW (PWORD  pwPath,
                          PWORD  pwSection,
                          PWORD  pwKey,
                          PDWORD pdData)
    {
    BOOL  fOk;
    DWORD dSize, n;
    DWORD dBytes = SBS_PAGE - SBS_MEMORY_;
    DWORD dData  = 0;
    PWORD pwData = NULL;

    while ((pwData = _tnewW (dSize = dBytes / WORD_)) != NULL)
        {
        if ((pwPath != NULL) && pwPath [0])
            {
            n = GetPrivateProfileStringW (pwSection, pwKey, gawNull,
                                          pwData, dSize, pwPath);
            }
        else
            {
            n = GetProfileStringW (pwSection, pwKey, gawNull,
                                   pwData, dSize);
            }
        if (n < dSize-2)
            {
            dBytes = (n+2) * WORD_;

            if ((pwData = _mresize (pwData, dBytes, &fOk)) != NULL)
                {
                pwData [n+1] = 0;
                dData        = n;
                }
            break;
            }
        pwData = _mfree (pwData);

        if ((n = dBytes + dBytes + SBS_MEMORY_) <= dBytes)
            {
            break;
            }
        dBytes = n;
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsIniReadExA (PBYTE  pbPath,
                            PBYTE  pbSection,
                            PBYTE  pbKey,
                            PDWORD pdData,
                            BOOL   fCopy)
    {
    PBYTE pbFile = _ffileA (pbPath);
    PBYTE pbData1;
    DWORD dData1, i, j, k, l, n;
    BOOL  fOk;
    DWORD dData  = 0;
    PBYTE pbData = SbsIniReadA (pbFile, pbSection, pbKey, &dData);

    if (pbFile != pbPath)
        {
        if ((pbSection != NULL) && (pbKey != NULL))
            {
            if (!dData)
                {
                _mfree (pbData);

                pbData = SbsIniReadA (pbPath, pbSection, pbKey,
                                      &dData);
                if (fCopy && dData)
                    {
                    SbsIniWriteA (pbFile, pbSection, pbKey, pbData);
                    }
                }
            }
        else
            {
            if (((pbData1 = SbsIniReadA (pbPath, pbSection, pbKey,
                                         &dData1))
                 != NULL)
                &&
                ((pbData1 = _tresizeA (pbData1, dData1 + dData + 1,
                                       NULL))
                 != NULL))
                {
                if (pbData != NULL)
                    {
                    for (i = 0, n = dData1; i < dData; i += j)
                        {
                        j = _tsizeA (pbData+i) + 1;

                        for (k = 0; k < dData1; k += l)
                            {
                            l = _tsizeA (pbData1+k) + 1;

                            if (!_tcompiA (pbData+i, pbData1+k))
                                break;
                            }
                        if (k == dData1)
                            {
                            _tcopyA2A (pbData1+n, pbData+i, j);
                            n += j;
                            }
                        }
                    pbData1 [dData1 = n] = 0;
                    pbData1 = _tresizeA (pbData1, dData1+1, &fOk);

                    _mfree (pbData);
                    }
                pbData = pbData1;
                dData  = dData1;
                }
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsIniReadExW (PWORD  pwPath,
                            PWORD  pwSection,
                            PWORD  pwKey,
                            PDWORD pdData,
                            BOOL   fCopy)
    {
    PWORD pwFile = _ffileW (pwPath);
    PWORD pwData1;
    DWORD dData1, i, j, k, l, n;
    BOOL  fOk;
    DWORD dData  = 0;
    PWORD pwData = SbsIniReadW (pwFile, pwSection, pwKey, &dData);

    if (pwFile != pwPath)
        {
        if ((pwSection != NULL) && (pwKey != NULL))
            {
            if (!dData)
                {
                _mfree (pwData);

                pwData = SbsIniReadW (pwPath, pwSection, pwKey,
                                      &dData);
                if (fCopy && dData)
                    {
                    SbsIniWriteW (pwFile, pwSection, pwKey, pwData);
                    }
                }
            }
        else
            {
            if (((pwData1 = SbsIniReadW (pwPath, pwSection, pwKey,
                                         &dData1))
                 != NULL)
                &&
                ((pwData1 = _tresizeW (pwData1, dData1 + dData + 1,
                                       NULL))
                 != NULL))
                {
                if (pwData != NULL)
                    {
                    for (i = 0, n = dData1; i < dData; i += j)
                        {
                        j = _tsizeW (pwData+i) + 1;

                        for (k = 0; k < dData1; k += l)
                            {
                            l = _tsizeW (pwData1+k) + 1;

                            if (!_tcompiW (pwData+i, pwData1+k))
                                break;
                            }
                        if (k == dData1)
                            {
                            _tcopyW2W (pwData1+n, pwData+i, j);
                            n += j;
                            }
                        }
                    pwData1 [dData1 = n] = 0;
                    pwData1 = _tresizeW (pwData1, dData1+1, &fOk);

                    _mfree (pwData);
                    }
                pwData = pwData1;
                dData  = dData1;
                }
            }
        }
    if (pdData != NULL) *pdData = dData;
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsIniReadRawA (PBYTE  pbPath,
                             PBYTE  pbSection,
                             PBYTE  pbKey,
                             PDWORD pdData)
    {
    return SbsIniReadExA (pbPath,
                          SAFEPTR (pbSection, gabSettings),
                          SAFEPTR (pbKey,     gabOptions ),
                          pdData, TRUE);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsIniReadRawW (PWORD  pwPath,
                             PWORD  pwSection,
                             PWORD  pwKey,
                             PDWORD pdData)
    {
    return SbsIniReadExW (pwPath,
                          SAFEPTR (pwSection, gawSettings),
                          SAFEPTR (pwKey,     gawOptions ),
                          pdData, TRUE);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniReadCodeA (PBYTE  pbPath,
                              PBYTE  pbSection,
                              PBYTE  pbKey,
                              PDWORD pdData)
    {
    PBYTE pbData;
    DWORD dData = 0;
    PVOID pData = NULL;

    if ((pbData = SbsIniReadRawA (pbPath, pbSection, pbKey,
                                  &dData))
        != NULL)
        {
        pData = SbsIniDecodeA (pbData, &dData);
        _mfree (pbData);
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniReadCodeW (PWORD  pwPath,
                              PWORD  pwSection,
                              PWORD  pwKey,
                              PDWORD pdData)
    {
    PWORD pwData;
    DWORD dData = 0;
    PVOID pData = NULL;

    if ((pwData = SbsIniReadRawW (pwPath, pwSection, pwKey,
                                  &dData))
        != NULL)
        {
        pData = SbsIniDecodeW (pwData, &dData);
        _mfree (pwData);
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniReadHexA (PBYTE  pbPath,
                             PBYTE  pbSection,
                             PBYTE  pbKey,
                             PDWORD pdData)
    {
    PBYTE pbData;
    DWORD dData = 0;
    PVOID pData = NULL;

    if ((pbData = SbsIniReadRawA (pbPath, pbSection, pbKey,
                                  &dData))
        != NULL)
        {
        pData = SbsHexDecodeA (pbData, &dData);
        _mfree (pbData);
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsIniReadHexW (PWORD  pwPath,
                             PWORD  pwSection,
                             PWORD  pwKey,
                             PDWORD pdData)
    {
    PWORD pwData;
    DWORD dData = 0;
    PVOID pData = NULL;

    if ((pwData = SbsIniReadRawW (pwPath, pwSection, pwKey,
                                  &dData))
        != NULL)
        {
        pData = SbsHexDecodeW (pwData, &dData);
        _mfree (pwData);
        }
    if (pdData != NULL) *pdData = dData;
    return pData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsIniReadStringA (PBYTE  pbPath,
                                PBYTE  pbSection,
                                PBYTE  pbKey,
                                PDWORD pdData)
    {
    PBYTE pbData = SbsIniReadCodeA (pbPath, pbSection, pbKey,
                                    NULL);

    if (pdData != NULL) *pdData = _tsizeA (pbData);
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsIniReadStringW (PWORD  pwPath,
                                PWORD  pwSection,
                                PWORD  pwKey,
                                PDWORD pdData)
    {
    DWORD dData;
    PBYTE pbData;
    PWORD pwData = NULL;

    if ((pbData = SbsIniReadCodeW (pwPath, pwSection, pwKey,
                                   &dData))
        != NULL)
        {
        pwData = _tnewA2W (pbData, dData+1);
        _mfree (pbData);
        }
    if (pdData != NULL) *pdData = _tsizeW (pwData);
    return pwData;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniReadStringExA (PBYTE pbPath,
                                  PBYTE pbSection,
                                  PBYTE pbKey,
                                  PBYTE pbBuffer,
                                  DWORD dBuffer)
    {
    DWORD dData  = 0;
    PBYTE pbData = SbsIniReadStringA (pbPath, pbSection, pbKey,
                                      &dData);

    _tcopynA (pbBuffer, pbData, dBuffer);
    _mfree   (pbData);
    return dData;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniReadStringExW (PWORD pwPath,
                                  PWORD pwSection,
                                  PWORD pwKey,
                                  PWORD pwBuffer,
                                  DWORD dBuffer)
    {
    DWORD dData  = 0;
    PWORD pwData = SbsIniReadStringW (pwPath, pwSection, pwKey,
                                      &dData);

    _tcopynW (pwBuffer, pwData, dBuffer);
    _mfree   (pwData);
    return dData;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniReadDwordA (PBYTE pbPath,
                               PBYTE pbSection,
                               PBYTE pbKey,
                               DWORD dDefault)
    {
    PBYTE pbData;
    DWORD dData = dDefault;

    if ((pbData = SbsIniReadRawA (pbPath, pbSection, pbKey,
                                  NULL))
        != NULL)
        {
        SbsDecimalReadA (pbData, &dData);
        _mfree (pbData);
        }
    return dData;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsIniReadDwordW (PWORD pwPath,
                               PWORD pwSection,
                               PWORD pwKey,
                               DWORD dDefault)
    {
    PWORD pwData;
    DWORD dData = dDefault;

    if ((pwData = SbsIniReadRawW (pwPath, pwSection, pwKey,
                                  NULL))
        != NULL)
        {
        SbsDecimalReadW (pwData, &dData);
        _mfree (pwData);
        }
    return dData;
    }

// -----------------------------------------------------------------

LONG WINAPI SbsIniReadLongA (PBYTE pbPath,
                             PBYTE pbSection,
                             PBYTE pbKey,
                             LONG  lDefault)
    {
    return SbsIniReadDwordA (pbPath, pbSection, pbKey, lDefault);
    }

// -----------------------------------------------------------------

LONG WINAPI SbsIniReadLongW (PWORD pwPath,
                             PWORD pwSection,
                             PWORD pwKey,
                             LONG  lDefault)
    {
    return SbsIniReadDwordW (pwPath, pwSection, pwKey, lDefault);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadBoolA (PBYTE pbPath,
                             PBYTE pbSection,
                             PBYTE pbKey,
                             BOOL  fDefault)
    {
    return SbsIniReadDwordA (pbPath, pbSection, pbKey,
                             fDefault != FALSE)
           != FALSE;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadBoolW (PWORD pwPath,
                             PWORD pwSection,
                             PWORD pwKey,
                             BOOL  fDefault)
    {
    return SbsIniReadDwordW (pwPath, pwSection, pwKey,
                             fDefault != FALSE)
           != FALSE;
    }

// -----------------------------------------------------------------

COLORREF WINAPI SbsIniReadColorA (PBYTE    pbPath,
                                  PBYTE    pbSection,
                                  PBYTE    pbKey,
                                  COLORREF crDefault)
    {
    DWORD     dData;
    PCOLORREF pcr;
    COLORREF  cr = crDefault;

    if ((pcr = SbsIniReadHexA (pbPath, pbSection, pbKey,
                               &dData))
        != NULL)
        {
        if (dData == COLORREF_) cr = *pcr;
        _mfree (pcr);
        }
    return cr;
    }

// -----------------------------------------------------------------

COLORREF WINAPI SbsIniReadColorW (PWORD    pwPath,
                                  PWORD    pwSection,
                                  PWORD    pwKey,
                                  COLORREF crDefault)
    {
    DWORD     dData;
    PCOLORREF pcr;
    COLORREF  cr = crDefault;

    if ((pcr = SbsIniReadHexW (pwPath, pwSection, pwKey,
                               &dData))
        != NULL)
        {
        if (dData == COLORREF_) cr = *pcr;
        _mfree (pcr);
        }
    return cr;
    }

// -----------------------------------------------------------------

PLOGFONTA WINAPI SbsIniReadFontA (PBYTE     pbPath,
                                  PBYTE     pbSection,
                                  PBYTE     pbKey,
                                  PLOGFONTA plfDefaultA)
    {
    DWORD     dData;
    PLOGFONTA plfA = SbsIniReadCodeA (pbPath, pbSection, pbKey,
                                      &dData);
    if (dData != LOGFONTA_)
        {
        _mfree (plfA);

        if ((plfA = _mnew (LOGFONTA_)) != NULL)
            {
            if (plfDefaultA != NULL)
                {
                *plfA = *plfDefaultA;
                }
            else
                {
                SbsFontSystemA (plfA, FALSE);
                }
            }
        }
    return plfA;
    }

// -----------------------------------------------------------------

PLOGFONTW WINAPI SbsIniReadFontW (PWORD     pwPath,
                                  PWORD     pwSection,
                                  PWORD     pwKey,
                                  PLOGFONTW plfDefaultW)
    {
    DWORD     dData;
    PLOGFONTA plfA;
    PLOGFONTW plfW = _mnew (LOGFONTW_);

    if (plfW != NULL)
        {
        plfA = SbsIniReadCodeW (pwPath, pwSection, pwKey, &dData);

        if (dData == LOGFONTA_)
            {
            SbsFontConvert (plfA, plfW, FALSE);
            }
        else
            {
            if (plfDefaultW != NULL)
                {
                *plfW = *plfDefaultW;
                }
            else
                {
                SbsFontSystemW (plfW, FALSE);
                }
            }
        _mfree (plfA);
        }
    return plfW;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadFontExA (PBYTE     pbPath,
                               PBYTE     pbSection,
                               PBYTE     pbKey,
                               PLOGFONTA plfBufferA)
    {
    PLOGFONTA plfA;
    DWORD     dData;
    BOOL      fOk = FALSE;

    plfA = SbsIniReadCodeA (pbPath, pbSection, pbKey, &dData);
    fOk  = (dData == LOGFONTA_);

    if (plfBufferA != NULL)
        {
        if (fOk)
            {
            *plfBufferA = *plfA;
            }
        else
            {
            SbsFontSystemA (plfBufferA, FALSE);
            }
        }
    _mfree (plfA);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadFontExW (PWORD     pwPath,
                               PWORD     pwSection,
                               PWORD     pwKey,
                               PLOGFONTW plfBufferW)
    {
    PLOGFONTA plfA;
    DWORD     dData;
    BOOL      fOk = FALSE;

    plfA = SbsIniReadCodeW (pwPath, pwSection, pwKey, &dData);
    fOk  = (dData == LOGFONTA_);

    if (plfBufferW != NULL)
        {
        if (fOk)
            {
            SbsFontConvert (plfA, plfBufferW, FALSE);
            }
        else
            {
            SbsFontSystemW (plfBufferW, FALSE);
            }
        }
    _mfree (plfA);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadDataA (PBYTE pbPath,
                             PBYTE pbSection,
                             PBYTE pbKey,
                             PVOID pData,
                             DWORD dType)
    {
    BOOL fOk = FALSE;

    if (pData != NULL)
        {
        switch (dType)
            {
            case SBS_INI_DWORD:
                {
                *(PDWORD) pData =
                    SbsIniReadDwordA (pbPath, pbSection, pbKey,
                                      *(PDWORD) pData);
                break;
                }
            case SBS_INI_LONG:
                {
                *(PLONG) pData =
                    SbsIniReadLongA  (pbPath, pbSection, pbKey,
                                      *(PLONG) pData);
                break;
                }
            case SBS_INI_BOOL:
                {
                *(PBOOL) pData =
                    SbsIniReadBoolA  (pbPath, pbSection, pbKey,
                                      *(PBOOL) pData);
                break;
                }
            case SBS_INI_COLOR:
                {
                *(PCOLORREF) pData =
                    SbsIniReadColorA (pbPath, pbSection, pbKey,
                                      *(PCOLORREF) pData);
                break;
                }
            case SBS_INI_FONT:
                {
                SbsIniReadFontExA    (pbPath, pbSection, pbKey,
                                      pData);
                break;
                }
            default:
                {
                SbsIniReadStringExA  (pbPath, pbSection, pbKey,
                                      pData, dType);
                break;
                }
            }
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniReadDataW (PWORD pwPath,
                             PWORD pwSection,
                             PWORD pwKey,
                             PVOID pData,
                             DWORD dType)
    {
    BOOL fOk = FALSE;

    if (pData != NULL)
        {
        switch (dType)
            {
            case SBS_INI_DWORD:
                {
                *(PDWORD) pData =
                    SbsIniReadDwordW (pwPath, pwSection, pwKey,
                                      *(PDWORD) pData);
                break;
                }
            case SBS_INI_LONG:
                {
                *(PLONG) pData =
                    SbsIniReadLongW  (pwPath, pwSection, pwKey,
                                      *(PLONG) pData);
                break;
                }
            case SBS_INI_BOOL:
                {
                *(PBOOL) pData =
                    SbsIniReadBoolW  (pwPath, pwSection, pwKey,
                                      *(PBOOL) pData);
                break;
                }
            case SBS_INI_COLOR:
                {
                *(PCOLORREF) pData =
                    SbsIniReadColorW (pwPath, pwSection, pwKey,
                                      *(PCOLORREF) pData);
                break;
                }
            case SBS_INI_FONT:
                {
                SbsIniReadFontExW    (pwPath, pwSection, pwKey,
                                      pData);
                break;
                }
            default:
                {
                SbsIniReadStringExW  (pwPath, pwSection, pwKey,
                                      pData, dType);
                break;
                }
            }
        fOk = TRUE;
        }
    return fOk;
    }

// =================================================================
// WRITE TO PROFILE
// =================================================================

BOOL WINAPI SbsIniWriteA (PBYTE pbPath,
                          PBYTE pbSection,
                          PBYTE pbKey,
                          PBYTE pbData)
    {
    PBYTE pbSections;
    DWORD i, j;
    DWORD n   = 1;
    BOOL  fOk = FALSE;

    if ((pbSections = (pbSection != NULL
                       ? pbSection
                       : SbsIniReadA (pbPath, NULL, NULL, &n)))
        != NULL)
        {
        fOk = TRUE;

        for (i = 0; i < n; i += j)
            {
            j = _tsizeA (pbSections+i) + 1;

            if ((pbPath != NULL) && pbPath [0])
                {
                if (!WritePrivateProfileStringA
                         (pbSections+i, pbKey, pbData, pbPath))
                    {
                    fOk = FALSE;
                    }
                }
            else
                {
                if (!WriteProfileStringA
                         (pbSections+i, pbKey, pbData))
                    {
                    fOk = FALSE;
                    }
                }
            }
        if (pbSection == NULL) _mfree (pbSections);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteW (PWORD pwPath,
                          PWORD pwSection,
                          PWORD pwKey,
                          PWORD pwData)
    {
    PWORD pwSections;
    DWORD i, j;
    DWORD n   = 1;
    BOOL  fOk = FALSE;

    if ((pwSections = (pwSection != NULL
                       ? pwSection
                       : SbsIniReadW (pwPath, NULL, NULL, &n)))
        != NULL)
        {
        fOk = TRUE;

        for (i = 0; i < n; i += j)
            {
            j = _tsizeW (pwSections+i) + 1;

            if ((pwPath != NULL) && pwPath [0])
                {
                if (!WritePrivateProfileStringW
                         (pwSections+i, pwKey, pwData, pwPath))
                    {
                    fOk = FALSE;
                    }
                }
            else
                {
                if (!WriteProfileStringW
                         (pwSections+i, pwKey, pwData))
                    {
                    fOk = FALSE;
                    }
                }
            }
        if (pwSection == NULL) _mfree (pwSections);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteExA (PBYTE pbPath,
                            PBYTE pbSection,
                            PBYTE pbKey,
                            PBYTE pbData)
    {
    return SbsIniWriteA (_ffileA (pbPath),
                         pbSection, pbKey, pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteExW (PWORD pwPath,
                            PWORD pwSection,
                            PWORD pwKey,
                            PWORD pwData)
    {
    return SbsIniWriteW (_ffileW (pwPath),
                         pwSection, pwKey, pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteRawA (PBYTE pbPath,
                             PBYTE pbSection,
                             PBYTE pbKey,
                             PBYTE pbData)
    {
    return SbsIniWriteExA (pbPath,
                           SAFEPTR (pbSection, gabSettings),
                           SAFEPTR (pbKey,     gabOptions ),
                           pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteRawW (PWORD pwPath,
                             PWORD pwSection,
                             PWORD pwKey,
                             PWORD pwData)
    {
    return SbsIniWriteExW (pwPath,
                           SAFEPTR (pwSection, gawSettings),
                           SAFEPTR (pwKey,     gawOptions ),
                           pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteCodeA (PBYTE pbPath,
                              PBYTE pbSection,
                              PBYTE pbKey,
                              PVOID pData,
                              DWORD dData)
    {
    PBYTE pbData = NULL;
    DWORD dData1 = dData;
    BOOL  fOk    = FALSE;

    if ((pData == NULL) ||
        ((pbData = SbsIniEncodeA (pData, &dData1)) != NULL))
        {
        fOk = SbsIniWriteRawA (pbPath, pbSection, pbKey, pbData);
        _mfree (pbData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteCodeW (PWORD pwPath,
                              PWORD pwSection,
                              PWORD pwKey,
                              PVOID pData,
                              DWORD dData)
    {
    PWORD pwData = NULL;
    DWORD dData1 = dData;
    BOOL  fOk    = FALSE;

    if ((pData == NULL) ||
        ((pwData = SbsIniEncodeW (pData, &dData1)) != NULL))
        {
        fOk = SbsIniWriteRawW (pwPath, pwSection, pwKey, pwData);
        _mfree (pwData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteHexA (PBYTE pbPath,
                             PBYTE pbSection,
                             PBYTE pbKey,
                             PVOID pData,
                             DWORD dData)
    {
    PBYTE pbData = NULL;
    DWORD dData1 = dData;
    BOOL  fOk    = FALSE;

    if ((pData == NULL) ||
        ((pbData = SbsHexEncodeA (pData, &dData1)) != NULL))
        {
        fOk = SbsIniWriteRawA (pbPath, pbSection, pbKey, pbData);
        _mfree (pbData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteHexW (PWORD pwPath,
                             PWORD pwSection,
                             PWORD pwKey,
                             PVOID pData,
                             DWORD dData)
    {
    PWORD pwData = NULL;
    DWORD dData1 = dData;
    BOOL  fOk    = FALSE;

    if ((pData == NULL) ||
        ((pwData = SbsHexEncodeW (pData, &dData1)) != NULL))
        {
        fOk = SbsIniWriteRawW (pwPath, pwSection, pwKey, pwData);
        _mfree (pwData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteStringA (PBYTE pbPath,
                                PBYTE pbSection,
                                PBYTE pbKey,
                                PBYTE pbData)
    {
    DWORD dData = _tsizeA (pbData) + 1;

    return SbsIniWriteCodeA (pbPath, pbSection, pbKey,
                             pbData, dData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteStringW (PWORD pwPath,
                                PWORD pwSection,
                                PWORD pwKey,
                                PWORD pwData)
    {
    PBYTE pbData = NULL;
    DWORD dData  = _tsizeW (pwData) + 1;
    BOOL  fOk    = FALSE;

    if ((pwData == NULL) ||
        ((pbData = _tnewW2A (pwData, dData)) != NULL))
        {
        fOk = SbsIniWriteCodeW (pwPath, pwSection, pwKey,
                                pbData, dData);
        _mfree (pbData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteDwordA (PBYTE  pbPath,
                               PBYTE  pbSection,
                               PBYTE  pbKey,
                               PDWORD pdData)
    {
    BYTE  abData [SBS_UNSIGNED32];
    PBYTE pbData = NULL;

    if (pdData != NULL)
        {
        _sprintfA (pbData = abData, "%lu", *pdData);
        }
    return SbsIniWriteRawA (pbPath, pbSection, pbKey, pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteDwordW (PWORD  pwPath,
                               PWORD  pwSection,
                               PWORD  pwKey,
                               PDWORD pdData)
    {
    WORD  awData [SBS_UNSIGNED32];
    PWORD pwData = NULL;

    if (pdData != NULL)
        {
        _sprintfW (pwData = awData, L"%lu", *pdData);
        }
    return SbsIniWriteRawW (pwPath, pwSection, pwKey, pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteLongA (PBYTE pbPath,
                              PBYTE pbSection,
                              PBYTE pbKey,
                              PLONG plData)
    {
    BYTE  abData [SBS_SIGNED32];
    PBYTE pbData = NULL;

    if (plData != NULL)
        {
        _sprintfA (pbData = abData, "%ld", *plData);
        }
    return SbsIniWriteRawA (pbPath, pbSection, pbKey, pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteLongW (PWORD pwPath,
                              PWORD pwSection,
                              PWORD pwKey,
                              PLONG plData)
    {
    WORD  awData [SBS_SIGNED32];
    PWORD pwData = NULL;

    if (plData != NULL)
        {
        _sprintfW (pwData = awData, L"%ld", *plData);
        }
    return SbsIniWriteRawW (pwPath, pwSection, pwKey, pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteBoolA (PBYTE pbPath,
                              PBYTE pbSection,
                              PBYTE pbKey,
                              PBOOL pfData)
    {
    BYTE  abData [] = " ";
    PBYTE pbData = NULL;

    if (pfData != NULL)
        {
        *(pbData = abData) = (*pfData ? '1' : '0');
        }
    return SbsIniWriteRawA (pbPath, pbSection, pbKey, pbData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteBoolW (PWORD pwPath,
                              PWORD pwSection,
                              PWORD pwKey,
                              PBOOL pfData)
    {
    WORD  awData [] = L" ";
    PWORD pwData = NULL;

    if (pfData != NULL)
        {
        *(pwData = awData) = (*pfData ? '1' : '0');
        }
    return SbsIniWriteRawW (pwPath, pwSection, pwKey, pwData);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteColorA (PBYTE     pbPath,
                               PBYTE     pbSection,
                               PBYTE     pbKey,
                               PCOLORREF pcr)
    {
    return SbsIniWriteHexA (pbPath, pbSection, pbKey,
                            pcr, COLORREF_);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteColorW (PWORD     pwPath,
                               PWORD     pwSection,
                               PWORD     pwKey,
                               PCOLORREF pcr)
    {
    return SbsIniWriteHexW (pwPath, pwSection, pwKey,
                            pcr, COLORREF_);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteFontA (PBYTE     pbPath,
                              PBYTE     pbSection,
                              PBYTE     pbKey,
                              PLOGFONTA plfA)
    {
    DWORD i;

    if (plfA != NULL)
        {
        i = _tsizeA (plfA->lfFaceName) + 1;
        _tzeroA (plfA->lfFaceName + i, LF_FACESIZE - i);
        }
    return SbsIniWriteCodeA (pbPath, pbSection, pbKey,
                             plfA, LOGFONTA_);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteFontW (PWORD     pwPath,
                              PWORD     pwSection,
                              PWORD     pwKey,
                              PLOGFONTW plfW)
    {
    DWORD     i;
    LOGFONTA  lfA;
    PLOGFONTA plfA = NULL;

    if (plfW != NULL)
        {
        i = _tsizeW (plfW->lfFaceName) + 1;
        _tzeroW (plfW->lfFaceName + i, LF_FACESIZE - i);

        SbsFontConvert (plfA = &lfA, plfW, TRUE);
        }
    return SbsIniWriteCodeW (pwPath, pwSection, pwKey,
                             plfA, LOGFONTA_);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteDataA (PBYTE pbPath,
                              PBYTE pbSection,
                              PBYTE pbKey,
                              PVOID pData,
                              DWORD dType)
    {
    BOOL fOk = FALSE;

    switch (dType)
        {
        case SBS_INI_STRING:
            {
            fOk = SbsIniWriteStringA (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        case SBS_INI_DWORD:
            {
            fOk = SbsIniWriteDwordA  (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        case SBS_INI_LONG:
            {
            fOk = SbsIniWriteLongA   (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        case SBS_INI_BOOL:
            {
            fOk = SbsIniWriteBoolA   (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        case SBS_INI_COLOR:
            {
            fOk = SbsIniWriteColorA  (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        case SBS_INI_FONT:
            {
            fOk = SbsIniWriteFontA   (pbPath, pbSection, pbKey,
                                      pData);
            break;
            }
        default:
            {
            fOk = SbsIniWriteCodeA   (pbPath, pbSection, pbKey,
                                      pData, dType);
            break;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniWriteDataW (PWORD pwPath,
                              PWORD pwSection,
                              PWORD pwKey,
                              PVOID pData,
                              DWORD dType)
    {
    BOOL fOk = FALSE;

    switch (dType)
        {
        case SBS_INI_STRING:
            {
            fOk = SbsIniWriteStringW (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        case SBS_INI_DWORD:
            {
            fOk = SbsIniWriteDwordW  (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        case SBS_INI_LONG:
            {
            fOk = SbsIniWriteLongW   (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        case SBS_INI_BOOL:
            {
            fOk = SbsIniWriteBoolW   (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        case SBS_INI_COLOR:
            {
            fOk = SbsIniWriteColorW  (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        case SBS_INI_FONT:
            {
            fOk = SbsIniWriteFontW   (pwPath, pwSection, pwKey,
                                      pData);
            break;
            }
        default:
            {
            fOk = SbsIniWriteCodeW   (pwPath, pwSection, pwKey,
                                      pData, dType);
            break;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteKeyA (PBYTE pbPath,
                              PBYTE pbSection,
                              PBYTE pbKey,
                              BOOL  fLocal)
    {
    return SbsIniWriteA ((fLocal ? _ffileA (pbPath) : pbPath),
                         SAFEPTR (pbSection, gabSettings),
                         SAFEPTR (pbKey,     gabOptions ),
                         NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteKeyW (PWORD pwPath,
                              PWORD pwSection,
                              PWORD pwKey,
                              BOOL  fLocal)
    {
    return SbsIniWriteW ((fLocal ? _ffileW (pwPath) : pwPath),
                         SAFEPTR (pwSection, gawSettings),
                         SAFEPTR (pwKey,     gawOptions ),
                         NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteKeysA (PBYTE pbPath,
                               PBYTE pbKey,
                               BOOL  fLocal)
    {
    return SbsIniWriteA ((fLocal ? _ffileA (pbPath) : pbPath),
                         NULL, SAFEPTR (pbKey, gabOptions ), NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteKeysW (PWORD pwPath,
                               PWORD pwKey,
                               BOOL  fLocal)
    {
    return SbsIniWriteW ((fLocal ? _ffileW (pwPath) : pwPath),
                         NULL, SAFEPTR (pwKey, gawOptions ), NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteSectionA (PBYTE pbPath,
                                  PBYTE pbSection,
                                  BOOL  fLocal)
    {
    return SbsIniWriteA ((fLocal ? _ffileA (pbPath) : pbPath),
                         SAFEPTR (pbSection, gabSettings),
                         NULL, NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteSectionW (PWORD pwPath,
                                  PWORD pwSection,
                                  BOOL  fLocal)
    {
    return SbsIniWriteW ((fLocal ? _ffileW (pwPath) : pwPath),
                         SAFEPTR (pwSection, gawSettings),
                         NULL, NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteAllA (PBYTE pbPath,
                              BOOL  fLocal)
    {
    return SbsIniWriteA ((fLocal ? _ffileA (pbPath) : pbPath),
                         NULL, NULL, NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsIniDeleteAllW (PWORD pwPath,
                              BOOL  fLocal)
    {
    return SbsIniWriteW ((fLocal ? _ffileW (pwPath) : pwPath),
                         NULL, NULL, NULL);
    }

// =================================================================
// PROCESS MANAGEMENT
// =================================================================

HANDLE WINAPI SbsProcessRunA (PBYTE  pbCommand,
                              DWORD  dFlags,
                              DWORD  dTimeout,
                              PDWORD pdExitCode)
    {
    STARTUPINFOA        siA;
    PROCESS_INFORMATION pi;
    DWORD               dExitCode;
    HANDLE              hProcess = NULL;

    if (pbCommand != NULL)
        {
        siA.cb              = sizeof (siA);
        siA.lpReserved      = NULL;
        siA.lpDesktop       = NULL;
        siA.lpTitle         = NULL;
        siA.dwX             = 160;
        siA.dwY             = 120;
        siA.dwXSize         = 320;
        siA.dwYSize         = 240;
        siA.dwXCountChars   = 80;
        siA.dwYCountChars   = 25;
        siA.dwFillAttribute = FOREGROUND_GRAY | BACKGROUND_BLACK;
        siA.dwFlags         = 0;
        siA.wShowWindow     = SW_SHOWNORMAL;
        siA.cbReserved2     = 0;
        siA.lpReserved2     = NULL;
        siA.hStdInput       = INVALID_HANDLE_VALUE;
        siA.hStdOutput      = INVALID_HANDLE_VALUE;
        siA.hStdError       = INVALID_HANDLE_VALUE;

        if (CreateProcessA (NULL, pbCommand, NULL, NULL, FALSE,
                            dFlags, NULL, NULL, &siA, &pi))
            {
            CloseHandle (pi.hThread);
            hProcess  =  pi.hProcess;

            if ((WaitForSingleObject (hProcess, dTimeout)
                 == WAIT_OBJECT_0)
                &&
                (pdExitCode != NULL)
                &&
                GetExitCodeProcess (hProcess, &dExitCode))
                {
                *pdExitCode = dExitCode;
                }
            }
        }
    return hProcess;
    }

// -----------------------------------------------------------------

HANDLE WINAPI SbsProcessRunW (PWORD  pwCommand,
                              DWORD  dFlags,
                              DWORD  dTimeout,
                              PDWORD pdExitCode)
    {
    STARTUPINFOW        siW;
    PROCESS_INFORMATION pi;
    DWORD               dExitCode;
    HANDLE              hProcess = NULL;

    if (pwCommand != NULL)
        {
        siW.cb              = sizeof (siW);
        siW.lpReserved      = NULL;
        siW.lpDesktop       = NULL;
        siW.lpTitle         = NULL;
        siW.dwX             = 160;
        siW.dwY             = 120;
        siW.dwXSize         = 320;
        siW.dwYSize         = 240;
        siW.dwXCountChars   = 80;
        siW.dwYCountChars   = 25;
        siW.dwFillAttribute = FOREGROUND_GRAY | BACKGROUND_BLACK;
        siW.dwFlags         = 0;
        siW.wShowWindow     = SW_SHOWNORMAL;
        siW.cbReserved2     = 0;
        siW.lpReserved2     = NULL;
        siW.hStdInput       = INVALID_HANDLE_VALUE;
        siW.hStdOutput      = INVALID_HANDLE_VALUE;
        siW.hStdError       = INVALID_HANDLE_VALUE;

        if (CreateProcessW (NULL, pwCommand, NULL, NULL, FALSE,
                            dFlags, NULL, NULL, &siW, &pi))
            {
            CloseHandle (pi.hThread);
            hProcess  =  pi.hProcess;

            if ((WaitForSingleObject (hProcess, dTimeout)
                 == WAIT_OBJECT_0)
                &&
                (pdExitCode != NULL)
                &&
                GetExitCodeProcess (hProcess, &dExitCode))
                {
                *pdExitCode = dExitCode;
                }
            }
        }
    return hProcess;
    }

// =================================================================
// THREAD MANAGEMENT
// =================================================================

PSBS_THREAD WINAPI SbsThreadOpenA (HWND        hOwner,
                                   DWORD       dTime,
                                   SBS_HANDLER Handler,
                                   PVOID       pContext,
                                   HICON       hIcon)
    {
    return SbsThreadOpenEx (hOwner, dTime, Handler, pContext, hIcon,
                            NULL,
                            SBS_DEFAULT_INFO_WIDTH,
                            SBS_DEFAULT_INFO_HEIGHT,
                            FALSE);
    }

// -----------------------------------------------------------------

PSBS_THREAD WINAPI SbsThreadOpenW (HWND        hOwner,
                                   DWORD       dTime,
                                   SBS_HANDLER Handler,
                                   PVOID       pContext,
                                   HICON       hIcon)
    {
    return SbsThreadOpenEx (hOwner, dTime, Handler, pContext, hIcon,
                            NULL,
                            SBS_DEFAULT_INFO_WIDTH,
                            SBS_DEFAULT_INFO_HEIGHT,
                            TRUE);
    }

// -----------------------------------------------------------------

PSBS_THREAD WINAPI SbsThreadOpenEx (HWND        hOwner,
                                    DWORD       dTime,
                                    SBS_HANDLER Handler,
                                    PVOID       pContext,
                                    HICON       hIcon,
                                    HBITMAP     hBitmap,
                                    DWORD       dWidth,
                                    DWORD       dHeight,
                                    BOOL        fUnicode)
    {
    SYSTEMTIME  st;
    FILETIME    ft;
    HANDLE      ahObjects [2];
    PSBS_THREAD pst = NULL;

    if ((pst = _mnew (SBS_THREAD_)) != NULL)
        {
        GetSystemTime        (&st);
        SystemTimeToFileTime (&st, &ft);

        SbsLockReset (&pst->Lock);

        pst->Info.fUnicode = fUnicode;
        pst->Info.hIcon    = hIcon;
        pst->Info.hBitmap  = hBitmap;
        pst->Info.crText   = SBS_DEFAULT_FOREGROUND;
        pst->Info.crArea   = SBS_DEFAULT_BACKGROUND;

        pst->Pad.pData     = NULL;
        pst->Pad.dData     = 0;
        pst->Pad.fData     = FALSE;

        pst->dWidth   = dWidth;
        pst->dHeight  = dHeight;
        pst->dTime    = dTime;
        pst->dTimer   = 0;
        pst->Handler  = Handler;
        pst->pContext = pContext;
        pst->dOwner   = GetCurrentThreadId ();
        pst->hOwner   = hOwner;
        pst->hWnd     = NULL;
        pst->hEvent   = CreateEvent (NULL, FALSE, FALSE, NULL);
        pst->hThread  = NULL;
        pst->dThread  = 0;
        pst->dSeconds = 0;
        pst->ftStart  = ft;
        pst->ftNow    = ft;

        if (fUnicode)
            {
            if ((!SbsWindowValid (hOwner)) ||
                (!GetWindowTextW (hOwner, pst->Info.awInfo,
                                          SBS_STRING)))
                {
                _tcopynW (pst->Info.awInfo, gawCaption, SBS_STRING);
                }
            }
        else
            {
            if ((!SbsWindowValid (hOwner)) ||
                (!GetWindowTextA (hOwner, pst->Info.abInfo,
                                          SBS_STRING)))
                {
                _tcopynA (pst->Info.abInfo, gabCaption, SBS_STRING);
                }
            }
        if (pst->hEvent != NULL)
            {
            if ((pst->hThread = CreateThread (NULL, 0,
                                              SbsThreadHandler,
                                              pst, 0,
                                              &pst->dThread))
                == NULL)
                {
                CloseHandle (pst->hEvent);
                pst->hEvent = NULL;
                }
            }
        if (pst->hEvent != NULL)
            {
            ahObjects [0] = pst->hThread;
            ahObjects [1] = pst->hEvent;

            if (WaitForMultipleObjects (2, ahObjects, FALSE,
                                        INFINITE)
                == WAIT_OBJECT_0)
                {
                CloseHandle (pst->hThread);
                pst->hThread = NULL;

                CloseHandle (pst->hEvent);
                pst->hEvent = NULL;
                }
            }
        if (pst->hEvent == NULL)
            {
            pst = _mfree (pst);
            }
        }
    return pst;
    }

// -----------------------------------------------------------------

PSBS_THREAD WINAPI SbsThreadClose (PSBS_THREAD pst)
    {
    if (pst != NULL)
        {
        if (SbsWindowValid (pst->hWnd))
            {
            SbsThreadSend       (pst, WM_APP_CLEANUP, 0, 0);
            WaitForSingleObject (pst->hThread, INFINITE);
            }
        CloseHandle (pst->hThread);
        CloseHandle (pst->hEvent);

        _mfree (pst);
        }
    return NULL;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadSend (PSBS_THREAD pst,
                              UINT        uiMsg,
                              WPARAM      wParam,
                              LPARAM      lParam)
    {
    return (pst != NULL
            ? (pst->Info.fUnicode ? SendMessageW : SendMessageA)
                                  (pst->hWnd, uiMsg, wParam, lParam)
            : -1);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadPost (PSBS_THREAD pst,
                           UINT        uiMsg,
                           WPARAM      wParam,
                           LPARAM      lParam)
    {
    return (pst != NULL
            ? (pst->Info.fUnicode ? PostMessageW : PostMessageA)
                                  (pst->hWnd, uiMsg, wParam, lParam)
            : FALSE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadPeek (PSBS_THREAD pst,
                           PMSG        pMsg,
                           UINT        uiFilterMin,
                           UINT        uiFilterMax,
                           UINT        uiRemove)
    {
    MSG  m;
    PMSG pm = SAFEPTR (pMsg, &m);

    return (pst != NULL

            ? (pst->Info.fUnicode
               ? PeekMessageW
               : PeekMessageA) (pm, NULL, uiFilterMin, uiFilterMax,
                                uiRemove)
            : FALSE);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadGet (PSBS_THREAD pst,
                             PMSG        pMsg,
                             UINT        uiFilterMin,
                             UINT        uiFilterMax)
    {
    MSG  m;
    PMSG pm = SAFEPTR (pMsg, &m);

    return (pst != NULL
            ? (pst->Info.fUnicode
               ? GetMessageW
               : GetMessageA) (pm, NULL, uiFilterMin, uiFilterMax)
            : -1);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadDispatch (PSBS_THREAD pst,
                                  PMSG        pMsg)
    {
    return ((pst != NULL) && (pMsg != NULL)
            ? (pst->Info.fUnicode
               ? DispatchMessageW
               : DispatchMessageA) (pMsg)
            : -1);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsThreadHandler (PSBS_THREAD pst)
    {
    RECT       r;
    MSG        m;
    SYSTEMTIME st;
    LRESULT    lr;

    if (pst != NULL)
        {
        r.left   = 0;
        r.top    = 0;
        r.right  = pst->dWidth;
        r.bottom = pst->dHeight;

        pst->hWnd =
            (pst->Info.fUnicode

             ? CreateWindowExW (0, gawSbsClass, pst->Info.awInfo,
                                WS_POPUP, SBS_RECT (r),
                                NULL, NULL, ghInstance, pst)

             : CreateWindowExA (0, gabSbsClass, pst->Info.abInfo,
                                WS_POPUP, SBS_RECT (r),
                                NULL, NULL, ghInstance, pst));

        if (pst->hWnd != NULL)
            {
            SbsWindowCenter (pst->hWnd, SW_HIDE);
            SbsWindowSelect (pst->hOwner);
            
            if (SbsThreadPost (pst, WM_APP_STARTUP, 0, 0))
                {
                while ((lr  = SbsThreadGet (pst, &m, 0, 0)) &&
                       (lr != -1))
                    {
                    GetSystemTime        (&st);
                    SystemTimeToFileTime (&st, &pst->ftNow);

                    pst->dSeconds = SBS_SECONDS (pst->ftStart,
                                                 pst->ftNow);
                    TranslateMessage  (&m);
                    SbsThreadDispatch (pst, &m);
                    }
                }
            }
        }
    return 0;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadSpy (PSBS_THREAD pst,
                          DWORD       dTimeout,
                          SBS_SPY     Spy,
                          PVOID       pContext)
    {
    MSG        m;
    SYSTEMTIME st;
    DWORD      dTime;
    BOOL       fQuit = FALSE;
    BOOL       fOk   = FALSE;

    if (SbsThreadLock (pst))
        {
        dTime = GetTickCount ();

        while (TRUE)
            {
            while (SbsThreadPeek (pst, &m, 0, 0, PM_NOREMOVE))
                {
                if (fQuit = (m.message == WM_QUIT)) break;

                if (SbsThreadPeek (pst, &m, 0, 0, PM_REMOVE))
                    {
                    GetSystemTime        (&st);
                    SystemTimeToFileTime (&st, &pst->ftNow);

                    pst->dSeconds = SBS_SECONDS (pst->ftStart,
                                                 pst->ftNow);
                    TranslateMessage  (&m);
                    SbsThreadDispatch (pst, &m);

                    if (fOk = Spy (pst, pContext)) break;
                    }
                }
            if (fQuit || fOk
                ||
                ((dTimeout != -1                     ) &&
                 (dTimeout <= GetTickCount () - dTime)))
                {
                break;
                }
            Sleep (0);
            }
        SbsThreadUnlock (pst);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadLock (PSBS_THREAD pst)
    {
    return (pst != NULL) && SbsLockSet (&pst->Lock, FALSE);
    }

// -----------------------------------------------------------------

VOID WINAPI SbsThreadUnlock (PSBS_THREAD pst)
    {
    if (pst != NULL) SbsLockReset (&pst->Lock);
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadAttach (PSBS_THREAD pst)
    {
    return (pst != NULL) &&
           AttachThreadInput (GetCurrentThreadId (),
                              pst->dThread, TRUE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadDetach (PSBS_THREAD pst)
    {
    return (pst != NULL) &&
           AttachThreadInput (GetCurrentThreadId (),
                              pst->dThread, FALSE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadTrace (PSBS_THREAD pst,
                            HICON       hIcon,
                            PVOID       pInfo,
                            DWORD       dDelay)
    {
    BOOL fUpdate = FALSE;
    BOOL fOk     = FALSE;

    if (pst != NULL)
        {
        fOk = TRUE;

        if (hIcon != NULL)
            {
            if (pst->Info.hIcon != hIcon)
                {
                pst->Info.hIcon = hIcon;
                fUpdate = TRUE;
                }
            }
        if (pInfo != NULL)
            {
            if (pst->Info.fUnicode)
                {
                if (_tcompW (pst->Info.awInfo, pInfo))
                    {
                    _tcopynW (pst->Info.awInfo, pInfo, SBS_STRING);
                    SetWindowTextW (pst->hWnd, pst->Info.awInfo);
                    fUpdate = TRUE;
                    }
                }
            else
                {
                if (_tcompA (pst->Info.abInfo, pInfo))
                    {
                    _tcopynA (pst->Info.abInfo, pInfo, SBS_STRING);
                    SetWindowTextA (pst->hWnd, pst->Info.abInfo);
                    fUpdate = TRUE;
                    }
                }
            }
        if ((hIcon != NULL) || (pInfo != NULL))
            {
            ShowWindow      (pst->hWnd, SW_SHOWNORMAL);
            SbsWindowSelect (pst->hWnd);

            if (fUpdate &&
                (!InvalidateRect (pst->hWnd, NULL, FALSE)))
                {
                fOk = FALSE;
                }
            if (!UpdateWindow (pst->hWnd))
                {
                fOk = FALSE;
                }
            }
        else
            {
            SbsWindowSelect (pst->hOwner);
            ShowWindow      (pst->hWnd, SW_HIDE);
            }
        if (dDelay != -1) Sleep (dDelay);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadDraw (PSBS_THREAD pst,
                           HDC         hDC)
    {
    RECT r;
    BOOL fCenter;
    BOOL fOk = FALSE;

    if ((pst != NULL) && (hDC != NULL) &&
        GetClientRect (pst->hWnd, &r))
        {
        fCenter = (pst->Info.hIcon == NULL);

        fOk = (pst->Info.fUnicode

               ? SbsDrawInfoW (hDC,
                               pst->Info.hBitmap, pst->Info.hIcon,
                               pst->Info.awInfo,
                               &pst->Info.crText, &pst->Info.crArea,
                               SBS_RECT (r), fCenter)

               : SbsDrawInfoA (hDC,
                               pst->Info.hBitmap, pst->Info.hIcon,
                               pst->Info.abInfo,
                               &pst->Info.crText, &pst->Info.crArea,
                               SBS_RECT (r), fCenter));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadUpdateA (PSBS_THREAD pst,
                              HICON       hIcon,
                              PBYTE       pbInfo)
    {
    PVOID pInfo, pInfo1;
    BOOL  fOk = FALSE;

    if (pst != NULL)
        {
        if (pst->Info.fUnicode)
            {
            if ((pInfo1 = pInfo = _tnewA2W (pbInfo, -1)) == NULL)
                {
                pInfo1 = TESTNULL (pbInfo, gawNull, NULL);
                }
            }
        else
            {
            if ((pInfo1 = pInfo = _tnewA2A (pbInfo, -1)) == NULL)
                {
                pInfo1 = TESTNULL (pbInfo, gabNull, NULL);
                }
            }
        fOk = (SbsThreadInfo (pst, hIcon, pInfo1) == TRUE);

        _mfree (pInfo);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsThreadUpdateW (PSBS_THREAD pst,
                              HICON       hIcon,
                              PWORD       pwInfo)
    {
    PVOID pInfo, pInfo1;
    BOOL  fOk = FALSE;

    if (pst != NULL)
        {
        if (pst->Info.fUnicode)
            {
            if ((pInfo1 = pInfo = _tnewW2W (pwInfo, -1)) == NULL)
                {
                pInfo1 = TESTNULL (pwInfo, gawNull, NULL);
                }
            }
        else
            {
            if ((pInfo1 = pInfo = _tnewW2A (pwInfo, -1)) == NULL)
                {
                pInfo1 = TESTNULL (pwInfo, gabNull, NULL);
                }
            }
        fOk = (SbsThreadInfo (pst, hIcon, pInfo1) == TRUE);

        _mfree (pInfo);
        }
    return fOk;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadInfo (PSBS_THREAD pst,
                              HICON       hIcon,
                              PVOID       pInfo)
    {
    LRESULT lr = -1;

    if (pst != NULL)
        {
        lr = SbsThreadSend (pst, WM_APP_INFO,
                            (WPARAM) hIcon,
                            (LPARAM) pInfo);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadIdle (PSBS_THREAD pst)
    {
    LRESULT lr = -1;

    if (pst != NULL)
        {
        lr = SbsThreadSend (pst, WM_APP_IDLE, 0, 0);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadCommand (PSBS_THREAD pst,
                                 DWORD       dCommand,
                                 PVOID       pCommand)
    {
    LRESULT lr = -1;

    if (pst != NULL)
        {
        lr = SbsThreadSend (pst, WM_APP_COMMAND,
                            (WPARAM) dCommand,
                            (LPARAM) pCommand);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadNotify (PSBS_THREAD pst,
                                DWORD       dCommand,
                                PVOID       pCommand)
    {
    LRESULT lr = -1;

    if ((pst != NULL) && (pst->Handler != NULL))
        {
        lr = pst->Handler (pst, dCommand, pCommand);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT CALLBACK SbsThreadWindow (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    return SbsWindowDispatchEx
               (hWnd, uiMsg, wParam, lParam, uiMsg,
                WM_APP_STARTUP,      SbsThreadWindowStartup,
                WM_APP_CLEANUP,      SbsThreadWindowCleanup,
                WM_APP_INFO,         SbsThreadWindowInfo,
                WM_APP_IDLE,         SbsThreadWindowIdle,
                WM_APP_COMMAND,      SbsThreadWindowCommand,
                WM_APP_MESSAGE,      SbsThreadWindowMessage,
                WM_PAINT,            SbsThreadWindowPaint,
                WM_DESTROY,          SbsThreadWindowDestroy,
                WM_TIMER,            SbsThreadWindowTimer,
                WM_COPYDATA,         SbsThreadWindowCopyData,
                WM_LBUTTONDOWN,      SbsWmLButtonDown,
                SBS_DEFAULT_HANDLER, SbsWmDefault);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowStartup (HWND        hWnd,
                                       UINT        uiMsg,
                                       WPARAM      wParam,
                                       LPARAM      lParam,
                                       PSBS_THREAD pst)
    {
    BOOL fOk = FALSE;

    if (SbsThreadNotify (pst, SBS_COMMAND_STARTUP, &pst->Pad)
        &&
        ((!pst->dTime) ||
         (pst->dTimer = SetTimer (hWnd, 1, pst->dTime, NULL))))
        {
        fOk = SetEvent (pst->hEvent);
        }
    if (fOk)
        {
        if (pst->dTime)
            {
            SbsThreadPost (pst, WM_TIMER,
                           (WPARAM) pst->dTimer,
                           (LPARAM) NULL);
            }
        }
    else
        {
        SbsThreadPost (pst, WM_APP_CLEANUP, 0, 0);
        }
    return fOk;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowCleanup (HWND        hWnd,
                                       UINT        uiMsg,
                                       WPARAM      wParam,
                                       LPARAM      lParam,
                                       PSBS_THREAD pst)
    {
    if (pst->dTimer)
        {
        KillTimer (hWnd, pst->dTimer);
        pst->dTimer = 0;
        }
    SbsThreadNotify (pst, SBS_COMMAND_CLEANUP, &pst->Pad);
    DestroyWindow   (hWnd);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowInfo (HWND        hWnd,
                                    UINT        uiMsg,
                                    WPARAM      wParam,
                                    LPARAM      lParam,
                                    PSBS_THREAD pst)
    {
    return SbsThreadTrace (pst, (HICON) wParam, (PVOID) lParam, -1);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowIdle (HWND        hWnd,
                                    UINT        uiMsg,
                                    WPARAM      wParam,
                                    LPARAM      lParam,
                                    PSBS_THREAD pst)
    {
    return (pst->Info.fUnicode ? SbsWindowIdleW (NULL)
                               : SbsWindowIdleA (NULL));
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowCommand (HWND        hWnd,
                                       UINT        uiMsg,
                                       WPARAM      wParam,
                                       LPARAM      lParam,
                                       PSBS_THREAD pst)
    {
    return SbsThreadNotify (pst, (DWORD) wParam, (PVOID) lParam);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowMessage (HWND        hWnd,
                                       UINT        uiMsg,
                                       WPARAM      wParam,
                                       LPARAM      lParam,
                                       PSBS_THREAD pst)
    {
    SBS_MESSAGE sm;

    sm.hWnd   = hWnd;
    sm.uiMsg  = uiMsg;
    sm.wParam = wParam;
    sm.lParam = lParam;
    sm.pst    = pst;

    return SbsThreadNotify (pst, SBS_COMMAND_MESSAGE, &sm);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowPaint (HWND        hWnd,
                                     UINT        uiMsg,
                                     WPARAM      wParam,
                                     LPARAM      lParam,
                                     PSBS_THREAD pst)
    {
    PAINTSTRUCT ps;

    if (GetUpdateRect (hWnd, NULL, FALSE))
        {
        BeginPaint    (hWnd, &ps);
        SbsThreadDraw (pst, ps.hdc);
        EndPaint      (hWnd, &ps);
        }
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowDestroy (HWND        hWnd,
                                       UINT        uiMsg,
                                       WPARAM      wParam,
                                       LPARAM      lParam,
                                       PSBS_THREAD pst)
    {
    PostQuitMessage (0);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowTimer (HWND        hWnd,
                                     UINT        uiMsg,
                                     WPARAM      wParam,
                                     LPARAM      lParam,
                                     PSBS_THREAD pst)
    {
    return SbsThreadNotify (pst, SBS_COMMAND_TIMER,
                                 &pst->Pad);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsThreadWindowCopyData (HWND        hWnd,
                                        UINT        uiMsg,
                                        WPARAM      wParam,
                                        LPARAM      lParam,
                                        PSBS_THREAD pst)
    {
    return SbsThreadNotify (pst, SBS_COMMAND_COPYDATA,
                                 (PCOPYDATASTRUCT) lParam);
    }

// =================================================================
// WINDOW MANAGEMENT
// =================================================================

BOOL WINAPI SbsWindowValid (HWND hWnd)
    {
    return (hWnd != NULL) && IsWindow (hWnd);
    }

// -----------------------------------------------------------------

LONG WINAPI SbsWindowLongRead (HWND hWnd,
                               INT  iIndex)
    {
    return (SbsWindowValid (hWnd)
            ? (IsWindowUnicode (hWnd)
               ? GetWindowLongW (hWnd, iIndex)
               : GetWindowLongA (hWnd, iIndex))
            : 0);
    }

// -----------------------------------------------------------------

LONG WINAPI SbsWindowLongWrite (HWND hWnd,
                                INT  iIndex,
                                LONG lData)
    {
    return (SbsWindowValid (hWnd)
            ? (IsWindowUnicode (hWnd)
               ? SetWindowLongW (hWnd, iIndex, lData)
               : SetWindowLongA (hWnd, iIndex, lData))
            : 0);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsWindowUserRead (HWND hWnd)
    {
    return (PVOID) SbsWindowLongRead (hWnd, GWL_USERDATA);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsWindowUserWrite (HWND  hWnd,
                                 PVOID pUser)
    {
    return (PVOID) SbsWindowLongWrite (hWnd, GWL_USERDATA,
                                       (LONG) pUser);
    }

// -----------------------------------------------------------------

WNDPROC WINAPI SbsWindowProcedure (HWND hWnd)
    {
    LONG    lData;
    WNDPROC wp = NULL;

    if (lData = SbsWindowLongRead (hWnd, GWL_WNDPROC))
        {
        wp = (WNDPROC) lData;
        }
    return wp;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowStyle (HWND hWnd)
    {
    LONG  lData;
    DWORD dStyle = 0;

    if (lData = SbsWindowLongRead (hWnd, GWL_STYLE))
        {
        dStyle = lData;
        }
    return dStyle;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowCall (WNDPROC wp,
                              HWND    hWnd,
                              UINT    uiMsg,
                              WPARAM  wParam,
                              LPARAM  lParam,
                              LRESULT lrError)
    {
    LRESULT lr = lrError;

    if ((wp != NULL) && SbsWindowValid (hWnd))
        {
        lr = (IsWindowUnicode (hWnd)
              ? CallWindowProcW (wp, hWnd, uiMsg, wParam, lParam)
              : CallWindowProcA (wp, hWnd, uiMsg, wParam, lParam));
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowDefault (HWND    hWnd,
                                 UINT    uiMsg,
                                 WPARAM  wParam,
                                 LPARAM  lParam,
                                 LRESULT lrError)
    {
    LRESULT lr = lrError;

    if (SbsWindowValid (hWnd))
        {
        lr = (IsWindowUnicode (hWnd)
              ? DefWindowProcW (hWnd, uiMsg, wParam, lParam)
              : DefWindowProcA (hWnd, uiMsg, wParam, lParam));
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowSend (HWND    hWnd,
                              UINT    uiMsg,
                              WPARAM  wParam,
                              LPARAM  lParam,
                              LRESULT lrError)
    {
    LRESULT lr = lrError;

    if (SbsWindowValid (hWnd))
        {
        lr = (IsWindowUnicode (hWnd)
              ? SendMessageW (hWnd, uiMsg, wParam, lParam)
              : SendMessageA (hWnd, uiMsg, wParam, lParam));
        }
    return lr;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowPost (HWND   hWnd,
                           UINT   uiMsg,
                           WPARAM wParam,
                           LPARAM lParam)
    {
    BOOL fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        fOk = (IsWindowUnicode (hWnd)
               ? PostMessageW (hWnd, uiMsg, wParam, lParam)
               : PostMessageA (hWnd, uiMsg, wParam, lParam));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowRectangle (HWND   hWnd,
                                PRECT  pr,
                                PDWORD pdMode)
    {
    WINDOWPLACEMENT wp;
    RECT            r;
    DWORD           dMode;
    BOOL            fOk   = FALSE;

    wp.length = WINDOWPLACEMENT_;

    if (SbsWindowValid (hWnd) && GetWindowPlacement (hWnd, &wp))
        {
        r.left   = wp.rcNormalPosition.left;
        r.top    = wp.rcNormalPosition.top;
        r.right  = wp.rcNormalPosition.right;
        r.bottom = wp.rcNormalPosition.bottom;
        dMode    = wp.showCmd;

        fOk = TRUE;
        }
    else
        {
        r.left   = 0;
        r.top    = 0;
        r.right  = SBS_DEFAULT_WINDOW_WIDTH;
        r.bottom = SBS_DEFAULT_WINDOW_HEIGHT;
        dMode    = SW_SHOWNORMAL;
        }
    if (pr     != NULL) *pr     = r;
    if (pdMode != NULL) *pdMode = dMode;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDesktop (PRECT pr)
    {
    BOOL fOk = FALSE;

    if (SystemParametersInfoA (SPI_GETWORKAREA, 0, pr, 0))
        {
        fOk = TRUE;
        }
    else
        {
        pr->left   = 0;
        pr->top    = 0;
        pr->right  = SBS_DEFAULT_DESKTOP_WIDTH;
        pr->bottom = SBS_DEFAULT_DESKTOP_HEIGHT;
        }
    return fOk;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsWindowClipRectangle (PRECT pr)
    {
    RECT rd;
    LONG lWidth, lHeight;

    SbsWindowDesktop (&rd);

    lWidth  = rd.right  - rd.left;
    lHeight = rd.bottom - rd.top;

    if ((pr->right - pr->left) > lWidth)
        {
        pr->right = pr->left + lWidth;
        }
    if ((pr->bottom - pr->top) > lHeight)
        {
        pr->bottom = pr->top + lHeight;
        }
    if (pr->left < 0)
        {
        pr->right -= pr->left;
        pr->left   = 0;
        }
    else
        {
        if (pr->right > lWidth)
            {
            pr->left  -= (pr->right - lWidth);
            pr->right  = lWidth;
            }
        }
    if (pr->top < 0)
        {
        pr->bottom -= pr->top;
        pr->top     = 0;
        }
    else
        {
        if (pr->bottom > lHeight)
            {
            pr->top    -= (pr->bottom - lHeight);
            pr->bottom  = lHeight;
            }
        }
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsWindowClipPosition (PDWORD pdColumn,
                                   PDWORD pdRow,
                                   PDWORD pdWidth,
                                   PDWORD pdHeight)
    {
    RECT  rd;
    DWORD dWidth, dHeight;

    SbsWindowDesktop (&rd);

    dWidth  = rd.right  - rd.left;
    dHeight = rd.bottom - rd.top;

    if (*pdWidth  > dWidth ) *pdWidth  = dWidth;
    if (*pdHeight > dHeight) *pdHeight = dHeight;

    if ((*pdColumn + *pdWidth < *pdColumn) ||
        (*pdColumn + *pdWidth > dWidth   ))
        {
        *pdColumn = (dWidth - *pdWidth);
        }
    if ((*pdRow + *pdHeight < *pdRow ) ||
        (*pdRow + *pdHeight > dHeight))
        {
        *pdRow = (dHeight - *pdHeight);
        }
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowMove (HWND  hWnd,
                           PRECT pr,
                           DWORD dMode)
    {
    WINDOWPLACEMENT wp;
    BOOL            fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        wp.length                  = WINDOWPLACEMENT_;
        wp.flags                   = 0;
        wp.showCmd                 = dMode;
        wp.ptMinPosition.x         = 0;
        wp.ptMinPosition.y         = 0;
        wp.ptMaxPosition.x         = 0;
        wp.ptMaxPosition.y         = 0;
        wp.rcNormalPosition.left   = pr->left;
        wp.rcNormalPosition.top    = pr->top;
        wp.rcNormalPosition.right  = pr->right;
        wp.rcNormalPosition.bottom = pr->bottom;

        fOk = SetWindowPlacement (hWnd, &wp);
        UpdateWindow (hWnd);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowMoveEx (HWND   hWnd,
                             DWORD  dMode,
                             PDWORD pdColumn,
                             PDWORD pdRow,
                             PDWORD pdWidth,
                             PDWORD pdHeight)
    {
    RECT r;
    BOOL fOk = FALSE;

    SbsWindowClipPosition (pdColumn, pdRow, pdWidth, pdHeight);

    if (SbsWindowValid (hWnd))
        {
        r.left   = *pdColumn;
        r.top    = *pdRow;
        r.right  = *pdColumn + *pdWidth;
        r.bottom = *pdRow    + *pdHeight;

        fOk = SbsWindowMove (hWnd, &r, dMode);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowRecord (HWND hWnd)
    {
    PSBS_WINDOW psw;
    RECT        r;
    BOOL        fOk = FALSE;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        SbsWindowRectangle (hWnd, &r, NULL))
        {
        psw->dColumn = (r.left > 0 ? r.left : 0);
        psw->dRow    = (r.top  > 0 ? r.top  : 0);
        psw->dWidth  = r.right  - r.left;
        psw->dHeight = r.bottom - r.top;

        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowZoom (HWND  hWnd,
                           DWORD dWidth,
                           DWORD dHeight,
                           BOOL  fClient)
    {
    SBS_WINDOW sw;
    RECT       rOuter, rInner;
    DWORD      dMode, dWidth1, dHeight1, dDelta, i;
    BOOL       fOk = FALSE;

    for (i = 0; i < 3; i++)
        {
        if (!SbsWindowRectangle (hWnd, &rOuter, &dMode)) break;

        if (dMode != SW_SHOWNORMAL)
            {
            ShowWindow (hWnd, SW_SHOWNORMAL);
            }
        sw.dColumn = (rOuter.left > 0 ? rOuter.left : 0);
        sw.dRow    = (rOuter.top  > 0 ? rOuter.top  : 0);
        sw.dWidth  = rOuter.right  - rOuter.left;
        sw.dHeight = rOuter.bottom - rOuter.top;

        if (fClient)
            {
            if (!GetClientRect (hWnd, &rInner)) break;
            }
        else
            {
            rInner = rOuter;
            }
        dWidth1  = (rInner.right  - rInner.left);
        dHeight1 = (rInner.bottom - rInner.top ) -
                   SbsStatusHeight (hWnd);

        if ((dWidth1 == dWidth) && (dHeight1 == dHeight))
            {
            fOk = TRUE;
            break;
            }
        dWidth1  = dWidth +
                   ((rOuter.right - rOuter.left) - dWidth1);

        dHeight1 = dHeight +
                   ((rOuter.bottom - rOuter.top) - dHeight1);

        if (dWidth1 > sw.dWidth)
            {
            dDelta      = (dWidth1 - sw.dWidth) / 2;
            sw.dColumn -= min (dDelta, sw.dColumn);
            }
        else
            {
            dDelta      = (sw.dWidth - dWidth1) / 2;
            sw.dColumn += dDelta;
            }
        sw.dWidth = dWidth1;

        if (dHeight1 > sw.dHeight)
            {
            dDelta   = (dHeight1 - sw.dHeight) / 2;
            sw.dRow -= min (dDelta, sw.dRow);
            }
        else
            {
            dDelta   = (sw.dHeight - dHeight1) / 2;
            sw.dRow += dDelta;
            }
        sw.dHeight = dHeight1;

        if (!SbsWindowMoveEx (hWnd, SW_SHOWNORMAL,
                              &sw.dColumn, &sw.dRow,
                              &sw.dWidth,  &sw.dHeight)) break;
        if (!fClient)
            {
            fOk = TRUE;
            break;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowRestore (HWND  hWnd,
                              DWORD dMode)
    {
    PSBS_WINDOW psw;
    BOOL        fOk = FALSE;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        fOk = SbsWindowMoveEx (hWnd, dMode,
                               &psw->dColumn, &psw->dRow,
                               &psw->dWidth,  &psw->dHeight);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowCenter (HWND  hWnd,
                             DWORD dMode)
    {
    RECT rd, rw;
    BOOL fOk = FALSE;

    if (SbsWindowDesktop   (&rd) &&
        SbsWindowRectangle (hWnd, &rw, NULL))
        {
        rw.right  -= rw.left;
        rw.bottom -= rw.top;
        rw.left    = ((rd.right  - rd.left) - rw.right ) / 2;
        rw.top     = ((rd.bottom - rd.top ) - rw.bottom) / 2;
        rw.right  += rw.left;
        rw.bottom += rw.top;

        fOk = SbsWindowMove (hWnd, &rw, dMode);
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowCenterColumn (PDWORD pdWidth)
    {
    RECT  r;
    DWORD dWidth, dWidthMax;

    SbsWindowDesktop (&r);
    dWidthMax = r.right - r.left;

    dWidth = ((pdWidth != NULL) && (*pdWidth != MAXDWORD)
              ? *pdWidth
              : SBS_DEFAULT_WINDOW_WIDTH);

    if (dWidth > dWidthMax) dWidth = dWidthMax;

    if (pdWidth != NULL) *pdWidth = dWidth;
    return (dWidthMax - dWidth) / 2;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowCenterRow (PDWORD pdHeight)
    {
    RECT  r;
    DWORD dHeight, dHeightMax;

    SbsWindowDesktop (&r);
    dHeightMax = r.bottom - r.top;

    dHeight = ((pdHeight != NULL) && (*pdHeight != MAXDWORD)
               ? *pdHeight
               : SBS_DEFAULT_WINDOW_HEIGHT);

    if (dHeight > dHeightMax) dHeight = dHeightMax;

    if (pdHeight != NULL) *pdHeight = dHeight;
    return (dHeightMax - dHeight) / 2;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsWindowSave (HWND hWnd)
    {
    PSBS_WINDOW psw;
    RECT        r;
    DWORD       dColumn, dRow, dWidth, dHeight;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        SbsWindowRectangle (hWnd, &r, NULL))
        {
        SbsWindowClipRectangle (&r);

        dColumn = r.left;
        dRow    = r.top;
        dWidth  = r.right  - r.left;
        dHeight = r.bottom - r.top;

        if (psw->fUnicode)
            {
            SbsIniWriteDwordW (psw->awProfile, psw->awWindow,
                               gawSettingsColumn,
                               &dColumn);

            SbsIniWriteDwordW (psw->awProfile, psw->awWindow,
                               gawSettingsRow,
                               &dRow);

            SbsIniWriteDwordW (psw->awProfile, psw->awWindow,
                               gawSettingsWidth,
                               &dWidth);

            SbsIniWriteDwordW (psw->awProfile, psw->awWindow,
                               gawSettingsHeight,
                               &dHeight);
            }
        else
            {
            SbsIniWriteDwordA (psw->abProfile, psw->abWindow,
                               gabSettingsColumn,
                               &dColumn);

            SbsIniWriteDwordA (psw->abProfile, psw->abWindow,
                               gabSettingsRow,
                               &dRow);

            SbsIniWriteDwordA (psw->abProfile, psw->abWindow,
                               gabSettingsWidth,
                               &dWidth);

            SbsIniWriteDwordA (psw->abProfile, psw->abWindow,
                               gabSettingsHeight,
                               &dHeight);
            }
        }
    return;
    }

// -----------------------------------------------------------------

PSBS_WINDOW WINAPI SbsWindowNewA (HWND    hWndParent,
                                  WNDPROC WndProc,
                                  PBYTE   pbWindow,
                                  PBYTE   pbCaption,
                                  PBYTE   pbClass,
                                  PBYTE   pbIcon,
                                  PBYTE   pbMenu,
                                  PVOID   pContext)
    {
    HINSTANCE   hInstance;
    HICON       hIcon;
    HFONT       hFont;
    HBRUSH      hBrush;
    COLORREF    crColor, crBrush;
    DWORD       dColumn, dRow, dWidth, dHeight;
    BYTE        abProfile [MAX_PATH];
    PBYTE       pbWindow1  = SAFEPTR (pbWindow,  gabWindow );
    PBYTE       pbCaption1 = SAFEPTR (pbCaption, gabCaption);
    PSBS_WINDOW psw        = NULL;

    if ((WndProc != NULL) && (pbClass != NULL) && pbClass [0] &&
        ((hInstance = GetModuleHandleA (NULL)) != NULL)       &&
        _fpathA (hInstance, gabIni, abProfile, MAX_PATH)      &&
        ((psw = _mnew (SBS_WINDOW_)) != NULL))
        {
        _tcopybA (psw->abProfile, abProfile );
        _tcopybA (psw->abWindow,  pbWindow1 );
        _tcopybA (psw->abCaption, pbCaption1);
        _tcopybA (psw->abClass,   pbClass   );
        _tcopybA (psw->abMenu,    pbMenu    );

        SbsIniReadFontExA (psw->abProfile, psw->abWindow,
                           gabSettingsFont, &psw->lfA);

        crColor = SbsIniReadColorA (psw->abProfile, psw->abWindow,
                                    gabSettingsColor,
                                    SBS_DEFAULT_FOREGROUND);

        crBrush = SbsIniReadColorA (psw->abProfile, psw->abWindow,
                                    gabSettingsBrush,
                                    SBS_DEFAULT_BACKGROUND);

        dColumn = SbsIniReadDwordA (psw->abProfile, psw->abWindow,
                                    gabSettingsColumn,
                                    MAXDWORD);

        dRow    = SbsIniReadDwordA (psw->abProfile, psw->abWindow,
                                    gabSettingsRow,
                                    MAXDWORD);

        dWidth  = SbsIniReadDwordA (psw->abProfile, psw->abWindow,
                                    gabSettingsWidth,
                                    MAXDWORD);

        dHeight = SbsIniReadDwordA (psw->abProfile, psw->abWindow,
                                    gabSettingsHeight,
                                    MAXDWORD);

        if (dColumn == MAXDWORD)
            {
            dColumn = SbsWindowCenterColumn (&dWidth);
            }
        if (dRow == MAXDWORD)
            {
            dRow = SbsWindowCenterRow (&dHeight);
            }
        if ((pbIcon == NULL) ||
            ((hIcon = LoadIconA (hInstance, pbIcon)) == NULL))
            {
            hIcon = ghiDefault;
            }
        if ((hFont = CreateFontIndirectA (&psw->lfA)) == NULL)
            {
            hFont = ghfDefault;
            }
        if ((hBrush = CreateSolidBrush (crBrush)) == NULL)
            {
            hBrush = ghbDefault;
            }
        psw->fUnicode          = FALSE;
        psw->pContext          = pContext;
        psw->hWndParent        = hWndParent;
        psw->hWnd              = NULL;
        psw->hStatus           = NULL;
        psw->hFont             = hFont;
        psw->hBrush            = hBrush;
        psw->crColor           = crColor;
        psw->crBrush           = crBrush;
        psw->dMode             = OPAQUE;
        psw->dColumn           = dColumn;
        psw->dRow              = dRow;
        psw->dWidth            = dWidth;
        psw->dHeight           = dHeight;

        psw->wcA.style         = CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS;
        psw->wcA.lpfnWndProc   = WndProc;
        psw->wcA.cbClsExtra    = 0;
        psw->wcA.cbWndExtra    = 0;
        psw->wcA.hInstance     = hInstance;
        psw->wcA.hIcon         = hIcon;
        psw->wcA.hCursor       = ghcDefault;
        psw->wcA.hbrBackground = hBrush;
        psw->wcA.lpszMenuName  = SAFESTR (psw->abMenu, NULL);
        psw->wcA.lpszClassName = psw->abClass;
        }
    return psw;
    }

// -----------------------------------------------------------------

PSBS_WINDOW WINAPI SbsWindowNewW (HWND    hWndParent,
                                  WNDPROC WndProc,
                                  PWORD   pwWindow,
                                  PWORD   pwCaption,
                                  PWORD   pwClass,
                                  PWORD   pwIcon,
                                  PWORD   pwMenu,
                                  PVOID   pContext)
    {
    HINSTANCE   hInstance;
    HICON       hIcon;
    HFONT       hFont;
    HBRUSH      hBrush;
    COLORREF    crColor, crBrush;
    DWORD       dColumn, dRow, dWidth, dHeight;
    WORD        awProfile [MAX_PATH];
    PWORD       pwWindow1  = SAFEPTR (pwWindow,  gawWindow );
    PWORD       pwCaption1 = SAFEPTR (pwCaption, gawCaption);
    PSBS_WINDOW psw        = NULL;

    if ((WndProc != NULL) && (pwClass != NULL) && pwClass [0] &&
        ((hInstance = GetModuleHandleW (NULL)) != NULL)       &&
        _fpathW (hInstance, gawIni, awProfile, MAX_PATH)      &&
        ((psw = _mnew (SBS_WINDOW_)) != NULL))
        {
        _tcopybW (psw->awProfile, awProfile );
        _tcopybW (psw->awWindow,  pwWindow1 );
        _tcopybW (psw->awCaption, pwCaption1);
        _tcopybW (psw->awClass,   pwClass   );
        _tcopybW (psw->awMenu,    pwMenu    );

        SbsIniReadFontExW (psw->awProfile, psw->awWindow,
                           gawSettingsFont, &psw->lfW);

        crColor = SbsIniReadColorW (psw->awProfile, psw->awWindow,
                                    gawSettingsColor,
                                    SBS_DEFAULT_FOREGROUND);

        crBrush = SbsIniReadColorW (psw->awProfile, psw->awWindow,
                                    gawSettingsBrush,
                                    SBS_DEFAULT_BACKGROUND);

        dColumn = SbsIniReadDwordW (psw->awProfile, psw->awWindow,
                                    gawSettingsColumn,
                                    MAXDWORD);

        dRow    = SbsIniReadDwordW (psw->awProfile, psw->awWindow,
                                    gawSettingsRow,
                                    MAXDWORD);

        dWidth  = SbsIniReadDwordW (psw->awProfile, psw->awWindow,
                                    gawSettingsWidth,
                                    MAXDWORD);

        dHeight = SbsIniReadDwordW (psw->awProfile, psw->awWindow,
                                    gawSettingsHeight,
                                    MAXDWORD);

        if (dColumn == MAXDWORD)
            {
            dColumn = SbsWindowCenterColumn (&dWidth);
            }
        if (dRow == MAXDWORD)
            {
            dRow = SbsWindowCenterRow (&dHeight);
            }
        if ((pwIcon == NULL) ||
            ((hIcon = LoadIconW (hInstance, pwIcon)) == NULL))
            {
            hIcon = ghiDefault;
            }
        if ((hFont = CreateFontIndirectW (&psw->lfW)) == NULL)
            {
            hFont = ghfDefault;
            }
        if ((hBrush = CreateSolidBrush (crBrush)) == NULL)
            {
            hBrush = ghbDefault;
            }
        psw->fUnicode          = TRUE;
        psw->pContext          = pContext;
        psw->hWndParent        = hWndParent;
        psw->hWnd              = NULL;
        psw->hStatus           = NULL;
        psw->hFont             = hFont;
        psw->hBrush            = hBrush;
        psw->crColor           = crColor;
        psw->crBrush           = crBrush;
        psw->dMode             = OPAQUE;
        psw->dColumn           = dColumn;
        psw->dRow              = dRow;
        psw->dWidth            = dWidth;
        psw->dHeight           = dHeight;

        psw->wcW.style         = CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS;
        psw->wcW.lpfnWndProc   = WndProc;
        psw->wcW.cbClsExtra    = 0;
        psw->wcW.cbWndExtra    = 0;
        psw->wcW.hInstance     = hInstance;
        psw->wcW.hIcon         = hIcon;
        psw->wcW.hCursor       = ghcDefault;
        psw->wcW.hbrBackground = hBrush;
        psw->wcW.lpszMenuName  = SAFESTR (psw->awMenu, NULL);
        psw->wcW.lpszClassName = psw->awClass;
        }
    return psw;
    }

// -----------------------------------------------------------------

HWND WINAPI SbsWindowCreateA (HWND    hWndParent,
                              WNDPROC WndProc,
                              PBYTE   pbWindow,
                              PBYTE   pbCaption,
                              PBYTE   pbClass,
                              PBYTE   pbIcon,
                              PBYTE   pbMenu,
                              PVOID   pContext,
                              DWORD   dStatus,
                              PDWORD  pdStatus)
    {
    PSBS_WINDOW psw = NULL;

    if ((psw = SbsWindowNewA (hWndParent, WndProc,
                              pbWindow, pbCaption, pbClass,
                              pbIcon, pbMenu, pContext))
        != NULL)
        {
        RegisterClassA (&psw->wcA);

        if ((psw->hWnd = CreateWindowA (psw->abClass,
                                        psw->abCaption,
                                        WS_OVERLAPPEDWINDOW,
                                        psw->dColumn, psw->dRow,
                                        psw->dWidth,  psw->dHeight,
                                        psw->hWndParent, NULL,
                                        psw->wcA.hInstance, psw))
            != NULL)
            {
            SbsWindowMoveEx (psw->hWnd, SW_HIDE,
                             &psw->dColumn, &psw->dRow,
                             &psw->dWidth,  &psw->dHeight);

            psw->hStatus = SbsStatusCreateA (psw->hWnd,
                                             IDW_STATUSBAR,
                                             dStatus, pdStatus);
            }
        else
            {
            UnregisterClassA (psw->abClass, psw->wcA.hInstance);
            _mfree (psw);
            }
        }    
    return (psw != NULL ? psw->hWnd : NULL);
    }

// -----------------------------------------------------------------

HWND WINAPI SbsWindowCreateW (HWND    hWndParent,
                              WNDPROC WndProc,
                              PWORD   pwWindow,
                              PWORD   pwCaption,
                              PWORD   pwClass,
                              PWORD   pwIcon,
                              PWORD   pwMenu,
                              PVOID   pContext,
                              DWORD   dStatus,
                              PDWORD  pdStatus)
    {
    PSBS_WINDOW psw = NULL;

    if ((psw = SbsWindowNewW (hWndParent, WndProc,
                              pwWindow, pwCaption, pwClass,
                              pwIcon, pwMenu, pContext))
        != NULL)
        {
        RegisterClassW (&psw->wcW);

        if ((psw->hWnd = CreateWindowW (psw->awClass,
                                        psw->awCaption,
                                        WS_OVERLAPPEDWINDOW,
                                        psw->dColumn, psw->dRow,
                                        psw->dWidth,  psw->dHeight,
                                        psw->hWndParent, NULL,
                                        psw->wcW.hInstance, psw))
            != NULL)
            {
            SbsWindowMoveEx (psw->hWnd, SW_HIDE,
                             &psw->dColumn, &psw->dRow,
                             &psw->dWidth,  &psw->dHeight);

            psw->hStatus = SbsStatusCreateW (psw->hWnd,
                                             IDW_STATUSBAR,
                                             dStatus, pdStatus);
            }
        else
            {
            UnregisterClassW (psw->awClass, psw->wcW.hInstance);
            _mfree (psw);
            }
        }    
    return (psw != NULL ? psw->hWnd : NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDestroy (HWND hWnd)
    {
    PSBS_WINDOW psw;
    BOOL        fUnicode;
    BOOL        fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        fUnicode = IsWindowUnicode (hWnd);

        SbsWindowSave (hWnd);

        if (fUnicode)
            {
            WinHelpW (hWnd, NULL, HELP_QUIT, 0);
            }
        else
            {
            WinHelpA (hWnd, NULL, HELP_QUIT, 0);
            }
        psw = SbsWindowUserRead (hWnd);
        fOk = DestroyWindow (hWnd);

        if (psw != NULL)
            {
            if (fUnicode)
                {
                UnregisterClassW (psw->awClass, psw->wcW.hInstance);
                }
            else
                {
                UnregisterClassA (psw->abClass, psw->wcA.hInstance);
                }
            SbsFontDelete  (psw->hFont);
            SbsBrushDelete (psw->hBrush);

            _mfree (psw);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

INT WINAPI SbsWindowMainA (WNDPROC WndProc,
                           PBYTE   pbWindow,
                           PBYTE   pbCaption,
                           PBYTE   pbClass,
                           PBYTE   pbIcon,
                           PBYTE   pbMenu,
                           PVOID   pContext,
                           DWORD   dMode,
                           UINT    uiMsg,
                           WPARAM  wParam,
                           LPARAM  lParam)
    {
    MSG     m;
    HWND    hWnd;
    LRESULT lr;
    DWORD   dStatus = -1;

    m.wParam = 0;

    if ((hWnd = SbsWindowCreateA (NULL, WndProc,
                                  pbWindow, pbCaption, pbClass,
                                  pbIcon, pbMenu, pContext,
                                  1, &dStatus))
        != NULL)
        {
        ShowWindow   (hWnd, dMode);
        UpdateWindow (hWnd);

        if (uiMsg != WM_NULL)
            {
            PostMessageA (hWnd, uiMsg, wParam, lParam);
            }
        while ((lr = GetMessageA (&m, NULL, 0, 0)) && (lr != -1))
            {
            TranslateMessage (&m);
            DispatchMessageA (&m);
            }
        }
    return m.wParam;
    }

// -----------------------------------------------------------------

INT WINAPI SbsWindowMainW (WNDPROC WndProc,
                           PWORD   pwWindow,
                           PWORD   pwCaption,
                           PWORD   pwClass,
                           PWORD   pwIcon,
                           PWORD   pwMenu,
                           PVOID   pContext,
                           DWORD   dMode,
                           UINT    uiMsg,
                           WPARAM  wParam,
                           LPARAM  lParam)
    {
    MSG     m;
    HWND    hWnd;
    LRESULT lr;
    DWORD   dStatus = -1;

    m.wParam = 0;

    if ((hWnd = SbsWindowCreateW (NULL, WndProc,
                                  pwWindow, pwCaption, pwClass,
                                  pwIcon, pwMenu, pContext,
                                  1, &dStatus))
        != NULL)
        {
        ShowWindow   (hWnd, dMode);
        UpdateWindow (hWnd);

        if (uiMsg != WM_NULL)
            {
            PostMessageW (hWnd, uiMsg, wParam, lParam);
            }
        while ((lr = GetMessageW (&m, NULL, 0, 0)) && (lr != -1))
            {
            TranslateMessage (&m);
            DispatchMessageW (&m);
            }
        }
    return m.wParam;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowIdleA (HWND hWnd)
    {
    MSG  m;
    BOOL fQuit = FALSE;

    while ((!fQuit) && PeekMessageA (&m, hWnd, 0, 0, PM_NOREMOVE))
        {
        if (m.message == WM_QUIT)
            {
            fQuit = TRUE;
            }
        else
            {
            if (PeekMessageA (&m, hWnd, 0, 0, PM_REMOVE))
                {
                TranslateMessage (&m);
                DispatchMessageA (&m);
                }
            }
        }
    return fQuit;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowIdleW (HWND hWnd)
    {
    MSG  m;
    BOOL fQuit = FALSE;

    while ((!fQuit) && PeekMessageW (&m, hWnd, 0, 0, PM_NOREMOVE))
        {
        if (m.message == WM_QUIT)
            {
            fQuit = TRUE;
            }
        else
            {
            if (PeekMessageW (&m, hWnd, 0, 0, PM_REMOVE))
                {
                TranslateMessage (&m);
                DispatchMessageW (&m);
                }
            }
        }
    return fQuit;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowDispatch (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam,
                                  PVOID  pUser,
                                  DWORD  dEvent,
                                  ...)
    {
    PROC   Handler;
    DWORD  i;
    PDWORD pdList = &dEvent + 1;

    i = 0;
    while ((pdList [i] != SBS_DEFAULT_HANDLER) &&
           (pdList [i] != SBS_DEFAULT_RESULT ) &&
           (pdList [i] != dEvent             )) i += 2;

    Handler = (PROC) pdList [i+1];

    return (pdList [i] != SBS_DEFAULT_RESULT
            ? Handler (hWnd, uiMsg, wParam, lParam, pUser)
            : (LRESULT) Handler);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowDispatchEx (HWND   hWnd,
                                    UINT   uiMsg,
                                    WPARAM wParam,
                                    LPARAM lParam,
                                    DWORD  dEvent,
                                    ...)
    {
    PVOID  pUser;
    PROC   Handler;
    DWORD  i;
    PDWORD pdList = &dEvent + 1;

    if (uiMsg == WM_CREATE)
        {
        pUser = ((LPCREATESTRUCT) lParam)->lpCreateParams;
        SbsWindowUserWrite (hWnd, pUser);
        }
    else
        {
        pUser = SbsWindowUserRead (hWnd);
        }
    i = 0;
    while ((pdList [i] != SBS_DEFAULT_HANDLER) &&
           (pdList [i] != SBS_DEFAULT_RESULT ) &&
           (pdList [i] != dEvent             )) i += 2;

    Handler = (PROC) pdList [i+1];

    return (pdList [i] != SBS_DEFAULT_RESULT
            ? Handler (hWnd, uiMsg, wParam, lParam, pUser)
            : (LRESULT) Handler);
    }

// -----------------------------------------------------------------

WNDPROC WINAPI SbsWindowSubclass (HWND    hWnd,
                                  WNDPROC wpNew,
                                  PVOID   pUser)
    {
    BYTE    abClassName [SBS_STRING];
    WORD    awClassName [SBS_STRING];
    WNDPROC wpOld = NULL;

    if (SbsWindowValid (hWnd)
        &&
        (IsWindowUnicode (hWnd)
         ? GetClassNameW  (hWnd, awClassName, SBS_STRING)
         : GetClassNameA  (hWnd, abClassName, SBS_STRING))
        &&
        (wpOld = (WNDPROC) SbsWindowLongRead (hWnd, GWL_WNDPROC)))
        {
        SbsWindowUserWrite (hWnd, pUser);
        SbsWindowLongWrite (hWnd, GWL_WNDPROC, (LONG) wpNew);
        }
    else
        {
        wpOld = NULL;
        }
    return wpOld;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWindowNotify (HWND hWnd,
                                INT  iMenu,
                                HWND hActive)
    {
    LRESULT lr = SbsWindowSend (hWnd, WM_COMMAND,
                                MAKEWPARAM (iMenu, 0),
                                (LPARAM) NULL,
                                -1);

    if (SbsWindowValid (hActive)) SetActiveWindow (hActive);
    return lr;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowRecurse (HWND  hWnd,
                               DWORD dMode)
    {
    DWORD n = 0;

    while (SbsWindowValid (hWnd))
        {
        ShowWindow (hWnd, dMode);
        hWnd = GetWindow (hWnd, GW_OWNER);
        n++;
        }
    return n;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowUpdate (HWND hWnd)
    {
    return SbsWindowValid (hWnd) && UpdateWindow (hWnd);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowSelect (HWND hWnd)
    {
    return SbsWindowValid (hWnd) && SetForegroundWindow (hWnd);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowActivate (HWND hWnd)
    {
    return SbsWindowValid (hWnd) && (SetActiveWindow (hWnd) !=NULL);
    }

// -----------------------------------------------------------------

INT WINAPI SbsWindowControl (LPARAM wParam,
                             LPARAM lParam)
    {
    return SbsWindowControlEx (wParam, lParam, NULL, NULL);
    }

// -----------------------------------------------------------------

INT WINAPI SbsWindowControlEx (LPARAM wParam,
                               LPARAM lParam,
                               PHWND  phItem,
                               PDWORD pdCode)
    {
    if (phItem != NULL) *phItem = (HWND)  lParam;
    if (pdCode != NULL) *pdCode = HIWORD (wParam);
    return (UINT) LOWORD (wParam);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowControlCode (LPARAM wParam,
                                   LPARAM lParam)
    {
    DWORD dCode;

    SbsWindowControlEx (wParam, lParam, NULL, &dCode);
    return dCode;
    }

// -----------------------------------------------------------------

HWND WINAPI SbsWindowControlHandle (LPARAM wParam,
                                    LPARAM lParam)
    {
    HWND hItem;

    SbsWindowControlEx (wParam, lParam, &hItem, NULL);
    return hItem;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowMenuSet (HWND  hWnd,
                              HMENU hMenu)
    {
    HMENU hm;
    BOOL  fOk = FALSE;

    if (hMenu != NULL)
        {
        if ((hm = GetMenu (hWnd)) != NULL) DestroyMenu (hm);
        fOk = SetMenu (hWnd, hMenu);
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowLabelA (PBYTE pbBuffer)
    {
    DWORD i;
    DWORD n = 0;

    if (pbBuffer != NULL)
        {
        for (i = 0; pbBuffer [i] == ' '; i++);

        while (pbBuffer [i])
            {
            if (pbBuffer [i] != '&')
                {
                pbBuffer [n++] = pbBuffer [i];
                }
            else
                {
                if (pbBuffer [i+1])
                    {
                    pbBuffer [n++] = pbBuffer [++i];
                    }
                }
            i++;
            }
        while (n && (pbBuffer [n-1] == ' ')) n--;
        while (n && (pbBuffer [n-1] == '.')) n--;
        while (n && (pbBuffer [n-1] == ' ')) n--;

        pbBuffer [n] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowLabelW (PWORD pwBuffer)
    {
    DWORD i;
    DWORD n = 0;

    if (pwBuffer != NULL)
        {
        for (i = 0; pwBuffer [i] == ' '; i++);

        while (pwBuffer [i])
            {
            if (pwBuffer [i] != '&')
                {
                pwBuffer [n++] = pwBuffer [i];
                }
            else
                {
                if (pwBuffer [i+1])
                    {
                    pwBuffer [n++] = pwBuffer [++i];
                    }
                }
            i++;
            }
        while (n && (pwBuffer [n-1] == ' ')) n--;
        while (n && (pwBuffer [n-1] == '.')) n--;
        while (n && (pwBuffer [n-1] == ' ')) n--;

        pwBuffer [n] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowMenuReadA (HWND  hWnd,
                                 DWORD dItem,
                                 PBYTE pbBuffer,
                                 DWORD dBuffer)
    {
    HMENU hMenu;
    DWORD n = 0;

    if ((pbBuffer != NULL) && dBuffer)
        {
        if ((hMenu = GetMenu (hWnd)) != NULL)
            {
            n = GetMenuStringA (hMenu, dItem, pbBuffer, dBuffer,
                                MF_BYCOMMAND);
            }
        pbBuffer [n] = 0;
        n = SbsWindowLabelA (pbBuffer);
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsWindowMenuReadW (HWND  hWnd,
                                 DWORD dItem,
                                 PWORD pwBuffer,
                                 DWORD dBuffer)
    {
    HMENU hMenu;
    DWORD n = 0;

    if ((pwBuffer != NULL) && dBuffer)
        {
        if ((hMenu = GetMenu (hWnd)) != NULL)
            {
            n = GetMenuStringW (hMenu, dItem, pwBuffer, dBuffer,
                                MF_BYCOMMAND);
            }
        pwBuffer [n] = 0;
        n = SbsWindowLabelW (pwBuffer);
        }
    return n;
    }

// -----------------------------------------------------------------

PSBS_WINDOW WINAPI SbsWindowFontA (HWND      hWnd,
                                   PLOGFONTA plfA,
                                   PCOLORREF pcrColor,
                                   PCOLORREF pcrBrush)
    {
    PSBS_WINDOW psw = SbsWindowUserRead (hWnd);

    if (psw != NULL)
        {
        if (plfA != NULL)
            {
            SbsIniReadFontExA (psw->abProfile,
                               psw->abWindow,
                               gabSettingsFont,
                               plfA);
            }
        if (pcrColor != NULL)
            {
            *pcrColor = SbsIniReadColorA (psw->abProfile,
                                          psw->abWindow,
                                          gabSettingsColor,
                                          SBS_DEFAULT_FOREGROUND);
            }
        if (pcrBrush != NULL)
            {
            *pcrBrush = SbsIniReadColorA (psw->abProfile,
                                          psw->abWindow,
                                          gabSettingsBrush,
                                          SBS_DEFAULT_BACKGROUND);
            }
        }
    else
        {
        SbsFontSystemA (plfA, FALSE);
        if (pcrColor != NULL) *pcrColor = SBS_DEFAULT_FOREGROUND;
        if (pcrBrush != NULL) *pcrBrush = SBS_DEFAULT_BACKGROUND;
        }
    return psw;
    }

// -----------------------------------------------------------------

PSBS_WINDOW WINAPI SbsWindowFontW (HWND      hWnd,
                                   PLOGFONTW plfW,
                                   PCOLORREF pcrColor,
                                   PCOLORREF pcrBrush)
    {
    PSBS_WINDOW psw = SbsWindowUserRead (hWnd);

    if (psw != NULL)
        {
        if (plfW != NULL)
            {
            SbsIniReadFontExW (psw->awProfile,
                               psw->awWindow,
                               gawSettingsFont,
                               plfW);
            }
        if (pcrColor != NULL)
            {
            *pcrColor = SbsIniReadColorW (psw->awProfile,
                                          psw->awWindow,
                                          gawSettingsColor,
                                          SBS_DEFAULT_FOREGROUND);
            }
        if (pcrBrush != NULL)
            {
            *pcrBrush = SbsIniReadColorW (psw->awProfile,
                                          psw->awWindow,
                                          gawSettingsBrush,
                                          SBS_DEFAULT_BACKGROUND);
            }
        }
    else
        {
        SbsFontSystemW (plfW, FALSE);
        if (pcrColor != NULL) *pcrColor = SBS_DEFAULT_FOREGROUND;
        if (pcrBrush != NULL) *pcrBrush = SBS_DEFAULT_BACKGROUND;
        }
    return psw;
    }

// -----------------------------------------------------------------

INT WINAPI SbsWindowDrawText (HWND  hWnd,
                              HDC   hDC,
                              PRECT pr,
                              PVOID pText,
                              DWORD dText,
                              DWORD dFormat)
    {
    PSBS_WINDOW psw;
    RECT        r;
    HFONT       hFont;
    COLORREF    crColor, crBrush;
    INT         iMode;
    INT         iHeight = 0;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        if (pr != NULL)
            {
            r = *pr;
            }
        else
            {
            GetClientRect (hWnd, &r);
            r.bottom -= SbsStatusHeight (hWnd);
            }
        hFont   = SelectObject (hDC, psw->hFont  );
        crColor = SetTextColor (hDC, psw->crColor);
        crBrush = SetBkColor   (hDC, psw->crBrush);
        iMode   = SetBkMode    (hDC, psw->dMode  );

        iHeight = (psw->fUnicode
                   ? DrawTextW (hDC, pText, dText, &r, dFormat)
                   : DrawTextA (hDC, pText, dText, &r, dFormat));

        if (hFont   != NULL       ) SelectObject (hDC, hFont  );
        if (crColor != CLR_INVALID) SetTextColor (hDC, crColor);
        if (crBrush != CLR_INVALID) SetBkColor   (hDC, crBrush);
        if (iMode                 ) SetBkMode    (hDC, iMode  );
        }
    return iHeight;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDataRead (HWND  hWnd,
                               PVOID pKey,
                               PVOID pData,
                               DWORD dType)
    {
    PSBS_WINDOW psw;
    BOOL        fOk = FALSE;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        fOk = (psw->fUnicode

               ? SbsIniReadDataW (psw->awProfile, psw->awWindow,
                                  pKey, pData, dType)

               : SbsIniReadDataA (psw->abProfile, psw->abWindow,
                                  pKey, pData, dType));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDataReadEx (HWND hWnd,
                                 ...)
    {
    DWORD          i;
    PSBS_WINDOW    psw;
    PSBS_PARAMETER psp = (PSBS_PARAMETER) (&hWnd + 1);
    BOOL           fOk = FALSE;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        fOk = TRUE;

        if (psw->fUnicode)
            {
            for (i = 0; psp [i].pwKey != NULL; i++)
                {
                if (!SbsIniReadDataW (psw->awProfile,
                                      psw->awWindow,
                                      psp [i].pwKey,
                                      psp [i].pData,
                                      psp [i].dType))
                    {
                    fOk = FALSE;
                    }
                }
            }
        else
            {
            for (i = 0; psp [i].pbKey != NULL; i++)
                {
                if (!SbsIniReadDataA (psw->abProfile,
                                      psw->abWindow,
                                      psp [i].pbKey,
                                      psp [i].pData,
                                      psp [i].dType))
                    {
                    fOk = FALSE;
                    }
                }
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDataWrite (HWND  hWnd,
                                PVOID pKey,
                                PVOID pData,
                                DWORD dType)
    {
    PSBS_WINDOW psw;
    BOOL        fOk = FALSE;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        fOk = (psw->fUnicode

               ? SbsIniWriteDataW (psw->awProfile, psw->awWindow,
                                   pKey, pData, dType)

               : SbsIniWriteDataA (psw->abProfile, psw->abWindow,
                                   pKey, pData, dType));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsWindowDataWriteEx (HWND hWnd,
                                  ...)
    {
    DWORD          i;
    PSBS_WINDOW    psw;
    PSBS_PARAMETER psp = (PSBS_PARAMETER) (&hWnd + 1);
    BOOL           fOk = FALSE;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        fOk = TRUE;

        if (psw->fUnicode)
            {
            for (i = 0; psp [i].pwKey != NULL; i++)
                {
                if (!SbsIniWriteDataW (psw->awProfile,
                                       psw->awWindow,
                                       psp [i].pwKey,
                                       psp [i].pData,
                                       psp [i].dType))
                    {
                    fOk = FALSE;
                    }
                }
            }
        else
            {
            for (i = 0; psp [i].pbKey != NULL; i++)
                {
                if (!SbsIniWriteDataA (psw->abProfile,
                                       psw->abWindow,
                                       psp [i].pbKey,
                                       psp [i].pData,
                                       psp [i].dType))
                    {
                    fOk = FALSE;
                    }
                }
            }
        }
    return fOk;
    }

// =================================================================
// DIALOG MANAGEMENT
// =================================================================

LRESULT WINAPI SbsDialogResult (HWND    hWnd,
                                UINT    uiMsg,
                                LRESULT lr)
    {
    LRESULT lr1 = lr;

    if ((uiMsg != WM_CHARTOITEM       ) &&
        (uiMsg != WM_COMPAREITEM      ) &&
        (uiMsg != WM_CTLCOLORBTN      ) &&
        (uiMsg != WM_CTLCOLORDLG      ) &&
        (uiMsg != WM_CTLCOLOREDIT     ) &&
        (uiMsg != WM_CTLCOLORLISTBOX  ) &&
        (uiMsg != WM_CTLCOLORSCROLLBAR) &&
        (uiMsg != WM_CTLCOLORSTATIC   ) &&
        (uiMsg != WM_INITDIALOG       ) &&
        (uiMsg != WM_QUERYDRAGICON    ) &&
        (uiMsg != WM_VKEYTOITEM       ))
        {
        if (SbsWindowValid (hWnd))
            {
            (IsWindowUnicode (hWnd)
             ? SetWindowLongW
             : SetWindowLongA) (hWnd, DWL_MSGRESULT, lr);
            }
        lr1 = TRUE;
        }
    return lr1;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogHandler (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser,
                                 PROC   Handler)
    {
    LRESULT lr = FALSE;

    if (Handler != NULL)
        {
        lr = SbsDialogResult (hWnd, uiMsg,
                              Handler (hWnd, uiMsg, wParam, lParam,
                                       pUser));
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogDispatch (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam,
                                  PVOID  pUser,
                                  DWORD  dEvent,
                                  ...)
    {
    PROC   Handler;
    DWORD  i;
    PDWORD pdList = &dEvent + 1;

    i = 0;
    while ((pdList [i] != SBS_DEFAULT_HANDLER) &&
           (pdList [i] != SBS_DEFAULT_RESULT ) &&
           (pdList [i] != dEvent             )) i += 2;

    Handler = (PROC) pdList [i+1];

    return (pdList [i] != SBS_DEFAULT_RESULT
            ? SbsDialogHandler (hWnd, uiMsg, wParam, lParam,
                                pUser, Handler)
            : (LRESULT) Handler);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogDispatchEx (HWND   hWnd,
                                    UINT   uiMsg,
                                    WPARAM wParam,
                                    LPARAM lParam,
                                    DWORD  dEvent,
                                    ...)
    {
    PVOID  pUser;
    PROC   Handler;
    DWORD  i;
    PDWORD pdList = &dEvent + 1;

    if (uiMsg == WM_INITDIALOG)
        {
        pUser = (PVOID) lParam;
        SbsWindowLongWrite (hWnd, DWL_USER, (LONG) pUser);
        }
    else
        {
        pUser = (PVOID) SbsWindowLongRead (hWnd, DWL_USER);
        }
    i = 0;
    while ((pdList [i] != SBS_DEFAULT_HANDLER) &&
           (pdList [i] != SBS_DEFAULT_RESULT ) &&
           (pdList [i] != dEvent             )) i += 2;

    Handler = (PROC) pdList [i+1];

    return (pdList [i] != SBS_DEFAULT_RESULT
            ? SbsDialogHandler (hWnd, uiMsg, wParam, lParam,
                                pUser, Handler)
            : (LRESULT) Handler);
    }

// -----------------------------------------------------------------

INT WINAPI SbsDialogFocus (HWND hWnd)
    {
    HWND hItem = GetFocus ();
    INT  iItem = 0;

    if (SbsWindowValid (hWnd))
        {
        iItem = (hItem != hWnd ? GetDlgCtrlID (hItem) : -1);
        }
    return iItem;
    }

// -----------------------------------------------------------------

WNDPROC WINAPI SbsDialogSubclass (HWND    hWnd,
                                  INT     iItem,
                                  WNDPROC wpNew,
                                  PVOID   pUser)
    {
    HWND    hItem;
    WNDPROC wpOld = NULL;

    if ((iItem != -1) &&
        ((hItem = GetDlgItem (hWnd, iItem)) != NULL))
        {
        wpOld = SbsWindowSubclass (hItem, wpNew, pUser);
        }
    return wpOld;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogNotify (HWND  hWnd,
                                INT   iItem,
                                DWORD dCode)
    {
    HWND    hItem;
    LRESULT lr = -1;

    if ((iItem != -1) &&
        ((hItem = GetDlgItem (hWnd, iItem)) != NULL))
        {
        lr = SbsWindowSend (hWnd, WM_COMMAND,
                            MAKEWPARAM (iItem, dCode),
                            (LPARAM) hItem,
                            lr);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogSelect (HWND hWnd,
                                INT  iItem)
    {
    HWND    hItem;
    LRESULT lr = -1;

    if ((iItem != -1) &&
        ((hItem = GetDlgItem (hWnd, iItem)) != NULL))
        {
        lr = SbsWindowSend (hWnd, WM_NEXTDLGCTL,
                            (WPARAM) hItem,
                            MAKELPARAM (TRUE, 0),
                            lr);
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsDialogSkip (HWND hWnd,
                              BOOL fBack)
    {
    return SbsWindowSend (hWnd, WM_NEXTDLGCTL,
                          (WPARAM) (fBack != FALSE),
                          MAKELPARAM (FALSE, 0),
                          -1);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogChecked (HWND hWnd,
                              INT  iItem)
    {
    return IsDlgButtonChecked (hWnd, iItem) == BST_CHECKED;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogCheck (HWND hWnd,
                            INT  iItem,
                            BOOL fChecked)
    {
    return CheckDlgButton (hWnd, iItem, (fChecked ? BST_CHECKED
                                                  : BST_UNCHECKED));
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogToggle (HWND hWnd,
                             INT  iItem)
    {
    BOOL fChecked = !SbsDialogChecked (hWnd, iItem);

    SbsDialogCheck (hWnd, iItem, fChecked);
    return fChecked;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogEnable (HWND hWnd,
                             INT  iItem,
                             BOOL fEnable)
    {
    HWND hItem = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    BOOL fOk   = FALSE;

    if (SbsWindowValid (hItem))
        {
        EnableWindow (hItem, (fEnable ? TRUE : FALSE));
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogEnabled (HWND hWnd,
                              INT  iItem)
    {
    HWND hItem = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    BOOL fOk   = FALSE;

    if (SbsWindowValid (hItem))
        {
        fOk = IsWindowEnabled (hItem);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogShow (HWND hWnd,
                           INT  iItem,
                           BOOL fShow)
    {
    HWND hItem = GetDlgItem (hWnd, iItem);
    BOOL fOk   = FALSE;

    if (SbsWindowValid (hItem))
        {
        ShowWindow (hItem, (fShow ? SW_SHOW : SW_HIDE));
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogSwitch (HWND hWnd,
                             INT  iHide,
                             INT  iShow)
    {
    BOOL fOk = TRUE;

    if (!SbsDialogShow (hWnd, iHide, FALSE)) fOk = FALSE;
    if (!SbsDialogShow (hWnd, iShow, TRUE )) fOk = FALSE;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogRedraw (HWND hWnd,
                             INT  iItem,
                             BOOL fErase,
                             BOOL fUpdate)
    {
    HWND hItem = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    BOOL fOk   = FALSE;

    if (SbsWindowValid (hItem))
        {
        fOk = TRUE;

        if (!InvalidateRect (hItem, NULL, fErase)) fOk = FALSE;
        if (fUpdate && (!UpdateWindow (hItem)))    fOk = FALSE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsDialogFont (HWND  hWnd,
                            INT   iItem,
                            HFONT hFont)
    {
    HDC   hDC;
    HWND  hItem  = (iItem != -1 ? GetDlgItem (hWnd, iItem) : NULL);
    HFONT hFont1 = SAFEPTR (hFont, ghfDefault);

    if (SbsWindowValid (hItem))
        {
        if ((hDC = GetDC (hItem)) != NULL)
            {
            if (IsWindowUnicode (hItem))
                {
                SendMessageW (hItem, WM_SETFONT,
                              (WPARAM) hFont1, 0);

                SendMessageW (hItem, WM_ERASEBKGND,
                              (WPARAM) hDC, 0);
                }
            else
                {
                SendMessageA (hItem, WM_SETFONT,
                              (WPARAM) hFont1, 0);

                SendMessageA (hItem, WM_ERASEBKGND,
                              (WPARAM) hDC, 0);
                }
            ReleaseDC (hItem, hDC);
            }
        }
    return hFont1;
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsDialogFontEx (HWND  hWnd,
                              INT   iItem,
                              HFONT hFont,
                              PRECT prClient)
    {
    RECT     rClient;
    LONG     lX1, lX2, lY1, lY2;
    LOGFONTA lfA;
    LOGFONTW lfW;
    HFONT    hFont1 = NULL;

    if (SbsWindowValid (hWnd) && (prClient != NULL) &&
        GetClientRect  (hWnd, &rClient))
        {
        lX1 = prClient->right  - prClient->left;
        lY1 = prClient->bottom - prClient->top;

        lX2 = rClient.right    - rClient.left;
        lY2 = rClient.bottom   - rClient.top;

        if (IsWindowUnicode (hWnd))
            {
            if (GetObjectW (SAFEPTR (hFont, ghfDefault),
                            LOGFONTW_, &lfW))
                {
                lfW.lfWidth  = (lfW.lfWidth  * lX2) / lX1;
                lfW.lfHeight = (lfW.lfHeight * lY2) / lY1;

                hFont1 = CreateFontIndirectW (&lfW);
                }
            }
        else
            {
            if (GetObjectA (SAFEPTR (hFont, ghfDefault),
                            LOGFONTA_, &lfA))
                {
                lfA.lfWidth  = (lfA.lfWidth  * lX2) / lX1;
                lfA.lfHeight = (lfA.lfHeight * lY2) / lY1;

                hFont1 = CreateFontIndirectA (&lfA);
                }
            }
        if (hFont1 != NULL) SbsDialogFont (hWnd, iItem, hFont1);
        }
    return hFont1;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogRectangle (HWND  hWnd,
                                INT   iItem,
                                PRECT pr)
    {
    WINDOWPLACEMENT wp;
    HWND            hItem;
    RECT            r   = {0, 0, 0, 0};
    BOOL            fOk = FALSE;

    if (iItem != -1)
        {
        hItem     = GetDlgItem (hWnd, iItem);
        wp.length = WINDOWPLACEMENT_;

        fOk = SbsWindowValid     (hItem) &&
              GetWindowPlacement (hItem, &wp);
        }
    else
        {
        fOk = SbsWindowValid (hWnd) &&
              GetClientRect  (hWnd, &wp.rcNormalPosition);
        }
    if (fOk) r = wp.rcNormalPosition;

    if (pr != NULL) *pr = r;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogMove (HWND  hWnd,
                           INT   iItem,
                           PRECT pr)
    {
    HWND hItem = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    RECT r     = {0, 0, 0, 0};
    BOOL fOk   = FALSE;

    if (pr != NULL) r = *pr;

    return SbsWindowValid (hItem) &&
           MoveWindow     (hItem, SBS_RECT (r), FALSE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogZoom (HWND  hWnd,
                           INT   iItem,
                           PRECT prItem,
                           PRECT prClient)
    {
    RECT rClient, rItem;
    LONG lX1, lX2, lY1, lY2;
    HWND hItem = (iItem != -1 ? GetDlgItem (hWnd, iItem) : NULL);
    BOOL fOk   = FALSE;

    if (SbsWindowValid (hWnd ) && (prClient != NULL) &&
        SbsWindowValid (hItem) && (prItem   != NULL) &&
        GetClientRect  (hWnd, &rClient))
        {
        rItem = *prItem;

        lX1 = prClient->right  - prClient->left;
        lY1 = prClient->bottom - prClient->top;

        lX2 = rClient.right    - rClient.left;
        lY2 = rClient.bottom   - rClient.top;

        rItem.left   = (rItem.left   * lX2) / lX1;
        rItem.right  = (rItem.right  * lX2) / lX1;
        rItem.top    = (rItem.top    * lY2) / lY1;
        rItem.bottom = (rItem.bottom * lY2) / lY1;

        fOk = MoveWindow (hItem, SBS_RECT (rItem), TRUE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PSBS_DIALOG WINAPI SbsDialogZoomOpen (HWND  hWnd,
                                      HFONT hFont1,
                                      HFONT hFont2,
                                      HFONT hFont3,
                                      HFONT hFont4,
                                      ...)
    {
    RECT            r;
    DWORD           iItem, iFont, i;
    HFONT           hf;
    PSBS_DIALOG_ARG psda = (PSBS_DIALOG_ARG) (&hFont4 + 1);
    PSBS_DIALOG     psd  = NULL;

    if (SbsWindowValid (hWnd) && GetClientRect (hWnd, &r))
        {
        for (i = 0; psda [i].iItem != -1; i++);

        if ((psd = _mnew (SBS_DIALOG__ (i))) != NULL)
            {
            psd->hWnd           = hWnd;
            psd->ahfNormal  [0] = hFont1;
            psd->ahfNormal  [1] = hFont2;
            psd->ahfNormal  [2] = hFont3;
            psd->ahfNormal  [3] = hFont4;
            psd->ahfCurrent [0] = hFont1;
            psd->ahfCurrent [1] = hFont2;
            psd->ahfCurrent [2] = hFont3;
            psd->ahfCurrent [3] = hFont4;
            psd->rNormal        = r;
            psd->rCurrent       = r;
            psd->dItems         = 0;

            for (i = 0; (iItem = psda [i].iItem) != -1; i++)
                {
                iFont = psda [i].iFont;

                if (SbsDialogRectangle (hWnd, iItem, &r))
                    {
                    psd->aItems [psd->dItems].iItem    = iItem;
                    psd->aItems [psd->dItems].iFont    = iFont;
                    psd->aItems [psd->dItems].rNormal  = r;
                    psd->aItems [psd->dItems].rCurrent = r;

                    if (iFont && (iFont <= SBS_DIALOG_FONTS) &&
                        ((hf = psd->ahfCurrent [iFont-1]) != NULL))
                        {
                        SbsDialogFont (hWnd, iItem, hf);
                        }
                    psd->dItems++;
                    }
                }
            }
        }
    return psd;
    }

// -----------------------------------------------------------------

PSBS_DIALOG WINAPI SbsDialogZoomClose (PSBS_DIALOG psd)
    {
    DWORD i;

    if (psd != NULL)
        {
        for (i = 0; i < SBS_DIALOG_FONTS; i++)
            {
            if (psd->ahfCurrent [i] != psd->ahfNormal [i])
                {
                SbsFontDelete (psd->ahfCurrent [i]);
                }
            }
        _mfree (psd);
        }
    return NULL;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogZoomUpdate (PSBS_DIALOG psd)
    {
    RECT     r;
    LOGFONTA lfA;
    LOGFONTW lfW;
    HFONT    hf;
    LONG     lX1, lX2, lY1, lY2;
    DWORD    iItem, iFont, i;
    BOOL     fUnicode;
    BOOL     fOk = FALSE;

    if ((psd != NULL) && SbsWindowValid (psd->hWnd)
                      && GetClientRect  (psd->hWnd, &r))
        {
        fUnicode = IsWindowUnicode (psd->hWnd);
        fOk      = TRUE;

        psd->rCurrent = r;

        lX1 = psd->rNormal .right  - psd->rNormal .left;
        lY1 = psd->rNormal .bottom - psd->rNormal .top;

        lX2 = psd->rCurrent.right  - psd->rCurrent.left;
        lY2 = psd->rCurrent.bottom - psd->rCurrent.top;

        for (i = 0; i < SBS_DIALOG_FONTS; i++)
            {
            if (psd->ahfNormal [i] != NULL)
                {
                hf = NULL;

                if (fUnicode)
                    {
                    if (GetObjectW (psd->ahfNormal [i],
                                    LOGFONTW_, &lfW))
                        {
                        lfW.lfWidth  = (lfW.lfWidth  * lX2) / lX1;
                        lfW.lfHeight = (lfW.lfHeight * lY2) / lY1;

                        hf = CreateFontIndirectW (&lfW);
                        }
                    }
                else
                    {
                    if (GetObjectA (psd->ahfNormal [i],
                                    LOGFONTA_, &lfA))
                        {
                        lfA.lfWidth  = (lfA.lfWidth  * lX2) / lX1;
                        lfA.lfHeight = (lfA.lfHeight * lY2) / lY1;

                        hf = CreateFontIndirectA (&lfA);
                        }
                    }
                if (psd->ahfCurrent [i] != psd->ahfNormal [i])
                    {
                    SbsFontDelete (psd->ahfCurrent [i]);
                    }
                if (hf != NULL)
                    {
                    psd->ahfCurrent [i] = hf;
                    }
                else
                    {
                    psd->ahfCurrent [i] = psd->ahfNormal [i];
                    fOk = FALSE;
                    }
                }
            }
        for (i = 0; i < psd->dItems; i++)
            {
            iItem = psd->aItems [i].iItem;
            iFont = psd->aItems [i].iFont;
            r     = psd->aItems [i].rNormal;

            r.left   = (r.left   * lX2) / lX1;
            r.right  = (r.right  * lX2) / lX1;
            r.top    = (r.top    * lY2) / lY1;
            r.bottom = (r.bottom * lY2) / lY1;

            psd->aItems [i].rCurrent = r;

            if (!SbsDialogMove (psd->hWnd, iItem, &r))
                {
                fOk = FALSE;
                }
            if (iFont && (iFont <= SBS_DIALOG_FONTS) &&
                ((hf = psd->ahfCurrent [iFont-1]) != NULL) &&
                (!SbsDialogFont (psd->hWnd, iItem, hf)))
                {
                fOk = FALSE;
                }
            }
        InvalidateRect (psd->hWnd, NULL, FALSE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsDialogReadA (HWND hWnd,
                             INT  iItem,
                             BOOL fTrim)
    {
    DWORD i, j, n;
    HWND  hItem  = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    PBYTE pbData = NULL;

    if (SbsWindowValid (hItem))
        {
        n = SendMessageA (hItem, WM_GETTEXTLENGTH, 0, 0) + 1;

        if ((pbData = _tnewA (n)) != NULL)
            {
            n = SendMessageA (hItem, WM_GETTEXT,
                              n, (LPARAM) pbData);
            if (fTrim)
                {
                i = 0;
                while ((i < n) && (pbData [i  ] == ' ')) i++;
                while ((n > i) && (pbData [n-1] == ' ')) n--;

                if (i)
                    {
                    j = n;
                    n = 0;
                    while (i < j) pbData [n++] = pbData [i++];
                    }
                }
            pbData [n] = 0;
            }
        }
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsDialogReadW (HWND hWnd,
                             INT  iItem,
                             BOOL fTrim)
    {
    DWORD i, j, n;
    HWND  hItem  = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    PWORD pwData = NULL;

    if (SbsWindowValid (hItem))
        {
        n = SendMessageW (hItem, WM_GETTEXTLENGTH, 0, 0) + 1;

        if ((pwData = _tnewW (n)) != NULL)
            {
            n = SendMessageW (hItem, WM_GETTEXT,
                              n, (LPARAM) pwData);
            if (fTrim)
                {
                i = 0;
                while ((i < n) && (pwData [i  ] == ' ')) i++;
                while ((n > i) && (pwData [n-1] == ' ')) n--;

                if (i)
                    {
                    j = n;
                    n = 0;
                    while (i < j) pwData [n++] = pwData [i++];
                    }
                }
            pwData [n] = 0;
            }
        }
    return pwData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsDialogReadEx (HWND hWnd,
                              INT  iItem,
                              BOOL fTrim)
    {
    PVOID pData = NULL;

    if (SbsWindowValid (hWnd))
        {
        pData = (IsWindowUnicode (hWnd)
                 ? (PVOID) SbsDialogReadW (hWnd, iItem, fTrim)
                 : (PVOID) SbsDialogReadA (hWnd, iItem, fTrim));
        }
    return pData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogWriteA (HWND  hWnd,
                             INT   iItem,
                             PBYTE pbText,
                             PBYTE pbDefault)
    {
    HWND  hItem  = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    PBYTE pbData = SAFEPTR (pbText, pbDefault);
    BOOL  fOk    = FALSE;

    if ((pbData != NULL) && SbsWindowValid (hItem))
        {
        fOk = (SendMessageA (hItem, WM_SETTEXT,
                             0, (LPARAM) pbData)
               == TRUE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogWriteW (HWND  hWnd,
                             INT   iItem,
                             PWORD pwText,
                             PWORD pwDefault)
    {
    HWND  hItem  = (iItem != -1 ? GetDlgItem (hWnd, iItem) : hWnd);
    PWORD pwData = SAFEPTR (pwText, pwDefault);
    BOOL  fOk    = FALSE;

    if ((pwData != NULL) && SbsWindowValid (hItem))
        {
        fOk = (SendMessageW (hItem, WM_SETTEXT,
                             0, (LPARAM) pwData)
               == TRUE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogWriteEx (HWND  hWnd,
                              INT   iItem,
                              PVOID pText,
                              PVOID pDefault)
    {
    BOOL fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        fOk = (IsWindowUnicode (hWnd)
               ? SbsDialogWriteW (hWnd, iItem, pText, pDefault)
               : SbsDialogWriteA (hWnd, iItem, pText, pDefault));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogWriteText (HWND      hWnd,
                                INT       iItem,
                                PSBS_TEXT pstText,
                                PSBS_TEXT pstDefault)
    {
    PBYTE pbText, pbDefault;
    PWORD pwText, pwDefault;
    BOOL  fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        if (IsWindowUnicode (hWnd))
            {
            pwText    = SAFEMEMBER (pstText,    pwData, NULL);
            pwDefault = SAFEMEMBER (pstDefault, pwData, NULL);

            fOk = SbsDialogWriteW (hWnd, iItem, pwText, pwDefault);
            }
        else
            {
            pbText    = SAFEMEMBER (pstText,    pbData, NULL);
            pbDefault = SAFEMEMBER (pstDefault, pbData, NULL);

            fOk = SbsDialogWriteA (hWnd, iItem, pbText, pbDefault);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogClear (HWND hWnd,
                            INT  iItem)
    {
    BOOL fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        fOk = (IsWindowUnicode (hWnd)
               ? SbsDialogWriteW (hWnd, iItem, gawNull, gawNull)
               : SbsDialogWriteA (hWnd, iItem, gabNull, gabNull));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogTest (HWND hWnd,
                           INT  iItem,
                           BOOL fTrim)
    {
    PVOID pData = SbsDialogReadEx (hWnd, iItem, fTrim);
    BOOL  fOk   = FALSE;

    if (pData != NULL)
        {
        fOk = (IsWindowUnicode (hWnd)
               ? *(PWORD) pData != 0
               : *(PBYTE) pData != 0);

        _mfree (pData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsDialogLabelA (HWND hWnd,
                              INT  iItem)
    {
    PBYTE pbData = SbsDialogReadA (hWnd, iItem, FALSE);

    SbsWindowLabelA (pbData);
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsDialogLabelW (HWND hWnd,
                              INT  iItem)
    {
    PWORD pwData = SbsDialogReadW (hWnd, iItem, FALSE);

    SbsWindowLabelW (pwData);
    return pwData;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsDialogLabelEx (HWND hWnd,
                               INT  iItem)
    {
    PVOID pData = NULL;

    if (SbsWindowValid (hWnd))
        {
        pData = (IsWindowUnicode (hWnd)
                 ? (PVOID) SbsDialogLabelW (hWnd, iItem)
                 : (PVOID) SbsDialogLabelA (hWnd, iItem));
        }
    return pData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogCopyA (HWND  hWnd,
                            INT   iFrom,
                            INT   iTo,
                            BOOL  fTrim,
                            PBYTE pbError)
    {
    PBYTE pbData;
    BOOL  fOk = FALSE;

    if ((pbData = SbsDialogReadA (hWnd, iFrom, fTrim)) != NULL)
        {
        fOk = SbsDialogWriteA (hWnd, iTo, pbData, pbError);
        _mfree (pbData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogCopyW (HWND  hWnd,
                            INT   iFrom,
                            INT   iTo,
                            BOOL  fTrim,
                            PWORD pwError)
    {
    PWORD pwData;
    BOOL  fOk = FALSE;

    if ((pwData = SbsDialogReadW (hWnd, iFrom, fTrim)) != NULL)
        {
        fOk = SbsDialogWriteW (hWnd, iTo, pwData, pwError);
        _mfree (pwData);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogCopyEx (HWND  hWnd,
                             INT   iFrom,
                             INT   iTo,
                             BOOL  fTrim,
                             PVOID pError)
    {
    BOOL fOk = FALSE;

    if (SbsWindowValid (hWnd))
        {
        fOk = (IsWindowUnicode (hWnd)
               ? SbsDialogCopyW (hWnd, iFrom, iTo, fTrim, pError)
               : SbsDialogCopyA (hWnd, iFrom, iTo, fTrim, pError));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogTime (HWND        hWnd,
                           INT         iItem,
                           PSYSTEMTIME pst,
                           BOOL        fGerman)
    {
    BOOL fOk = FALSE;

    if ((pst != NULL) && SbsWindowValid (hWnd))
        {
        fOk = (IsWindowUnicode (hWnd)

            ? _dprintfW (hWnd, iItem,
                         (fGerman ? gawTimeDE   : gawTimeUS),
                         (fGerman ? pst->wDay   : pst->wMonth),
                         (fGerman ? pst->wMonth : pst->wDay),
                         pst->wYear,
                         pst->wHour,
                         pst->wMinute,
                         pst->wSecond)

            : _dprintfA (hWnd, iItem,
                         (fGerman ? gabTimeDE   : gabTimeUS),
                         (fGerman ? pst->wDay   : pst->wMonth),
                         (fGerman ? pst->wMonth : pst->wDay),
                         pst->wYear,
                         pst->wHour,
                         pst->wMinute,
                         pst->wSecond));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogTimeUtc (HWND      hWnd,
                              INT       iItem,
                              PFILETIME pft,
                              BOOL      fGerman)
    {
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToSystemTime (pft, &st))
        {
        pst = &st;
        }
    return SbsDialogTime (hWnd, iItem, pst, fGerman);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDialogTimeLocal (HWND      hWnd,
                                INT       iItem,
                                PFILETIME pft,
                                BOOL      fGerman)
    {
    FILETIME    ft;
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToLocalFileTime (pft, &ft)
                      && FileTimeToSystemTime    (&ft, &st))
        {
        pst = &st;
        }
    return SbsDialogTime (hWnd, iItem, pst, fGerman);
    }

// =================================================================
// LIST BOX MANAGEMENT
// =================================================================

PSBS_LIST WINAPI SbsListCreate (HWND hWnd,
                                INT  iItem,
                                BOOL fCombo)
    {
    return SbsListCreateEx (hWnd, iItem, fCombo, TRUE, NULL);
    }

// -----------------------------------------------------------------

PSBS_LIST WINAPI SbsListCreateEx (HWND    hWnd,
                                  INT     iItem,
                                  BOOL    fCombo,
                                  BOOL    fClear,
                                  HCURSOR hCursor)
    {
    HWND      hItem;
    PSBS_LIST psl = NULL;

    if (((hItem = GetDlgItem (hWnd, iItem)) != NULL) &&
        ((psl   = _mnew (SBS_LIST_))        != NULL))
        {
        psl->fUnicode  = IsWindowUnicode (hWnd);
        psl->fOk       = TRUE;
        psl->dStatus   = SBS_LIST_OK;
        psl->dCount    = 0;
        psl->iItem     = iItem;
        psl->hItem     = hItem;
        psl->hWnd      = hWnd;
        psl->hcWait    = SAFEPTR (hCursor, ghcWait);
        psl->hcDefault = ghcDefault;

        if (psl->fCombo = fCombo)
            {
            psl->dError                       = CB_ERR;
            psl->dErrorSpace                  = CB_ERRSPACE;

            psl->auiMsg [SBS_ADDFILE        ] = WM_NULL;
            psl->auiMsg [SBS_ADDSTRING      ] = CB_ADDSTRING;
            psl->auiMsg [SBS_DELETESTRING   ] = CB_DELETESTRING;
            psl->auiMsg [SBS_DIR            ] = CB_DIR;
            psl->auiMsg [SBS_FINDSTRING     ] = CB_FINDSTRING;
            psl->auiMsg [SBS_FINDSTRINGEXACT] = CB_FINDSTRINGEXACT;
            psl->auiMsg [SBS_GETCOUNT       ] = CB_GETCOUNT;
            psl->auiMsg [SBS_GETCURSEL      ] = CB_GETCURSEL;
            psl->auiMsg [SBS_GETDROPPEDSTATE] = CB_GETDROPPEDSTATE;
            psl->auiMsg [SBS_GETEDITSEL     ] = CB_GETEDITSEL;
            psl->auiMsg [SBS_GETITEMDATA    ] = CB_GETITEMDATA;
            psl->auiMsg [SBS_GETSEL         ] = WM_NULL;
            psl->auiMsg [SBS_GETSELCOUNT    ] = WM_NULL;
            psl->auiMsg [SBS_GETSELITEMS    ] = WM_NULL;
            psl->auiMsg [SBS_GETTEXT        ] = CB_GETLBTEXT;
            psl->auiMsg [SBS_GETTEXTLEN     ] = CB_GETLBTEXTLEN;
            psl->auiMsg [SBS_GETTOPINDEX    ] = CB_GETTOPINDEX;
            psl->auiMsg [SBS_INITSTORAGE    ] = CB_INITSTORAGE;
            psl->auiMsg [SBS_INSERTSTRING   ] = CB_INSERTSTRING;
            psl->auiMsg [SBS_ITEMFROMPOINT  ] = WM_NULL;
            psl->auiMsg [SBS_LIMITTEXT      ] = CB_LIMITTEXT;
            psl->auiMsg [SBS_RESETCONTENT   ] = CB_RESETCONTENT;
            psl->auiMsg [SBS_SELECTSTRING   ] = CB_SELECTSTRING;
            psl->auiMsg [SBS_SETCOLUMNWIDTH ] = WM_NULL;
            psl->auiMsg [SBS_SETCURSEL      ] = CB_SETCURSEL;
            psl->auiMsg [SBS_SETEDITSEL     ] = CB_SETEDITSEL;
            psl->auiMsg [SBS_SETITEMDATA    ] = CB_SETITEMDATA;
            psl->auiMsg [SBS_SETREDRAW      ] = WM_SETREDRAW;
            psl->auiMsg [SBS_SETSEL         ] = WM_NULL;
            psl->auiMsg [SBS_SETTABSTOPS    ] = WM_NULL;
            psl->auiMsg [SBS_SETTOPINDEX    ] = CB_SETTOPINDEX;
            psl->auiMsg [SBS_SHOWDROPDOWN   ] = CB_SHOWDROPDOWN;

            psl->adCode [SBS_CLOSEUP        ] = CBN_CLOSEUP;
            psl->adCode [SBS_DBLCLK         ] = CBN_DBLCLK;
            psl->adCode [SBS_SELCHANGE      ] = CBN_SELCHANGE;
            }
        else
            {
            psl->dError                       = LB_ERR;
            psl->dErrorSpace                  = LB_ERRSPACE;

            psl->auiMsg [SBS_ADDFILE        ] = LB_ADDFILE;
            psl->auiMsg [SBS_ADDSTRING      ] = LB_ADDSTRING;
            psl->auiMsg [SBS_DELETESTRING   ] = LB_DELETESTRING;
            psl->auiMsg [SBS_DIR            ] = LB_DIR;
            psl->auiMsg [SBS_FINDSTRING     ] = LB_FINDSTRING;
            psl->auiMsg [SBS_FINDSTRINGEXACT] = LB_FINDSTRINGEXACT;
            psl->auiMsg [SBS_GETCOUNT       ] = LB_GETCOUNT;
            psl->auiMsg [SBS_GETCURSEL      ] = LB_GETCURSEL;
            psl->auiMsg [SBS_GETDROPPEDSTATE] = WM_NULL;
            psl->auiMsg [SBS_GETEDITSEL     ] = WM_NULL;
            psl->auiMsg [SBS_GETITEMDATA    ] = LB_GETITEMDATA;
            psl->auiMsg [SBS_GETSEL         ] = LB_GETSEL;
            psl->auiMsg [SBS_GETSELCOUNT    ] = LB_GETSELCOUNT;
            psl->auiMsg [SBS_GETSELITEMS    ] = LB_GETSELITEMS;
            psl->auiMsg [SBS_GETTEXT        ] = LB_GETTEXT;
            psl->auiMsg [SBS_GETTEXTLEN     ] = LB_GETTEXTLEN;
            psl->auiMsg [SBS_GETTOPINDEX    ] = LB_GETTOPINDEX;
            psl->auiMsg [SBS_INITSTORAGE    ] = LB_INITSTORAGE;
            psl->auiMsg [SBS_INSERTSTRING   ] = LB_INSERTSTRING;
            psl->auiMsg [SBS_ITEMFROMPOINT  ] = LB_ITEMFROMPOINT;
            psl->auiMsg [SBS_LIMITTEXT      ] = WM_NULL;
            psl->auiMsg [SBS_RESETCONTENT   ] = LB_RESETCONTENT;
            psl->auiMsg [SBS_SELECTSTRING   ] = LB_SELECTSTRING;
            psl->auiMsg [SBS_SETCOLUMNWIDTH ] = LB_SETCOLUMNWIDTH;
            psl->auiMsg [SBS_SETCURSEL      ] = LB_SETCURSEL;
            psl->auiMsg [SBS_SETEDITSEL     ] = WM_NULL;
            psl->auiMsg [SBS_SETITEMDATA    ] = LB_SETITEMDATA;
            psl->auiMsg [SBS_SETREDRAW      ] = WM_SETREDRAW;
            psl->auiMsg [SBS_SETSEL         ] = LB_SETSEL;
            psl->auiMsg [SBS_SETTABSTOPS    ] = LB_SETTABSTOPS;
            psl->auiMsg [SBS_SETTOPINDEX    ] = LB_SETTOPINDEX;
            psl->auiMsg [SBS_SHOWDROPDOWN   ] = WM_NULL;

            psl->adCode [SBS_CLOSEUP        ] = 0;
            psl->adCode [SBS_DBLCLK         ] = LBN_DBLCLK;
            psl->adCode [SBS_SELCHANGE      ] = LBN_SELCHANGE;
            }
        if (fClear)
            {
            SbsListClear (psl);
            }
        else
            {
            if ((psl->dCount = SbsListSend (psl, SBS_GETCOUNT,
                                            (WPARAM) 0,
                                            (LPARAM) 0))
                == psl->dError)
                {
                psl = _mfree (psl);
                }
            }
        }
    return psl;
    }

// -----------------------------------------------------------------

PSBS_LIST WINAPI SbsListDestroy (PSBS_LIST psl)
    {
    return _mfree (psl);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListOpen (PSBS_LIST psl,
                         BOOL      fClear)
    {
    BOOL fOk = FALSE;

    if (psl != NULL)
        {
        if (fClear) SbsListClear (psl);

        SetCursor (psl->hcWait);

        SbsListSend (psl, SBS_SETREDRAW,
                     (WPARAM) FALSE,
                     (LPARAM) 0);
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListClose (PSBS_LIST psl,
                           DWORD     dIndex)
    {
    return SbsListCloseEx (psl, dIndex, TRUE);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListCloseEx (PSBS_LIST psl,
                             DWORD     dIndex,
                             BOOL      fNotify)
    {
    DWORD dIndex1 = -1;

    if (psl != NULL)
        {
        dIndex1 = SbsListSelectEx (psl, dIndex, fNotify);

        SbsListSend (psl, SBS_SETREDRAW,
                     (WPARAM) TRUE,
                     (LPARAM) 0);

        SetCursor     (psl->hcDefault);
        SbsListUpdate (psl, TRUE);
        }
    return dIndex1;
    }

// -----------------------------------------------------------------

PSBS_LIST WINAPI SbsListNew (HWND        hWnd,
                             INT         iItem,
                             BOOL        fCombo,
                             DWORD       dIndex,
                             SBS_LISTNEW Callback,
                             PVOID       pContext,
                             PDWORD      pdCount)
    {
    PVOID     pText;
    DWORD     dData, dId;
    DWORD     dCount = 0;
    PSBS_LIST psl    = SbsListCreate (hWnd, iItem, fCombo);

    if (SbsListOpen (psl, FALSE))
        {
        if (Callback != NULL)
            {
            dId = 0;

            do  {
                psl->dStatus = Callback (psl, dId++, pContext,
                                         &pText, &dData);

                if (psl->dStatus == SBS_LIST_ADD)
                    {
                    if (psl->fOk)
                        {
                        psl->fOk = (SbsListAdd (psl, pText, dData)
                                    != -1);
                        }
                    dCount++;
                    }
                }
            while (psl->dStatus >= SBS_LIST_CONTINUE);
            }
        SbsListClose (psl, dIndex);
        }
    if (pdCount != NULL) *pdCount = dCount;
    return psl;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListClear (PSBS_LIST psl)
    {
    BOOL fOk = FALSE;

    if (psl != NULL)
        {
        SbsListSend (psl, SBS_RESETCONTENT,
                     (WPARAM) 0,
                     (LPARAM) 0);

        fOk = SbsListUpdate (psl, TRUE);
        psl->dCount = 0;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListUpdate (PSBS_LIST psl,
                           BOOL      fRedraw)
    {
    BOOL fOk = FALSE;

    if (psl != NULL)
        {
        fOk = TRUE;

        if (fRedraw && (!InvalidateRect (psl->hItem, NULL, FALSE)))
            fOk = FALSE;

        if (!UpdateWindow (psl->hItem))
            fOk = FALSE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListInsert (PSBS_LIST psl,
                            PVOID     pText,
                            DWORD     dData,
                            DWORD     dIndex)
    {
    PVOID pText1;
    DWORD dIndex1 = -1;

    if (psl != NULL)
        {
        pText1  = SAFEPTR (pText, (PVOID) dData);

        dIndex1 = (dIndex < psl->dCount
                   ? dIndex
                   : (dIndex != -1 ? -1 : SbsListIndex (psl)));

        if (dIndex1 == -1)
            {
            dIndex1 = SbsListSend (psl, SBS_ADDSTRING,
                                   (WPARAM) 0,
                                   (LPARAM) pText1);
            }
        else
            {
            dIndex1 = SbsListSend (psl, SBS_INSERTSTRING,
                                   (WPARAM) dIndex1,
                                   (LPARAM) pText1);
            }
        if ((dIndex1 != psl->dError     ) &&
            (dIndex1 != psl->dErrorSpace))
            {
            if (pText != NULL)
                {
                SbsListSend (psl, SBS_SETITEMDATA,
                             (WPARAM) dIndex1,
                             (LPARAM) dData);
                }
            psl->dCount++;
            }
        else
            {
            dIndex1 = -1;
            }
        }
    return dIndex1;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListAdd (PSBS_LIST psl,
                         PVOID     pText,
                         DWORD     dData)
    {
    return SbsListInsert (psl, pText, dData, SBS_LIST_END);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListRemove (PSBS_LIST psl,
                            DWORD     dIndex)
    {
    DWORD dCurrent, dCount;
    DWORD dIndex1 = -1;

    if (psl != NULL)
        {
        dCurrent = SbsListIndex (psl);
        dIndex1  = SbsListItem  (psl, dIndex, dCurrent);

        if ((dIndex1 != -1)
            &&
            ((dCount = SbsListSend (psl, SBS_DELETESTRING,
                                    (WPARAM) dIndex1,
                                    (LPARAM) 0))
             != psl->dError))
            {
            if ((dCurrent != -1) && (dCurrent > dIndex1))
                {
                dCurrent--;
                }
            SbsListSelect (psl, dCurrent);
            psl->dCount = dCount;
            }
        }
    return dIndex1;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListCount (PSBS_LIST psl)
    {
    return (psl != NULL ? psl->dCount : 0);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListItem (PSBS_LIST psl,
                          DWORD     dIndex,
                          DWORD     dDefault)
    {
    return (psl == NULL
            ? -1
            : (dIndex < psl->dCount
               ? dIndex
               : (dIndex != -1
                  ? (psl->dCount ? psl->dCount - 1 : -1)
                  : dDefault)));
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListIndex (PSBS_LIST psl)
    {
    DWORD dIndex = -1;

    if (psl != NULL)
        {
        if ((dIndex = SbsListSend (psl, SBS_GETCURSEL,
                                   (WPARAM) 0,
                                   (LPARAM) 0))
            == psl->dError)
            {
            dIndex = -1;
            }
        }
    return dIndex;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListEnd (PSBS_LIST psl)
    {
    DWORD dCount = SbsListCount (psl);
    DWORD dIndex = SbsListIndex (psl);

    return dCount && (dIndex != -1) && (dIndex == dCount-1);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSelect (PSBS_LIST psl,
                            DWORD     dIndex)
    {
    return SbsListSelectEx (psl, dIndex, TRUE);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSelectEx (PSBS_LIST psl,
                              DWORD     dIndex,
                              BOOL      fNotify)
    {
    DWORD dIndex1 = -1;

    if (psl != NULL)
        {
        dIndex1 = SbsListItem (psl, dIndex, -1);

        SbsListSend (psl, SBS_SETCURSEL,
                     (WPARAM) dIndex1,
                     (LPARAM) 0);

        if (fNotify) SbsListSelChange (psl);
        }
    return dIndex1;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSelectText (PSBS_LIST psl,
                                PVOID     pText,
                                DWORD     dDefault)
    {
    DWORD dIndex = -1;

    if (psl != NULL)
        {
        if (pText != NULL)
            {
            dIndex = SbsListFind (psl, -1, pText, NULL);
            }
        dIndex = SbsListSelect (psl, (dIndex != -1 ? dIndex
                                                   : dDefault));
        }
    return dIndex;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSkip (PSBS_LIST psl,
                          DWORD     dDelta,
                          BOOL      fDown)
    {
    DWORD dIndex = -1;

    if (psl != NULL)
        {
        if (fDown)
            {
            if ((dIndex = SbsListIndex (psl)) != -1)
                {
                dIndex = (dIndex < -1 - dDelta ? dIndex + dDelta
                                               : psl->dCount - 1);
                }
            else
                {
                dIndex = (psl->dCount ? 0 : -1);
                }
            }
        else
            {
            if ((dIndex = SbsListIndex (psl)) != -1)
                {
                dIndex = (dIndex > dDelta ? dIndex - dDelta : 0);
                }
            else
                {
                dIndex = (psl->dCount ? psl->dCount - 1 : -1);
                }
            }
        SbsListSelect (psl, dIndex);
        dIndex = SbsListIndex (psl);
        }
    return dIndex;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListUp (PSBS_LIST psl)
    {
    return SbsListSkip (psl, 1, FALSE);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListDown (PSBS_LIST psl)
    {
    return SbsListSkip (psl, 1, TRUE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListNotify (PSBS_LIST       psl,
                           SBS_LIST_NOTIFY Notify)
    {
    DWORD dCode;
    BOOL  fOk = FALSE;

    if ((psl != NULL) && (Notify < SBS_LIST_NOTIFY_)
                      && (dCode = psl->adCode [Notify]))
        {
        (psl->fUnicode ? SendMessageW : SendMessageA)
            (psl->hWnd, WM_COMMAND,
             MAKEWPARAM (psl->iItem, dCode),
             (LPARAM) psl->hItem);

        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListSelChange (PSBS_LIST psl)
    {
    return SbsListNotify (psl, SBS_SELCHANGE);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListDoubleClick (PSBS_LIST psl)
    {
    return SbsListNotify (psl, SBS_DBLCLK);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSize (PSBS_LIST psl)
    {
    DWORD dIndex, n;
    DWORD dSize = 0;

    if (psl != NULL)
        {
        for (dIndex = 0; dIndex < psl->dCount; dIndex++)
            {
            if ((n = SbsListSend (psl, SBS_GETTEXTLEN,
                                  (WPARAM) dIndex,
                                  (LPARAM) 0))
                != psl->dError)
                {
                dSize += (n + 1);
                }
            }
        dSize++;
        }
    return dSize;
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsListText (PSBS_LIST psl,
                          DWORD     dIndex)
    {
    DWORD dCurrent, dIndex1, n;
    PVOID pText = NULL;

    if (psl != NULL)
        {
        dCurrent = SbsListIndex (psl);
        dIndex1  = SbsListItem  (psl, dIndex, dCurrent);

        if (dIndex1 != -1)
            {
            if (((n = SbsListSend (psl, SBS_GETTEXTLEN,
                                   (WPARAM) dIndex1,
                                   (LPARAM) 0))
                 != psl->dError)
                &&
                ((pText = (psl->fUnicode ? (PVOID) _tnewW (n+1)
                                         : (PVOID) _tnewA (n+1)))
                 != NULL)
                &&
                (SbsListSend (psl, SBS_GETTEXT,
                              (WPARAM) dIndex1,
                              (LPARAM) pText)
                 == psl->dError))
                {
                pText = _mfree (pText);
                }
            }
        }
    return pText;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsListData (PSBS_LIST psl,
                         DWORD     dIndex,
                         PDWORD    pdData)
    {
    DWORD dCurrent, dIndex1;
    DWORD dData = SAFEVAL (pdData, -1);
    BOOL  fOk   = FALSE;

    if (psl != NULL)
        {
        dCurrent = SbsListIndex (psl);
        dIndex1  = SbsListItem  (psl, dIndex, dCurrent);

        if (dIndex1 != -1)
            {
            dData = SbsListSend (psl, SBS_GETITEMDATA,
                                 (WPARAM) dIndex1,
                                 (LPARAM) 0);
            fOk = TRUE;
            }
        }
    if (pdData != NULL) *pdData = dData;
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListFind (PSBS_LIST psl,
                          DWORD     dIndex,
                          PVOID     pText,
                          PDWORD    pdData)
    {
    DWORD dIndex1 = -1;

    if (psl != NULL)
        {
        dIndex1 = dIndex;

        while (TRUE)
            {
            if (pText != NULL)
                {
                if (((dIndex1 = SbsListSend (psl,
                                             SBS_FINDSTRINGEXACT,
                                             (WPARAM) dIndex1,
                                             (LPARAM) pText))
                     == psl->dError)
                    ||
                    ((dIndex != -1) && (dIndex >= dIndex1)))
                    {
                    dIndex1 = -1;
                    break;
                    }
                }
            else
                {
                if ((dIndex1 = (dIndex1 != -1 ? dIndex1+1 : 0))
                    >= psl->dCount)
                    {
                    dIndex1 = -1;
                    break;
                    }
                }
            if ((pdData == NULL)
                ||
                (SbsListSend (psl, SBS_GETITEMDATA,
                              (WPARAM) dIndex1,
                              (LPARAM) 0)
                 == *pdData))
                {
                break;
                }
            }
        }
    return dIndex1;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsListSend (PSBS_LIST        psl,
                          SBS_LIST_MESSAGE Message,
                          WPARAM           wParam,
                          LPARAM           lParam)
    {
    UINT  uiMsg;
    DWORD dResult = -1;

    if (psl != NULL)
        {
        if ((Message < SBS_LIST_MESSAGE_) &&
            ((uiMsg = psl->auiMsg [Message]) != WM_NULL))
            {
            dResult = (psl->fUnicode
                       ? SendMessageW
                       : SendMessageA) (psl->hItem, uiMsg,
                                        wParam, lParam);
            }
        else
            {
            dResult = psl->dError;
            }
        }
    return dResult;
    }

// =================================================================
// DEFAULT ABOUT BOX
// =================================================================

INT WINAPI SbsAboutBoxA (HWND  hWnd,
                         HICON hIcon,
                         PBYTE pbCaption,
                         PBYTE pbText1,
                         PBYTE pbText2,
                         PBYTE pbText3,
                         PBYTE pbText4)
    {
    SBS_ABOUT sa;
    INT       iId = IDABORT;

    if (hIcon == NULL)
        {
        iId = _mprintfA (hWnd, MB_OK | MB_ICONINFORMATION,
                         pbCaption, "%s\n%s\n%s\n%s",
                         pbText1, pbText2, pbText3, pbText4);
        }
    else
        {
        sa.fUnicode  = FALSE;
        sa.pbCaption = pbCaption;
        sa.pbText1   = pbText1;
        sa.pbText2   = pbText2;
        sa.pbText3   = pbText3;
        sa.pbText4   = pbText4;
        sa.IconProc1 = NULL;
        sa.IconProc2 = NULL;
        sa.hIcon1    = hIcon;
        sa.hIcon2    = hIcon;
        sa.hWnd      = hWnd;

        iId = DialogBoxParamA (ghInstance, SA(ABOUT_DIALOG), hWnd,
                               (DLGPROC) SbsAboutDlgProc,
                               (LPARAM)  &sa);
        }
    return iId;
    }

// -----------------------------------------------------------------

INT WINAPI SbsAboutBoxW (HWND  hWnd,
                         HICON hIcon,
                         PWORD pwCaption,
                         PWORD pwText1,
                         PWORD pwText2,
                         PWORD pwText3,
                         PWORD pwText4)
    {
    SBS_ABOUT sa;
    INT       iId = IDABORT;

    if (hIcon == NULL)
        {
        iId = _mprintfW (hWnd, MB_OK | MB_ICONINFORMATION,
                         pwCaption, L"%s\n%s\n%s\n%s",
                         pwText1, pwText2, pwText3, pwText4);
        }
    else
        {
        sa.fUnicode  = TRUE;
        sa.pwCaption = pwCaption;
        sa.pwText1   = pwText1;
        sa.pwText2   = pwText2;
        sa.pwText3   = pwText3;
        sa.pwText4   = pwText4;
        sa.IconProc1 = NULL;
        sa.IconProc2 = NULL;
        sa.hIcon1    = hIcon;
        sa.hIcon2    = hIcon;
        sa.hWnd      = hWnd;

        iId = DialogBoxParamW (ghInstance, SW(ABOUT_DIALOG), hWnd,
                               (DLGPROC) SbsAboutDlgProc,
                               (LPARAM)  &sa);
        }
    return iId;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutIconProc1 (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    PSBS_ABOUT  psa;
    PAINTSTRUCT ps;
    LRESULT     lr = 0;

    if ((psa = SbsWindowUserRead (hWnd)) != NULL)
        {
        if ((psa->hIcon1 != NULL) &&
            (uiMsg == WM_PAINT)   &&
            (BeginPaint (hWnd, &ps) != NULL))
            {
            DrawIcon (ps.hdc, 0, 0, psa->hIcon1);
            EndPaint (hWnd, &ps);
            }
        else
            {
            lr = (IsWindowUnicode (hWnd)

                  ? CallWindowProcW ((DLGPROC) psa->IconProc1,
                                     hWnd, uiMsg, wParam, lParam)

                  : CallWindowProcA ((DLGPROC) psa->IconProc1,
                                     hWnd, uiMsg, wParam, lParam));
            }
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutIconProc2 (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    PSBS_ABOUT  psa;
    PAINTSTRUCT ps;
    LRESULT     lr = 0;

    if ((psa = SbsWindowUserRead (hWnd)) != NULL)
        {
        if ((uiMsg == WM_PAINT)   &&
            (psa->hIcon2 != NULL) &&
            (BeginPaint (hWnd, &ps) != NULL))
            {
            DrawIcon (ps.hdc, 0, 0, psa->hIcon2);
            EndPaint (hWnd, &ps);
            }
        else
            {
            lr = (IsWindowUnicode (hWnd)

                  ? CallWindowProcW ((DLGPROC) psa->IconProc2,
                                     hWnd, uiMsg, wParam, lParam)

                  : CallWindowProcA ((DLGPROC) psa->IconProc2,
                                     hWnd, uiMsg, wParam, lParam));
            }
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutDlgInitDialog (HWND       hWnd,
                                      UINT       uiMsg,
                                      WPARAM     wParam,
                                      LPARAM     lParam,
                                      PSBS_ABOUT psa)
    {
    if (psa != NULL)
        {
        if (psa->fUnicode)
            {
            SetWindowTextW  (hWnd,            psa->pwCaption);
            SetDlgItemTextW (hWnd, IDT_TEXT1, psa->pwText1);
            SetDlgItemTextW (hWnd, IDT_TEXT2, psa->pwText2);
            SetDlgItemTextW (hWnd, IDT_TEXT3, psa->pwText3);
            SetDlgItemTextW (hWnd, IDT_TEXT4, psa->pwText4);
            }
        else
            {
            SetWindowTextA  (hWnd,            psa->pbCaption);
            SetDlgItemTextA (hWnd, IDT_TEXT1, psa->pbText1);
            SetDlgItemTextA (hWnd, IDT_TEXT2, psa->pbText2);
            SetDlgItemTextA (hWnd, IDT_TEXT3, psa->pbText3);
            SetDlgItemTextA (hWnd, IDT_TEXT4, psa->pbText4);
            }
        psa->IconProc1 =
            SbsDialogSubclass (hWnd, IDI_ICON1,
                               (WNDPROC) SbsAboutIconProc1, psa);
        psa->IconProc2 =
            SbsDialogSubclass (hWnd, IDI_ICON2,
                               (WNDPROC) SbsAboutIconProc2, psa);

        SbsWindowCenter (hWnd, SW_SHOWNORMAL);
        }
    else
        {
        EndDialog (hWnd, IDABORT);
        }
    return TRUE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutCmdOk (HWND       hWnd,
                              UINT       uiMsg,
                              WPARAM     wParam,
                              LPARAM     lParam,
                              PSBS_ABOUT psa)
    {
    EndDialog (hWnd, IDOK);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutCmdCancel (HWND       hWnd,
                                  UINT       uiMsg,
                                  WPARAM     wParam,
                                  LPARAM     lParam,
                                  PSBS_ABOUT psa)
    {
    EndDialog (hWnd, IDCANCEL);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsAboutDlgCommand (HWND       hWnd,
                                   UINT       uiMsg,
                                   WPARAM     wParam,
                                   LPARAM     lParam,
                                   PSBS_ABOUT psa)
    {
    return SbsDialogDispatch
               (hWnd, uiMsg, wParam, lParam, psa,
                SbsWindowControl (wParam, lParam),
                IDOK,               SbsAboutCmdOk,
                IDCANCEL,           SbsAboutCmdCancel,
                SBS_DEFAULT_RESULT, FALSE);
    }

// -----------------------------------------------------------------

LRESULT CALLBACK SbsAboutDlgProc (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam)
    {
    return SbsDialogDispatchEx
               (hWnd, uiMsg, wParam, lParam,
                uiMsg,
                WM_INITDIALOG,      SbsAboutDlgInitDialog,
                WM_LBUTTONDOWN,     SbsWmLButtonDown,
                WM_COMMAND,         SbsAboutDlgCommand,
                WM_CTLCOLORDLG,     SbsWmCtlColorDlg,
                WM_CTLCOLORSTATIC,  SbsWmCtlColorStatic,
                WM_CTLCOLOREDIT,    SbsWmCtlColorEdit,
                WM_CTLCOLORLISTBOX, SbsWmCtlColorListBox,
                WM_CTLCOLORBTN,     SbsWmCtlColorBtn,
                SBS_DEFAULT_RESULT, FALSE);
    }

// =================================================================
// STATUS BAR MANAGEMENT
// =================================================================

HWND WINAPI SbsStatusCreateA (HWND   hWnd,
                              DWORD  dId,
                              DWORD  dParts,
                              PDWORD pdWidths)
    {
    HWND hStatus = NULL;

    if (dParts && (pdWidths != NULL))
        {
        hStatus = CreateStatusWindowA
                      (WS_CHILD | WS_VISIBLE | WS_BORDER,
                       gabNull, hWnd, (UINT) dId);

        if (hStatus != NULL)
            {
            SendMessageA (hStatus, SB_SETPARTS,
                          (WPARAM) dParts,
                          (LPARAM) pdWidths);
 
            SendMessageA (hStatus, SB_SIMPLE,
                          (WPARAM) FALSE,
                          (LPARAM) 0);
            }
        }
    return hStatus;
    }
     
// -----------------------------------------------------------------

HWND WINAPI SbsStatusCreateW (HWND   hWnd,
                              DWORD  dId,
                              DWORD  dParts,
                              PDWORD pdWidths)
    {
    HWND hStatus = NULL;

    if (dParts && (pdWidths != NULL))
        {
        hStatus = CreateStatusWindowW
                      (WS_CHILD | WS_VISIBLE | WS_BORDER,
                       gawNull, hWnd, (UINT) dId);

        if (hStatus != NULL)
            {
            SendMessageW (hStatus, SB_SETPARTS,
                          (WPARAM) dParts,
                          (LPARAM) pdWidths);
 
            SendMessageW (hStatus, SB_SIMPLE,
                          (WPARAM) FALSE,
                          (LPARAM) 0);
            }
        }
    return hStatus;
    }
     
// -----------------------------------------------------------------

BOOL WINAPI SbsStatusMove (HWND  hWnd,
                           DWORD dWidth,
                           DWORD dHeight)
    {
    PSBS_WINDOW psw;
    RECT        r;
    BOOL        fOk = FALSE;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        (psw->hStatus != NULL))
        {
        fOk = (GetWindowRect (psw->hStatus, &r)
               &&
               MoveWindow (psw->hStatus,
                           0, (dHeight - (r.bottom - r.top)),
                           dWidth, r.bottom - r.top,
                           TRUE));
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsStatusUpdateA (HWND  hWnd,
                              DWORD dPart,
                              DWORD dType,
                              PBYTE pbFormat,
                              ...)
    {
    PSBS_WINDOW psw;
    PBYTE       pbBuffer;
    BOOL        fOk = FALSE;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        (psw->hStatus                     != NULL)
        &&
        ((pbBuffer = _vbprintfA (NULL, pbFormat, (&pbFormat)+1))
         != NULL))
        {
        SendMessageA (psw->hStatus, SB_SETTEXTA,
                      (WPARAM) (dPart | dType),
                      (LPARAM) pbBuffer);

        _mfree (pbBuffer);

        UpdateWindow (psw->hStatus);
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsStatusUpdateW (HWND  hWnd,
                              DWORD dPart,
                              DWORD dType,
                              PWORD pwFormat,
                              ...)
    {
    PSBS_WINDOW psw;
    PWORD       pwBuffer;
    BOOL        fOk = FALSE;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        (psw->hStatus                     != NULL)
        &&
        ((pwBuffer = _vbprintfW (NULL, pwFormat, (&pwFormat)+1))
         != NULL))
        {
        SendMessageW (psw->hStatus, SB_SETTEXTW,
                      (WPARAM) (dPart | dType),
                      (LPARAM) pwBuffer);

        _mfree (pwBuffer);

        UpdateWindow (psw->hStatus);
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsStatusWidth (HWND hWnd)
    {
    PSBS_WINDOW psw;
    RECT        r;
    DWORD       dWidth = 0;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        (psw->hStatus != NULL))
        {
        GetWindowRect (psw->hStatus, &r);
        dWidth = r.right - r.left;
        }
    return dWidth;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsStatusHeight (HWND hWnd)
    {
    PSBS_WINDOW psw;
    RECT        r;
    DWORD       dHeight = 0;

    if (((psw = SbsWindowUserRead (hWnd)) != NULL) &&
        (psw->hStatus != NULL))
        {
        GetWindowRect (psw->hStatus, &r);
        dHeight = r.bottom - r.top;
        }
    return dHeight;
    }

// =================================================================
// SELECTION DIALOGS
// =================================================================

VOID WINAPI SbsSelectFontA (HWND hWnd,
                            BOOL fFixed)
    {
    PSBS_WINDOW psw;
    CHOOSEFONTA cfA;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        cfA.lStructSize    = CHOOSEFONTA_;
        cfA.hInstance      = NULL;
        cfA.hwndOwner      = hWnd;
        cfA.hDC            = NULL;
        cfA.lpLogFont      = &psw->lfA;
        cfA.rgbColors      = psw->crColor;
        cfA.lCustData      = (LPARAM) NULL;
        cfA.lpfnHook       = NULL;
        cfA.lpTemplateName = NULL;
        cfA.lpszStyle      = NULL;
        cfA.nFontType      = SCREEN_FONTTYPE;
        cfA.nSizeMin       = 8;
        cfA.nSizeMax       = 72;

        cfA.Flags          = CF_FORCEFONTEXIST      |
                             CF_SCREENFONTS         |
                             CF_EFFECTS             |
                             CF_INITTOLOGFONTSTRUCT |
                             (fFixed ? CF_FIXEDPITCHONLY : 0);

        if (ChooseFontA (&cfA))
            {
            psw->crColor = cfA.rgbColors;

            SbsIniWriteColorA (psw->abProfile, psw->abWindow,
                               gabSettingsColor, &psw->crColor);

            SbsIniWriteFontA  (psw->abProfile, psw->abWindow,
                               gabSettingsFont,  &psw->lfA);

            SbsFontDelete (psw->hFont);

            if ((psw->hFont = CreateFontIndirectA (&psw->lfA))
                == NULL)
                {
                psw->hFont = ghfDefault;
                }
            InvalidateRect (hWnd, NULL, TRUE);
            }
        }
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsSelectFontW (HWND hWnd,
                            BOOL fFixed)
    {
    PSBS_WINDOW psw;
    CHOOSEFONTW cfW;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        cfW.lStructSize    = CHOOSEFONTW_;
        cfW.hInstance      = NULL;
        cfW.hwndOwner      = hWnd;
        cfW.hDC            = NULL;
        cfW.lpLogFont      = &psw->lfW;
        cfW.rgbColors      = psw->crColor;
        cfW.lCustData      = (LPARAM) NULL;
        cfW.lpfnHook       = NULL;
        cfW.lpTemplateName = NULL;
        cfW.lpszStyle      = NULL;
        cfW.nFontType      = SCREEN_FONTTYPE;
        cfW.nSizeMin       = 8;
        cfW.nSizeMax       = 72;

        cfW.Flags          = CF_FORCEFONTEXIST      |
                             CF_SCREENFONTS         |
                             CF_EFFECTS             |
                             CF_INITTOLOGFONTSTRUCT |
                             (fFixed ? CF_FIXEDPITCHONLY : 0);

        if (ChooseFontW (&cfW))
            {
            psw->crColor = cfW.rgbColors;

            SbsIniWriteColorW (psw->awProfile, psw->awWindow,
                               gawSettingsColor, &psw->crColor);

            SbsIniWriteFontW  (psw->awProfile, psw->awWindow,
                               gawSettingsFont,  &psw->lfW);

            SbsFontDelete (psw->hFont);

            if ((psw->hFont = CreateFontIndirectW (&psw->lfW))
                == NULL)
                {
                psw->hFont = ghfDefault;
                }
            InvalidateRect (hWnd, NULL, TRUE);
            }
        }
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsSelectBrushA (HWND hWnd)
    {
    PSBS_WINDOW  psw;
    CHOOSECOLORA ccA;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        ccA.lStructSize    = CHOOSECOLORA_;
        ccA.hInstance      = NULL;
        ccA.hwndOwner      = hWnd;
        ccA.rgbResult      = psw->crBrush;
        ccA.lpCustColors   = gacrGrayShades;
        ccA.lCustData      = (LPARAM) NULL;
        ccA.lpfnHook       = NULL;
        ccA.lpTemplateName = NULL;
        ccA.Flags          = CC_FULLOPEN | CC_RGBINIT;

        if (ChooseColorA (&ccA))
            {
            psw->crBrush = ccA.rgbResult;

            SbsIniWriteColorA (psw->abProfile, psw->abWindow,
                               gabSettingsBrush, &psw->crBrush);

            SbsBrushDelete (psw->hBrush);

            if ((psw->hBrush = CreateSolidBrush (psw->crBrush))
                == NULL)
                {
                psw->hBrush = ghbDefault;
                }
            psw->wcA.hbrBackground = psw->hBrush;

            SetClassLongA (hWnd, GCL_HBRBACKGROUND,
                           (LONG) psw->hBrush);

            InvalidateRect (hWnd, NULL, TRUE);
            }
        }
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsSelectBrushW (HWND hWnd)
    {
    PSBS_WINDOW  psw;
    CHOOSECOLORW ccW;

    if ((psw = SbsWindowUserRead (hWnd)) != NULL)
        {
        ccW.lStructSize    = CHOOSECOLORW_;
        ccW.hInstance      = NULL;
        ccW.hwndOwner      = hWnd;
        ccW.rgbResult      = psw->crBrush;
        ccW.lpCustColors   = gacrGrayShades;
        ccW.lCustData      = (LPARAM) NULL;
        ccW.lpfnHook       = NULL;
        ccW.lpTemplateName = NULL;
        ccW.Flags          = CC_FULLOPEN | CC_RGBINIT;

        if (ChooseColorW (&ccW))
            {
            psw->crBrush = ccW.rgbResult;

            SbsIniWriteColorW (psw->awProfile, psw->awWindow,
                               gawSettingsBrush, &psw->crBrush);

            SbsBrushDelete (psw->hBrush);

            if ((psw->hBrush = CreateSolidBrush (psw->crBrush))
                == NULL)
                {
                psw->hBrush = ghbDefault;
                }
            psw->wcW.hbrBackground = psw->hBrush;

            SetClassLongW (hWnd, GCL_HBRBACKGROUND,
                           (LONG) psw->hBrush);

            InvalidateRect (hWnd, NULL, TRUE);
            }
        }
    return;
    }

// -----------------------------------------------------------------

PPBYTE WINAPI SbsSelectFilesA (HWND   hWnd,
                               PBYTE  pbCaption,
                               PBYTE  pbFilter,
                               PBYTE  pbPreset,
                               BOOL   fSave,
                               PDWORD pdCount)
    {
    OPENFILENAMEA ofn;
    BYTE          abFolder [MAX_PATH];
    PBYTE         pbBuffer, pbFolder, pbFile, pbNext;
    DWORD         dFlags, dFolder, dBytes, i, n;
    DWORD         dCount   = 0;
    PPBYTE        ppbPaths = NULL;

    if ((_tsizeA (pbPreset) < MAX_PATH) &&
        ((pbBuffer = _tnewA (SBS_SELECT)) != NULL))
        {
        _tcopyA (abFolder, pbPreset);
        _fnormalA (abFolder, MAX_PATH, FALSE, FALSE);

        i = _fleafA (abFolder);
        _tcopynA (pbBuffer, abFolder+i, SBS_SELECT);

        abFolder [i] = 0;
        _fnormalA (abFolder, MAX_PATH, TRUE, TRUE);

        dFlags = (fSave ? OFN_OVERWRITEPROMPT : OFN_FILEMUSTEXIST)
               | (pdCount != NULL ? OFN_ALLOWMULTISELECT : 0)
               | OFN_HIDEREADONLY | OFN_EXPLORER;

        ofn.lStructSize       = OPENFILENAME_SIZE_VERSION_400A;
        ofn.hInstance         = NULL;
        ofn.hwndOwner         = hWnd;
        ofn.lpstrFilter       = pbFilter;
        ofn.lpstrCustomFilter = NULL;
        ofn.nMaxCustFilter    = 0;
        ofn.nFilterIndex      = 1;
        ofn.lpstrFile         = pbBuffer;
        ofn.nMaxFile          = SBS_SELECT;
        ofn.lpstrFileTitle    = NULL;
        ofn.nMaxFileTitle     = 0;
        ofn.lpstrInitialDir   = abFolder;
        ofn.lpstrTitle        = pbCaption;
        ofn.nFileOffset       = 0;
        ofn.nFileExtension    = 0;
        ofn.lpstrDefExt       = NULL;
        ofn.lCustData         = (LPARAM) NULL;
        ofn.lpfnHook          = NULL;
        ofn.lpTemplateName    = NULL;
        ofn.Flags             = dFlags;

        if ((fSave ? GetSaveFileNameA : GetOpenFileNameA) (&ofn))
            {
            if (pdCount != NULL)
                {
                n = 0;
                while (pbBuffer [n])
                    {
                    dCount++;
                    n += _tsizeA (pbBuffer+n) + 1;
                    }
                }
            else
                {
                if (n = _tsizeA (pbBuffer))
                    {
                    dCount++;
                    n++;
                    }
                }
            if (dCount)
                {
                if (dCount == 1)
                    {
                    pbFolder = pbBuffer;
                    dFolder  = _fleafA (pbBuffer);

                    pbFile   = pbBuffer + dFolder;
                    }
                else
                    {
                    pbFolder = pbBuffer;
                    dFolder  = _tsizeA (pbBuffer);

                    pbFile   = pbBuffer + dFolder + 1;
                    n       -=            dFolder + 1;

                    if (dFolder && (pbFolder [dFolder-1] != '\\'))
                        {
                        pbFolder [dFolder++] = '\\';
                        }
                    n += --dCount * dFolder;
                    }
                dBytes = (dCount * PVOID_) + n;

                if ((ppbPaths = _mnew (dBytes)) != NULL)
                    {
                    pbNext = (PBYTE) (ppbPaths + dCount);

                    for (i = 0; i < dCount; i++)
                        {
                        ppbPaths [i] = pbNext;

                        pbNext = _tcopyA2A (pbNext,
                                            pbFolder, dFolder);

                        n = _tcopyA (pbNext, pbFile) + 1;

                        pbNext += n;
                        pbFile += n;
                        }
                    }
                }
            }
        _mfree (pbBuffer);
        }
    if (pdCount != NULL) *pdCount = dCount;
    return ppbPaths;
    }

// -----------------------------------------------------------------

PPWORD WINAPI SbsSelectFilesW (HWND   hWnd,
                               PWORD  pwCaption,
                               PWORD  pwFilter,
                               PWORD  pwPreset,
                               BOOL   fSave,
                               PDWORD pdCount)
    {
    OPENFILENAMEW ofn;
    WORD          awFolder [MAX_PATH];
    PWORD         pwBuffer, pwFolder, pwFile, pwNext;
    DWORD         dFlags, dFolder, dBytes, i, n;
    DWORD         dCount   = 0;
    PPWORD        ppwPaths = NULL;

    if ((_tsizeW (pwPreset) < MAX_PATH) &&
        ((pwBuffer = _tnewW (SBS_SELECT)) != NULL))
        {
        _tcopyW (awFolder, pwPreset);
        _fnormalW (awFolder, MAX_PATH, FALSE, FALSE);

        i = _fleafW (awFolder);
        _tcopynW (pwBuffer, awFolder+i, SBS_SELECT);

        awFolder [i] = 0;
        _fnormalW (awFolder, MAX_PATH, TRUE, TRUE);

        dFlags = (fSave ? OFN_OVERWRITEPROMPT : OFN_FILEMUSTEXIST)
               | (pdCount != NULL ? OFN_ALLOWMULTISELECT : 0)
               | OFN_HIDEREADONLY | OFN_EXPLORER;

        ofn.lStructSize       = OPENFILENAME_SIZE_VERSION_400W;
        ofn.hInstance         = NULL;
        ofn.hwndOwner         = hWnd;
        ofn.lpstrFilter       = pwFilter;
        ofn.lpstrCustomFilter = NULL;
        ofn.nMaxCustFilter    = 0;
        ofn.nFilterIndex      = 1;
        ofn.lpstrFile         = pwBuffer;
        ofn.nMaxFile          = SBS_SELECT;
        ofn.lpstrFileTitle    = NULL;
        ofn.nMaxFileTitle     = 0;
        ofn.lpstrInitialDir   = awFolder;
        ofn.lpstrTitle        = pwCaption;
        ofn.nFileOffset       = 0;
        ofn.nFileExtension    = 0;
        ofn.lpstrDefExt       = NULL;
        ofn.lCustData         = (LPARAM) NULL;
        ofn.lpfnHook          = NULL;
        ofn.lpTemplateName    = NULL;
        ofn.Flags             = dFlags;

        if ((fSave ? GetSaveFileNameW : GetOpenFileNameW) (&ofn))
            {
            if (pdCount != NULL)
                {
                n = 0;
                while (pwBuffer [n])
                    {
                    dCount++;
                    n += _tsizeW (pwBuffer+n) + 1;
                    }
                }
            else
                {
                if (n = _tsizeW (pwBuffer))
                    {
                    dCount++;
                    n++;
                    }
                }
            if (dCount)
                {
                if (dCount == 1)
                    {
                    pwFolder = pwBuffer;
                    dFolder  = _fleafW (pwBuffer);

                    pwFile   = pwBuffer + dFolder;
                    }
                else
                    {
                    pwFolder = pwBuffer;
                    dFolder  = _tsizeW (pwBuffer);

                    pwFile   = pwBuffer + dFolder + 1;
                    n       -=            dFolder + 1;

                    if (dFolder && (pwFolder [dFolder-1] != '\\'))
                        {
                        pwFolder [dFolder++] = '\\';
                        }
                    n += --dCount * dFolder;
                    }
                dBytes = (dCount * PVOID_) + (n * WORD_);

                if ((ppwPaths = _mnew (dBytes)) != NULL)
                    {
                    pwNext = (PWORD) (ppwPaths + dCount);

                    for (i = 0; i < dCount; i++)
                        {
                        ppwPaths [i] = pwNext;

                        pwNext = _tcopyW2W (pwNext,
                                            pwFolder, dFolder);

                        n = _tcopyW (pwNext, pwFile) + 1;

                        pwNext += n;
                        pwFile += n;
                        }
                    }
                }
            }
        _mfree (pwBuffer);
        }
    if (pdCount != NULL) *pdCount = dCount;
    return ppwPaths;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsSelectFileA (HWND  hWnd,
                             PBYTE pbCaption,
                             PBYTE pbFilter,
                             PBYTE pbPreset,
                             BOOL  fSave)
    {
    PPBYTE ppbPaths;
    PBYTE  pbPath = NULL;

    if ((ppbPaths = SbsSelectFilesA (hWnd, pbCaption, pbFilter,
                                     pbPreset, fSave, NULL))
        != NULL)
        {
        _tcopyA2A (pbPath = (PBYTE) ppbPaths, *ppbPaths, -1);
        }
    return pbPath;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsSelectFileW (HWND  hWnd,
                             PWORD pwCaption,
                             PWORD pwFilter,
                             PWORD pwPreset,
                             BOOL  fSave)
    {
    PPWORD ppwPaths;
    PWORD  pwPath = NULL;

    if ((ppwPaths = SbsSelectFilesW (hWnd, pwCaption, pwFilter,
                                     pwPreset, fSave, NULL))
        != NULL)
        {
        _tcopyW2W (pwPath = (PWORD) ppwPaths, *ppwPaths, -1);
        }
    return pwPath;
    }

// =================================================================
// WINDOW MESSAGE HANDLERS
// =================================================================

LRESULT WINAPI SbsWmClose (HWND   hWnd,
                           UINT   uiMsg,
                           WPARAM wParam,
                           LPARAM lParam,
                           PVOID  pUser)
    {
    SbsWindowDestroy (hWnd);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmDestroy (HWND   hWnd,
                             UINT   uiMsg,
                             WPARAM wParam,
                             LPARAM lParam,
                             PVOID  pUser)
    {
    PostQuitMessage (0);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmSize (HWND   hWnd,
                          UINT   uiMsg,
                          WPARAM wParam,
                          LPARAM lParam,
                          PVOID  pUser)
    {
    SbsStatusMove (hWnd, LOWORD (lParam), HIWORD (lParam));
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmActivate (HWND   hWnd,
                              UINT   uiMsg,
                              WPARAM wParam,
                              LPARAM lParam,
                              PVOID  pUser)
    {
    DWORD dMode   = LOWORD (wParam);
    BOOL  fIconic = HIWORD (wParam);

    if (((dMode == WA_ACTIVE) || (dMode == WA_CLICKACTIVE)) &&
        fIconic)
        {
        ShowWindow (hWnd, SW_RESTORE);
        }
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmNcPaint (HWND   hWnd,
                             UINT   uiMsg,
                             WPARAM wParam,
                             LPARAM lParam,
                             PVOID  pUser)
    {
    if (wParam == 1)
        {
        InvalidateRect (hWnd, NULL, FALSE);
        UpdateWindow   (hWnd);
        }
    return DefWindowProc (hWnd, uiMsg, wParam, lParam);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmLButtonDown (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    SbsWindowPost (hWnd, WM_NCLBUTTONDOWN, HTCAPTION, lParam);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmQueryOpen (HWND   hWnd,
                               UINT   uiMsg,
                               WPARAM wParam,
                               LPARAM lParam,
                               PVOID  pUser)
    {
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmPaletteChanged (HWND   hWnd,
                                    UINT   uiMsg,
                                    WPARAM wParam,
                                    LPARAM lParam,
                                    PVOID  pUser)
    {
    if ((HWND) wParam != hWnd) InvalidateRect (hWnd, NULL, FALSE);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmQueryNewPalette (HWND   hWnd,
                                     UINT   uiMsg,
                                     WPARAM wParam,
                                     LPARAM lParam,
                                     PVOID  pUser)
    {
    InvalidateRect (hWnd, NULL, FALSE);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColorDlg (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    return (LRESULT) ghbDefault;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColorStatic (HWND   hWnd,
                                    UINT   uiMsg,
                                    WPARAM wParam,
                                    LPARAM lParam,
                                    PVOID  pUser)
    {
    SetBkMode ((HDC) wParam, TRANSPARENT);
    return (LRESULT) ghbDefault;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColorEdit (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam,
                                  PVOID  pUser)
    {
    BOOL fDisabled = ((SbsWindowLongRead ((HWND) lParam, GWL_STYLE)
                       & WS_DISABLED) != 0);

    return (LRESULT) (fDisabled ? ghbDefault
                                : GetStockObject (SBS_BRUSH_EDIT));
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColorListBox (HWND   hWnd,
                                     UINT   uiMsg,
                                     WPARAM wParam,
                                     LPARAM lParam,
                                     PVOID  pUser)
    {
    return (LRESULT) GetStockObject (SBS_BRUSH_LIST);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColorBtn (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    SetBkMode ((HDC) wParam, TRANSPARENT);
    return (LRESULT) GetStockObject (SBS_BRUSH_BUTTON);
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCtlColor (HWND   hWnd,
                              UINT   uiMsg,
                              WPARAM wParam,
                              LPARAM lParam,
                              PVOID  pUser)
    {
    LRESULT lr = FALSE;

    switch (HIWORD (lParam))
        {
        case CTLCOLOR_DLG:
            lr = SbsWmCtlColorDlg
                     (hWnd, uiMsg, wParam, LOWORD (lParam), pUser);
            break;

        case CTLCOLOR_STATIC:
            lr = SbsWmCtlColorStatic
                     (hWnd, uiMsg, wParam, LOWORD (lParam), pUser);
            break;

        case CTLCOLOR_EDIT:
            lr = SbsWmCtlColorEdit
                     (hWnd, uiMsg, wParam, LOWORD (lParam), pUser);
            break;

        case CTLCOLOR_LISTBOX:
            lr = SbsWmCtlColorListBox
                     (hWnd, uiMsg, wParam, LOWORD (lParam), pUser);
            break;

        case CTLCOLOR_BTN:
            lr = SbsWmCtlColorBtn
                     (hWnd, uiMsg, wParam, LOWORD (lParam), pUser);
            break;
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCommandExit (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    SbsWindowPost (hWnd, WM_CLOSE, (WPARAM) NULL, (LPARAM) NULL);
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCommandFont (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    if (SbsWindowValid (hWnd))
        {
        (IsWindowUnicode (hWnd)
         ? SbsSelectFontW (hWnd, FALSE)
         : SbsSelectFontA (hWnd, FALSE));
        }
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCommandColor (HWND   hWnd,
                                  UINT   uiMsg,
                                  WPARAM wParam,
                                  LPARAM lParam,
                                  PVOID  pUser)
    {
    if (SbsWindowValid (hWnd))
        {
        (IsWindowUnicode (hWnd)
         ? SbsSelectBrushW (hWnd)
         : SbsSelectBrushA (hWnd));
        }
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmCommandHelp (HWND   hWnd,
                                 UINT   uiMsg,
                                 WPARAM wParam,
                                 LPARAM lParam,
                                 PVOID  pUser)
    {
    if (SbsWindowValid (hWnd))
        {
        (IsWindowUnicode (hWnd)
         ? WinHelpW (hWnd, NULL, HELP_HELPONHELP, 0)
         : WinHelpA (hWnd, NULL, HELP_HELPONHELP, 0));
        }
    return FALSE;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmAppTimer (HWND   hWnd,
                              UINT   uiMsg,
                              WPARAM wParam,
                              LPARAM lParam,
                              PVOID  pUser)
    {
    LRESULT lr = -1;

    if (SbsWindowValid (hWnd))
        {
        lr = (IsWindowUnicode (hWnd)
              ? SbsWindowIdleW (NULL)
              : SbsWindowIdleA (NULL));
        }
    return lr;
    }

// -----------------------------------------------------------------

LRESULT WINAPI SbsWmDefault (HWND   hWnd,
                             UINT   uiMsg,
                             WPARAM wParam,
                             LPARAM lParam,
                             PVOID  pUser)
    {
    return SbsWindowDefault (hWnd, uiMsg, wParam, lParam, -1);
    }

// =================================================================
// GDI SUPPORT
// =================================================================

BOOL WINAPI SbsDrawShades (PCOLORREF   pcr,
                           PSBS_SHADES pss)
    {
    COLORREF cr  = SBS_COLOR_BACKGROUND (pcr);
    BOOL     fOk = FALSE;

    if (pss != NULL)
        {
        pss->crBlack  = SBS_COLOR_BLACK;
        pss->crWhite  = SBS_COLOR_WHITE;
        pss->crNormal = cr;

        pss->crDark   = RGB (((DWORD) GetRValue (cr) *   2) / 3,
                             ((DWORD) GetGValue (cr) *   2) / 3,
                             ((DWORD) GetBValue (cr) *   2) / 3);

        pss->crLight  = RGB (((DWORD) GetRValue (cr) + 256) / 2,
                             ((DWORD) GetGValue (cr) + 256) / 2,
                             ((DWORD) GetBValue (cr) + 256) / 2);

        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsDrawStyle (DWORD dFlags,
                           DWORD dLeft,
                           DWORD dRight,
                           DWORD dTop,
                           DWORD dBottom)
    {
    return SBS_STYLE (dFlags, dLeft, dRight, dTop, dBottom);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawMargins (DWORD  dStyle,
                            PDWORD pdLeft,
                            PDWORD pdRight,
                            PDWORD pdTop,
                            PDWORD pdBottom)
    {
    if (pdLeft   != NULL)
       *pdLeft    = SBS_STYLE_MARGIN_LEFT   (dStyle);

    if (pdRight  != NULL)
       *pdRight   = SBS_STYLE_MARGIN_RIGHT  (dStyle);

    if (pdTop    != NULL)
       *pdTop     = SBS_STYLE_MARGIN_TOP    (dStyle);

    if (pdBottom != NULL)
       *pdBottom  = SBS_STYLE_MARGIN_BOTTOM (dStyle);

    return (dStyle & SBS_STYLE_MARGINS) != 0;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawNormalize (LONG  lX,
                              LONG  lY,
                              LONG  lWidth,
                              LONG  lHeight,
                              PRECT pr)
    {
    BOOL fOk = FALSE;

    if (pr != NULL)
        {
        if (lWidth < 0)
            {
            pr->left   = lX + 1 + lWidth;
            pr->right  = lX + 1;
            }
        else
            {
            pr->left   = lX;
            pr->right  = lX + lWidth;
            }
        if (lHeight < 0)
            {
            pr->top    = lY + 1 + lHeight;
            pr->bottom = lY + 1;
            }
        else
            {
            pr->top    = lY;
            pr->bottom = lY + lHeight;
            }
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawNormalizeEx (LONG  lX,
                                LONG  lY,
                                LONG  lWidth,
                                LONG  lHeight,
                                PRECT pr,
                                DWORD dStyle)
    {
    DWORD dLeft, dRight, dTop, dBottom, dWidth, dHeight;
    BOOL  fOk = FALSE;

    if (SbsDrawNormalize (lX, lY, lWidth, lHeight, pr))
        {
        if (SbsDrawMargins (dStyle, &dLeft, &dRight,
                                    &dTop,  &dBottom))
            {
            dWidth  = pr->right  - pr->left;
            dHeight = pr->bottom - pr->top;

            if (dWidth >= dLeft + dRight)
                {
                pr->left   += dLeft;
                pr->right  -= dRight;
                }
            else
                {
                pr->left   += dWidth / 2;
                pr->right   = pr->left;
                }
            if (dHeight >= dTop + dBottom)
                {
                pr->top    += dTop;
                pr->bottom -= dBottom;
                }
            else
                {
                pr->top    += dHeight / 2;
                pr->bottom  = pr->top;
                }
            }
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawTest (PRECT pr,
                         DWORD dPixels)
    {
    BOOL fOk1 = FALSE;
    BOOL fOk2 = FALSE;

    if (pr != NULL)
        {
        if (pr->right >= pr->left)
            {
            fOk1 = ((DWORD) (pr->right - pr->left) >= 2 * dPixels);
            }
        else
            {
            fOk1 = ((DWORD) (pr->left - pr->right) >= 2 * dPixels);
            }
        if (pr->bottom >= pr->top)
            {
            fOk2 = ((DWORD) (pr->bottom - pr->top) >= 2 * dPixels);
            }
        else
            {
            fOk2 = ((DWORD) (pr->top - pr->bottom) >= 2 * dPixels);
            }
        }
    return fOk1 && fOk2;
    }

// -----------------------------------------------------------------

LONG WINAPI SbsDrawMetricsA (HDC   hDC,
                             PLONG plTop,
                             PLONG plBottom)
    {
    TEXTMETRICA tmA;
    LONG        lTop    = 0;
    LONG        lBottom = 0;
    LONG        lHeight = 0;

    if ((hDC != NULL) && GetTextMetricsA (hDC, &tmA) &&
        (tmA.tmHeight > 0))
        {
        lHeight = tmA.tmHeight;

        if (tmA.tmExternalLeading > 0)
            {
            if (lHeight + tmA.tmExternalLeading >= lHeight)
                {
                lTop    = tmA.tmExternalLeading / 2;
                lBottom = lTop + (tmA.tmExternalLeading & 1);
                }
            else
                {
                lHeight = 0;
                }
            }
        }
    if (plTop    != NULL) *plTop    = lTop;
    if (plBottom != NULL) *plBottom = lBottom;
    return lHeight;
    }

// -----------------------------------------------------------------

LONG WINAPI SbsDrawMetricsW (HDC   hDC,
                             PLONG plTop,
                             PLONG plBottom)
    {
    TEXTMETRICW tmW;
    LONG        lTop    = 0;
    LONG        lBottom = 0;
    LONG        lMiddle = 0;

    if ((hDC != NULL) && GetTextMetricsW (hDC, &tmW) &&
        (tmW.tmHeight > 0))
        {
        lMiddle = tmW.tmHeight;

        if (tmW.tmExternalLeading > 0)
            {
            if (lMiddle + tmW.tmExternalLeading >= lMiddle)
                {
                lTop    = tmW.tmExternalLeading / 2;
                lBottom = lTop + (tmW.tmExternalLeading & 1);
                }
            else
                {
                lMiddle = 0;
                }
            }
        }
    if (plTop    != NULL) *plTop    = lTop;
    if (plBottom != NULL) *plBottom = lBottom;
    return lMiddle;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawLine (HDC      hDC,
                         COLORREF cr,
                         LONG     lX,
                         LONG     lY,
                         LONG     lLength,
                         BOOL     fVertical)
    {
    HPEN  hp, hp1;
    DWORD lX1, lY1;
    POINT p;
    BOOL  fRestore;
    BOOL  fOk = FALSE;

    if (hDC != NULL)
        {
        if ((cr != CLR_INVALID) && lLength)
            {
            fRestore = MoveToEx (hDC, 0, 0, &p);

            if (MoveToEx (hDC, lX, lY, NULL))
                {
                if ((hp = CreatePen (PS_SOLID, 0, cr)) != NULL)
                    {
                    if ((hp1 = SelectObject (hDC, hp)) != NULL)
                        {
                        if (fVertical)
                            {
                            lX1 = lX;
                            lY1 = lY + lLength;
                            }
                        else
                            {
                            lX1 = lX + lLength;
                            lY1 = lY;
                            }
                        fOk = LineTo (hDC, lX1, lY1);
                        SelectObject (hDC, hp1);
                        }
                    DeleteObject (hp);
                    }
                }
            if (fRestore) MoveToEx (hDC, p.x, p.y, NULL);
            }
        else
            {
            fOk = TRUE;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawAngle (HDC      hDC,
                          COLORREF cr,
                          LONG     lX,
                          LONG     lY,
                          LONG     lWidth,
                          LONG     lHeight)
    {
    BOOL fOk = FALSE;

    if (hDC != NULL)
        {
        fOk = TRUE;

        if ((cr != CLR_INVALID) && lWidth && lHeight)
            {
            if (!SbsDrawLine (hDC, cr, lX, lY, lHeight, TRUE))
                {
                fOk = FALSE;
                }
            if (!SbsDrawLine (hDC, cr, lX, lY, lWidth, FALSE))
                {
                fOk = FALSE;
                }
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawRectangle (HDC      hDC,
                              COLORREF cr,
                              LONG     lX,
                              LONG     lY,
                              LONG     lWidth,
                              LONG     lHeight,
                              BOOL     fFill)
    {
    HBRUSH hb, hb1;
    RECT   r;
    BOOL   fOk = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if ((cr != CLR_INVALID) && lWidth && lHeight)
            {
            if (fFill)
                {
                fOk = FALSE;

                if ((hb = CreateSolidBrush (cr)) != NULL)
                    {
                    if ((hb1 = SelectObject (hDC, hb)) != NULL)
                        {
                        fOk = FillRect (hDC, &r, hb);
                        SelectObject (hDC, hb1);
                        }
                    DeleteObject (hb);
                    }
                }
            else
                {
                if (!SbsDrawAngle (hDC, cr,
                                   r.left, r.top,
                                   r.right-r.left, r.bottom-r.top))
                    {
                    fOk = FALSE;
                    }
                if (!SbsDrawAngle (hDC, cr,
                                   r.right-1, r.bottom-1,
                                   r.left-r.right, r.top-r.bottom))
                    {
                    fOk = FALSE;
                    }
                }
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawSingle (HDC      hDC,
                           COLORREF crLeftTop,
                           COLORREF crRightBottom,
                           LONG     lX,
                           LONG     lY,
                           LONG     lWidth,
                           LONG     lHeight)
    {
    RECT r;
    BOOL fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

    if (r.bottom-r.top >= 1)
        {
        if ((crLeftTop != CLR_INVALID)
            &&
            (!SbsDrawAngle (hDC, crLeftTop,
                            r.left, r.top,
                            r.right-r.left, r.bottom-r.top-1)))
            {
            fOk = FALSE;
            }
        if ((crRightBottom != CLR_INVALID)
            &&
            (!SbsDrawAngle (hDC, crRightBottom,
                            r.right-1, r.bottom-1,
                            r.left-r.right, r.top-r.bottom+1)))
            {
            fOk = FALSE;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawDouble (HDC      hDC,
                           COLORREF crLeftTopOuter,
                           COLORREF crLeftTopInner,
                           COLORREF crRightBottomInner,
                           COLORREF crRightBottomOuter,
                           LONG     lX,
                           LONG     lY,
                           LONG     lWidth,
                           LONG     lHeight)
    {
    RECT r;
    BOOL fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

    if (!SbsDrawSingle (hDC, crLeftTopOuter, crRightBottomOuter,
                        SBS_RECT (r)))
        {
        fOk = FALSE;
        }
    if (SbsDrawTest (&r, 1)
        &&
        (!SbsDrawSingle (hDC, crLeftTopInner, crRightBottomInner,
                         SBS_RECT_EX (r, 1))))
        {
        fOk = FALSE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawFrame (HDC       hDC,
                          PCOLORREF pcr,
                          LONG      lX,
                          LONG      lY,
                          LONG      lWidth,
                          LONG      lHeight,
                          BOOL      fFlat,
                          BOOL      fPopOut)
    {
    SBS_SHADES ss;
    COLORREF   crLeftTopOuter;
    COLORREF   crLeftTopInner;
    COLORREF   crRightBottomInner;
    COLORREF   crRightBottomOuter;
    COLORREF   cr = SBS_COLOR_BACKGROUND (pcr);

    SbsDrawShades (&cr, &ss);

    if (fFlat)
        {
        if (fPopOut)
            {
            crLeftTopOuter     = ss.crWhite;
            crLeftTopInner     = ss.crDark;
            crRightBottomInner = ss.crWhite;
            crRightBottomOuter = ss.crDark;
            }
        else
            {
            crLeftTopOuter     = ss.crDark;
            crLeftTopInner     = ss.crWhite;
            crRightBottomInner = ss.crDark;
            crRightBottomOuter = ss.crWhite;
            }
        }
    else
        {
        if (fPopOut)
            {
            crLeftTopOuter     = ss.crWhite;
            crLeftTopInner     = ss.crLight;
            crRightBottomInner = ss.crDark;
            crRightBottomOuter = ss.crBlack;
            }
        else
            {
            crLeftTopOuter     = ss.crBlack;
            crLeftTopInner     = ss.crDark;
            crRightBottomInner = ss.crLight;
            crRightBottomOuter = ss.crWhite;
            }
        }
    return SbsDrawDouble (hDC,
                          crLeftTopOuter,     crLeftTopInner, 
                          crRightBottomInner, crRightBottomOuter,
                          lX, lY, lWidth, lHeight);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawArea (HDC       hDC,
                         PCOLORREF pcr,
                         LONG      lX,
                         LONG      lY,
                         LONG      lWidth,
                         LONG      lHeight,
                         BOOL      fFlat,
                         BOOL      fPopOut)
    {
    RECT     r;
    COLORREF cr  = SBS_COLOR_BACKGROUND (pcr);
    BOOL     fOk = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if (!SbsDrawFrame (hDC, &cr, SBS_RECT (r), fFlat, fPopOut))
            {
            fOk = FALSE;
            }
        if (SbsDrawTest (&r, 2))
            {
            r.left += 2; r.right  -= 2;
            r.top  += 2; r.bottom -= 2;

            if (!SbsDrawRectangle (hDC, cr, SBS_RECT (r), TRUE))
                {
                fOk = FALSE;
                }
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawBitmap (HDC     hDC,
                           HBITMAP hBitmap,
                           LONG    lX,
                           LONG    lY,
                           LONG    lWidth,
                           LONG    lHeight)
    {
    RECT     r;
    HDC      hDC1;
    HGDIOBJ  hgo;
    BITMAP   bm;
    COLORREF cr;
    BOOL     fOk = FALSE;

    if (hBitmap != NULL)
        {
        if (SbsDrawNormalize (lX, lY, lWidth, lHeight, &r) &&
            (hDC != NULL)                                  &&
            ((hDC1 = CreateCompatibleDC (hDC)) != NULL))
            {
            if ((hgo = SelectObject (hDC1, hBitmap)) != NULL)
                {
                if (GetObject (hBitmap, sizeof (BITMAP), &bm))
                    {
                    fOk = StretchBlt (hDC, SBS_RECT (r), hDC1,
                                      0, 0, bm.bmWidth, bm.bmHeight,
                                      SRCCOPY);
                    }
                SelectObject (hDC1, hgo);
                }
            DeleteDC (hDC1);
            }
        }
    else
        {
        fOk = (hDC != NULL) &&
              ((cr = GetBkColor (hDC)) != CLR_INVALID)
              &&
              SbsDrawRectangle (hDC, cr, lX, lY, lWidth, lHeight,
                                TRUE);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawTextA (HDC       hDC,
                          PBYTE     pbText,
                          PCOLORREF pcr,
                          LONG      lX,
                          LONG      lY,
                          LONG      lWidth,
                          LONG      lHeight,
                          DWORD     dStyle)
    {
    RECT     r, r1;
    PBYTE    pbData;
    INT      iMode;
    LONG     lText, lArea, lOffset;
    LONG     lLines, lLine, lStart, lTop, lMiddle, lBottom;
    DWORD    dData, dFormat, i, n;
    COLORREF cr  = SBS_COLOR_FOREGROUND (pcr);
    BOOL     fOk = FALSE;

    if (lMiddle = SbsDrawMetricsA (hDC, &lTop, &lBottom))
        {
        lLine = lTop + lMiddle + lBottom;

        fOk   = SbsDrawNormalizeEx (lX, lY, lWidth, lHeight, &r,
                                    dStyle);

        if ((pbText != NULL)    && pbText [0] &&
            (r.left != r.right) && (r.top != r.bottom))
            {
            for (n = 1, i = 0; pbText [i]; i++)
                {
                if (pbText [i] == '\n') n++;
                }
            if (n < (DWORD) (lLines = MAXLONG / lLine))
                {
                lLines = n;
                }
            lText   = lLines   * lLine;
            lArea   = r.bottom - r.top;

            dFormat = DT_VCENTER | DT_SINGLELINE
                      |
                      (dStyle & SBS_STYLE_PREFIX ? 0 : DT_NOPREFIX)
                      |
                      (dStyle & SBS_STYLE_LEFT
                       ? (dStyle & SBS_STYLE_RIGHT
                          ? DT_CENTER
                          : DT_LEFT)
                       : (dStyle & SBS_STYLE_RIGHT
                          ? DT_RIGHT
                          : DT_CENTER));

            lOffset = (dStyle & SBS_STYLE_TOP
                       ? (dStyle & SBS_STYLE_BOTTOM
                          ? (lArea - lText) / 2
                          : 0)
                       : (dStyle & SBS_STYLE_BOTTOM
                          ? (lArea - lText)
                          : (lArea - lText) / 2));

            if (lOffset < 0)
                {
                lStart   = (-lOffset) / lLine;
                lOffset += lStart     * lLine;
                }
            else
                {
                lStart = 0;
                }
            for (n = 0, pbData = pbText; *pbData; pbData++)
                {
                if (n == (DWORD) lStart) break;
                if (*pbData == '\n') n++;
                }
            cr    = SetTextColor (hDC, cr         );
            iMode = SetBkMode    (hDC, TRANSPARENT);

            r1.left  = r.left;
            r1.right = r.right;

            while (*pbData)
                {
                r1.top    = r.top + lOffset + lTop;
                r1.bottom = r.top + lOffset + lTop + lMiddle;

                if (r1.top >= r.bottom) break;

                for (dData = 0;
                     pbData [dData] && (pbData [dData] != '\r')
                                    && (pbData [dData] != '\n');
                     dData++);

                if (dData &&
                    (!DrawTextA (hDC, pbData, dData, &r1, dFormat)))
                    {
                    fOk = FALSE;
                    }
                pbData += dData;

                if (*pbData && (*pbData++ == '\n'))
                    {
                    lOffset += lLine;
                    }
                }
            if (cr != CLR_INVALID) SetTextColor (hDC, cr   );
            if (iMode            ) SetBkMode    (hDC, iMode);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawTextW (HDC       hDC,
                          PWORD     pwText,
                          PCOLORREF pcr,
                          LONG      lX,
                          LONG      lY,
                          LONG      lWidth,
                          LONG      lHeight,
                          DWORD     dStyle)
    {
    RECT     r, r1;
    PWORD    pwData;
    INT      iMode;
    LONG     lText, lArea, lOffset;
    LONG     lLines, lLine, lStart, lTop, lMiddle, lBottom;
    DWORD    dData, dFormat, i, n;
    COLORREF cr  = SBS_COLOR_FOREGROUND (pcr);
    BOOL     fOk = FALSE;

    if (lMiddle = SbsDrawMetricsW (hDC, &lTop, &lBottom))
        {
        lLine = lTop + lMiddle + lBottom;

        fOk   = SbsDrawNormalizeEx (lX, lY, lWidth, lHeight, &r,
                                    dStyle);

        if ((pwText != NULL)    && pwText [0] &&
            (r.left != r.right) && (r.top != r.bottom))
            {
            for (n = 1, i = 0; pwText [i]; i++)
                {
                if (pwText [i] == '\n') n++;
                }
            if (n < (DWORD) (lLines = MAXLONG / lLine))
                {
                lLines = n;
                }
            lText   = lLines   * lLine;
            lArea   = r.bottom - r.top;

            dFormat = DT_VCENTER | DT_SINGLELINE
                      |
                      (dStyle & SBS_STYLE_PREFIX ? 0 : DT_NOPREFIX)
                      |
                      (dStyle & SBS_STYLE_LEFT
                       ? (dStyle & SBS_STYLE_RIGHT
                          ? DT_CENTER
                          : DT_LEFT)
                       : (dStyle & SBS_STYLE_RIGHT
                          ? DT_RIGHT
                          : DT_CENTER));

            lOffset = (dStyle & SBS_STYLE_TOP
                       ? (dStyle & SBS_STYLE_BOTTOM
                          ? (lArea - lText) / 2
                          : 0)
                       : (dStyle & SBS_STYLE_BOTTOM
                          ? (lArea - lText)
                          : (lArea - lText) / 2));

            if (lOffset < 0)
                {
                lStart   = (-lOffset) / lLine;
                lOffset += lStart     * lLine;
                }
            else
                {
                lStart = 0;
                }
            for (n = 0, pwData = pwText; *pwData; pwData++)
                {
                if (n == (DWORD) lStart) break;
                if (*pwData == '\n') n++;
                }
            cr    = SetTextColor (hDC, cr         );
            iMode = SetBkMode    (hDC, TRANSPARENT);

            r1.left  = r.left;
            r1.right = r.right;

            while (*pwData)
                {
                r1.top    = r.top + lOffset + lTop;
                r1.bottom = r.top + lOffset + lTop + lMiddle;

                if (r1.top >= r.bottom) break;

                for (dData = 0;
                     pwData [dData] && (pwData [dData] != '\r')
                                    && (pwData [dData] != '\n');
                     dData++);

                if (dData &&
                    (!DrawTextW (hDC, pwData, dData, &r1, dFormat)))
                    {
                    fOk = FALSE;
                    }
                pwData += dData;

                if (*pwData && (*pwData++ == '\n'))
                    {
                    lOffset += lLine;
                    }
                }
            if (cr != CLR_INVALID) SetTextColor (hDC, cr   );
            if (iMode            ) SetBkMode    (hDC, iMode);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawBoxA (HDC       hDC,
                         HBITMAP   hBitmap,
                         PBYTE     pbText,
                         PCOLORREF pcrText,
                         PCOLORREF pcrArea,
                         LONG      lX,
                         LONG      lY,
                         LONG      lWidth,
                         LONG      lHeight,
                         DWORD     dStyle)
    {
    RECT     r;
    COLORREF crText = SBS_COLOR_FOREGROUND (pcrText);
    COLORREF crArea = SBS_COLOR_BACKGROUND (pcrArea);
    BOOL     fOk    = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if (dStyle & SBS_STYLE_FRAME)
            {
            if (!SbsDrawFrame (hDC, &crArea, SBS_RECT (r),
                               dStyle & SBS_STYLE_FLAT,
                               dStyle & SBS_STYLE_POP_OUT))
                {
                fOk = FALSE;
                }
            if (SbsDrawTest (&r, 2))
                {
                r.left += 2; r.right  -= 2;
                r.top  += 2; r.bottom -= 2;
                }
            else
                {
                r.left = r.right;
                r.top  = r.bottom;
                }
            }
        if (hBitmap != NULL)
            {
            if (!SbsDrawBitmap (hDC, hBitmap, SBS_RECT (r)))
                {
                fOk = FALSE;
                }
            }
        else
            {
            if (!SbsDrawRectangle (hDC, crArea, SBS_RECT (r), TRUE))
                {
                fOk = FALSE;
                }
            }
        if (!SbsDrawTextA (hDC, pbText, &crText,
                           SBS_RECT (r), dStyle))
            {
            fOk = FALSE;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawBoxW (HDC       hDC,
                         HBITMAP   hBitmap,
                         PWORD     pwText,
                         PCOLORREF pcrText,
                         PCOLORREF pcrArea,
                         LONG      lX,
                         LONG      lY,
                         LONG      lWidth,
                         LONG      lHeight,
                         DWORD     dStyle)
    {
    RECT     r;
    COLORREF crText = SBS_COLOR_FOREGROUND (pcrText);
    COLORREF crArea = SBS_COLOR_BACKGROUND (pcrArea);
    BOOL     fOk    = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if (dStyle & SBS_STYLE_FRAME)
            {
            if (!SbsDrawFrame (hDC, &crArea, SBS_RECT (r),
                               dStyle & SBS_STYLE_FLAT,
                               dStyle & SBS_STYLE_POP_OUT))
                {
                fOk = FALSE;
                }
            if (SbsDrawTest (&r, 2))
                {
                r.left += 2; r.right  -= 2;
                r.top  += 2; r.bottom -= 2;
                }
            else
                {
                r.left = r.right;
                r.top  = r.bottom;
                }
            }
        if (hBitmap != NULL)
            {
            if (!SbsDrawBitmap (hDC, hBitmap, SBS_RECT (r)))
                {
                fOk = FALSE;
                }
            }
        else
            {
            if (!SbsDrawRectangle (hDC, crArea, SBS_RECT (r), TRUE))
                {
                fOk = FALSE;
                }
            }
        if (!SbsDrawTextW (hDC, pwText, &crText,
                           SBS_RECT (r), dStyle))
            {
            fOk = FALSE;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawButtonA (HDC       hDC,
                            HBITMAP   hBitmap,
                            PBYTE     pbText,
                            PCOLORREF pcrText,
                            PCOLORREF pcrArea,
                            LONG      lX,
                            LONG      lY,
                            LONG      lWidth,
                            LONG      lHeight,
                            BOOL      fClicked)
    {
    DWORD dStyle = SBS_STYLE_FRAME | SBS_STYLE_PREFIX |
                   (fClicked ? 0 : SBS_STYLE_POP_OUT);

    return SbsDrawBoxA (hDC, hBitmap, pbText, pcrText, pcrArea,
                        lX, lY, lWidth, lHeight, dStyle);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawButtonW (HDC       hDC,
                            HBITMAP   hBitmap,
                            PWORD     pwText,
                            PCOLORREF pcrText,
                            PCOLORREF pcrArea,
                            LONG      lX,
                            LONG      lY,
                            LONG      lWidth,
                            LONG      lHeight,
                            BOOL      fClicked)
    {
    DWORD dStyle = SBS_STYLE_FRAME | SBS_STYLE_PREFIX |
                   (fClicked ? 0 : SBS_STYLE_POP_OUT);

    return SbsDrawBoxW (hDC, hBitmap, pwText, pcrText, pcrArea,
                        lX, lY, lWidth, lHeight, dStyle);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawInfoA (HDC       hDC,
                          HBITMAP   hBitmap,
                          HICON     hIcon,
                          PBYTE     pbText,
                          PCOLORREF pcrText,
                          PCOLORREF pcrArea,
                          LONG      lX,
                          LONG      lY,
                          LONG      lWidth,
                          LONG      lHeight,
                          BOOL      fCenter)
    {
    RECT     r;
    DWORD    dX, dY, dWidth, dHeight, dLeft;
    DWORD    dFlags = (fCenter ? 0 : SBS_STYLE_LEFT);
    COLORREF crText = SBS_COLOR_FOREGROUND (pcrText);
    COLORREF crArea = SBS_COLOR_BACKGROUND (pcrArea);
    BOOL     fOk    = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if (!SbsDrawFrame (hDC, &crArea, SBS_RECT (r), FALSE, TRUE))
            {
            fOk = FALSE;
            }
        if (SbsDrawTest (&r, 2))
            {
            r.left += 2; r.right  -= 2;
            r.top  += 2; r.bottom -= 2;
            }
        else
            {
            r.left = r.right;
            r.top  = r.bottom;
            }
        if (hBitmap != NULL)
            {
            if (!SbsDrawBitmap (hDC, hBitmap, SBS_RECT (r)))
                {
                fOk = FALSE;
                }
            }
        else
            {
            if (!SbsDrawRectangle (hDC, crArea, SBS_RECT (r), TRUE))
                {
                fOk = FALSE;
                }
            }
        if (hIcon != NULL)
            {
            dWidth  = r.right  - r.left;
            dHeight = r.bottom - r.top;

            dX = (dWidth  > ICON_PIXELS
                  ? (dWidth  - ICON_PIXELS) / 2
                  : 0);

            dY = (dHeight > ICON_PIXELS
                  ? (dHeight - ICON_PIXELS) / 2
                  : 0);

            dX = min (dX, dY);

            if (!DrawIcon (hDC, r.left + dX, r.top + dY, hIcon))
                {
                fOk = FALSE;
                }
            dLeft = dX + ICON_PIXELS + (fCenter ? 0 : dX);
            }
        else
            {
            dLeft = 0;
            }
        if (!SbsDrawTextA (hDC, pbText, &crText, SBS_RECT (r),
                           SbsDrawStyle (dFlags, dLeft, 0, 0, 0)))
            {
            fOk = FALSE;
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsDrawInfoW (HDC       hDC,
                          HBITMAP   hBitmap,
                          HICON     hIcon,
                          PWORD     pwText,
                          PCOLORREF pcrText,
                          PCOLORREF pcrArea,
                          LONG      lX,
                          LONG      lY,
                          LONG      lWidth,
                          LONG      lHeight,
                          BOOL      fCenter)
    {
    RECT     r;
    DWORD    dX, dY, dWidth, dHeight, dLeft;
    DWORD    dFlags = (fCenter ? 0 : SBS_STYLE_LEFT);
    COLORREF crText = SBS_COLOR_FOREGROUND (pcrText);
    COLORREF crArea = SBS_COLOR_BACKGROUND (pcrArea);
    BOOL     fOk    = FALSE;

    if (hDC != NULL)
        {
        fOk = SbsDrawNormalize (lX, lY, lWidth, lHeight, &r);

        if (!SbsDrawFrame (hDC, &crArea, SBS_RECT (r), FALSE, TRUE))
            {
            fOk = FALSE;
            }
        if (SbsDrawTest (&r, 2))
            {
            r.left += 2; r.right  -= 2;
            r.top  += 2; r.bottom -= 2;
            }
        else
            {
            r.left = r.right;
            r.top  = r.bottom;
            }
        if (hBitmap != NULL)
            {
            if (!SbsDrawBitmap (hDC, hBitmap, SBS_RECT (r)))
                {
                fOk = FALSE;
                }
            }
        else
            {
            if (!SbsDrawRectangle (hDC, crArea, SBS_RECT (r), TRUE))
                {
                fOk = FALSE;
                }
            }
        if (hIcon != NULL)
            {
            dWidth  = r.right  - r.left;
            dHeight = r.bottom - r.top;

            dX = (dWidth  > ICON_PIXELS
                  ? (dWidth  - ICON_PIXELS) / 2
                  : 0);

            dY = (dHeight > ICON_PIXELS
                  ? (dHeight - ICON_PIXELS) / 2
                  : 0);

            dX = min (dX, dY);

            if (!DrawIcon (hDC, r.left + dX, r.top + dY, hIcon))
                {
                fOk = FALSE;
                }
            dLeft = dX + ICON_PIXELS + (fCenter ? 0 : dX);
            }
        else
            {
            dLeft = 0;
            }
        if (!SbsDrawTextW (hDC, pwText, &crText, SBS_RECT (r),
                           SbsDrawStyle (dFlags, dLeft, 0, 0, 0)))
            {
            fOk = FALSE;
            }
        }
    return fOk;
    }

// =================================================================
// BRUSH MANAGEMENT
// =================================================================

HBRUSH WINAPI SbsBrushDelete (HBRUSH hBrush)
    {
    if (hBrush != NULL) DeleteObject (hBrush);
    return NULL;
    }

// =================================================================
// FONT MANAGEMENT
// =================================================================

HFONT WINAPI SbsFontDelete (HFONT hFont)
    {
    if (hFont != NULL) DeleteObject (hFont);
    return NULL;
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontObjectA (PLOGFONTA plfA,
                             PBYTE     pbName,
                             LONG      lHeight,
                             BOOL      fBold,
                             BOOL      fItalic,
                             BOOL      fCreate)
    {
    LOGFONTA lfA;
    HFONT    hFont = NULL;

    lfA.lfHeight         = lHeight;
    lfA.lfWidth          = 0;
    lfA.lfEscapement     = 0;
    lfA.lfOrientation    = 0;
    lfA.lfWeight         = (fBold ? FW_BOLD : FW_NORMAL);
    lfA.lfItalic         = (fItalic != FALSE);
    lfA.lfUnderline      = FALSE;
    lfA.lfStrikeOut      = FALSE;
    lfA.lfOutPrecision   = OUT_DEFAULT_PRECIS;
    lfA.lfClipPrecision  = CLIP_DEFAULT_PRECIS;
    lfA.lfQuality        = DRAFT_QUALITY;
    lfA.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    lfA.lfCharSet = (!_tcompiA (pbName, "Terminal")
                     ? OEM_CHARSET
                     : (!_tcompiA (pbName, "Symbol")
                        ? SYMBOL_CHARSET
                        : ANSI_CHARSET));

    _tcopynA (lfA.lfFaceName, SAFESTR (pbName, "System"),
              LF_FACESIZE);

    if (fCreate)       hFont = CreateFontIndirectA (&lfA);
    if (plfA != NULL) *plfA  = lfA;
    return hFont;
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontObjectW (PLOGFONTW plfW,
                             PWORD     pwName,
                             LONG      lHeight,
                             BOOL      fBold,
                             BOOL      fItalic,
                             BOOL      fCreate)
    {
    LOGFONTW lfW;
    HFONT    hFont = NULL;

    lfW.lfHeight         = lHeight;
    lfW.lfWidth          = 0;
    lfW.lfEscapement     = 0;
    lfW.lfOrientation    = 0;
    lfW.lfWeight         = (fBold ? FW_BOLD : FW_NORMAL);
    lfW.lfItalic         = (fItalic != FALSE);
    lfW.lfUnderline      = FALSE;
    lfW.lfStrikeOut      = FALSE;
    lfW.lfOutPrecision   = OUT_DEFAULT_PRECIS;
    lfW.lfClipPrecision  = CLIP_DEFAULT_PRECIS;
    lfW.lfQuality        = DRAFT_QUALITY;
    lfW.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    lfW.lfCharSet = (!_tcompiW (pwName, L"Terminal")
                     ? OEM_CHARSET
                     : (!_tcompiW (pwName, L"Symbol")
                        ? SYMBOL_CHARSET
                        : ANSI_CHARSET));

    _tcopynW (lfW.lfFaceName, SAFESTR (pwName, L"System"),
              LF_FACESIZE);

    if (fCreate)       hFont = CreateFontIndirectW (&lfW);
    if (plfW != NULL) *plfW  = lfW;
    return hFont;
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontSystemA (PLOGFONTA plfA,
                             BOOL      fCreate)
    {
    return SbsFontObjectA (plfA, "System", -13,
                           TRUE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontSystemW (PLOGFONTW plfW,
                             BOOL      fCreate)
    {
    return SbsFontObjectW (plfW, L"System", -13,
                           TRUE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontFixedsysA (PLOGFONTA plfA,
                               BOOL      fCreate)
    {
    return SbsFontObjectA (plfA, "Fixedsys", -12,
                           FALSE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontFixedsysW (PLOGFONTW plfW,
                               BOOL      fCreate)
    {
    return SbsFontObjectW (plfW, L"Fixedsys", -12,
                           FALSE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontTerminalA (PLOGFONTA plfA,
                               BOOL      fCreate)
    {
    return SbsFontObjectA (plfA, "Terminal", -12,
                           FALSE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontTerminalW (PLOGFONTW plfW,
                               BOOL      fCreate)
    {
    return SbsFontObjectW (plfW, L"Terminal", -12,
                           FALSE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontSymbolA (PLOGFONTA plfA,
                             BOOL      fCreate)
    {
    return SbsFontObjectA (plfA, "Symbol", -13,
                           TRUE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontSymbolW (PLOGFONTW plfW,
                             BOOL      fCreate)
    {
    return SbsFontObjectW (plfW, L"Symbol", -13,
                           TRUE, FALSE, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontStandardA (PLOGFONTA plfA,
                               DWORD     dSize,
                               BOOL      fBold,
                               BOOL      fItalic,
                               BOOL      fNarrow,
                               BOOL      fCreate)
    {
    DWORD dSize1 = (dSize ? (dSize > 0x3FFFFFFF ? 72 : dSize) : 8);

    return SbsFontObjectA (plfA, (fNarrow ? "Tahoma" : "Verdana"),
                           0 - (((dSize1 * 4) + 1) / 3),
                           fBold, fItalic, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontStandardW (PLOGFONTW plfW,
                               DWORD     dSize,
                               BOOL      fBold,
                               BOOL      fItalic,
                               BOOL      fNarrow,
                               BOOL      fCreate)
    {
    DWORD dSize1 = (dSize ? (dSize > 0x3FFFFFFF ? 72 : dSize) : 8);

    return SbsFontObjectW (plfW, (fNarrow ? L"Tahoma" : L"Verdana"),
                           0 - (((dSize1 * 4) + 1) / 3),
                           fBold, fItalic, fCreate);
    }

// -----------------------------------------------------------------

HFONT WINAPI SbsFontDefault (DWORD dFont)
    {
    HFONT hFont = ghfDefault;

    switch (dFont)
        {
        case SBS_FONT_LARGE: hFont = ghfLarge; break;
        case SBS_FONT_SMALL: hFont = ghfSmall; break;
        case SBS_FONT_LIGHT: hFont = ghfLight; break;
        }
    return hFont;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsFontConvert (PLOGFONTA plfA,
                            PLOGFONTW plfW,
                            BOOL      fToAnsi)
    {
    BOOL fOk = FALSE;

    if ((plfA != NULL) && (plfW != NULL))
        {
        if (fToAnsi)
            {
            plfA->lfHeight         = plfW->lfHeight;
            plfA->lfWidth          = plfW->lfWidth;
            plfA->lfEscapement     = plfW->lfEscapement;
            plfA->lfOrientation    = plfW->lfOrientation;
            plfA->lfWeight         = plfW->lfWeight;
            plfA->lfItalic         = plfW->lfItalic;
            plfA->lfUnderline      = plfW->lfUnderline;
            plfA->lfStrikeOut      = plfW->lfStrikeOut;
            plfA->lfCharSet        = plfW->lfCharSet;
            plfA->lfOutPrecision   = plfW->lfOutPrecision;
            plfA->lfClipPrecision  = plfW->lfClipPrecision;
            plfA->lfQuality        = plfW->lfQuality;
            plfA->lfPitchAndFamily = plfW->lfPitchAndFamily;

            _tcopyW2A (plfA->lfFaceName,
                       plfW->lfFaceName, LF_FACESIZE);
            }
        else
            {
            plfW->lfHeight         = plfA->lfHeight;
            plfW->lfWidth          = plfA->lfWidth;
            plfW->lfEscapement     = plfA->lfEscapement;
            plfW->lfOrientation    = plfA->lfOrientation;
            plfW->lfWeight         = plfA->lfWeight;
            plfW->lfItalic         = plfA->lfItalic;
            plfW->lfUnderline      = plfA->lfUnderline;
            plfW->lfStrikeOut      = plfA->lfStrikeOut;
            plfW->lfCharSet        = plfA->lfCharSet;
            plfW->lfOutPrecision   = plfA->lfOutPrecision;
            plfW->lfClipPrecision  = plfA->lfClipPrecision;
            plfW->lfQuality        = plfA->lfQuality;
            plfW->lfPitchAndFamily = plfA->lfPitchAndFamily;

            _tcopyA2W (plfW->lfFaceName,
                       plfA->lfFaceName, LF_FACESIZE);
            }
        fOk = TRUE;
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsFontCodeA (HWND hWnd,
                          BOOL fFixed)
    {
    LOGFONTA    lfA;
    CHOOSEFONTA cfA;
    BYTE        abWeight [SBS_SIGNED32];
    PBYTE       pbWeight;
    PBYTE       pbData;
    BOOL        fOk = FALSE;

    SbsFontSystemA (&lfA, FALSE);

    cfA.lStructSize    = CHOOSEFONTA_;
    cfA.hInstance      = NULL;
    cfA.hwndOwner      = hWnd;
    cfA.hDC            = NULL;
    cfA.lpLogFont      = &lfA;
    cfA.rgbColors      = SBS_COLOR_BLACK;
    cfA.lCustData      = (LPARAM) NULL;
    cfA.lpfnHook       = NULL;
    cfA.lpTemplateName = NULL;
    cfA.lpszStyle      = NULL;
    cfA.nFontType      = SCREEN_FONTTYPE;
    cfA.nSizeMin       = 8;
    cfA.nSizeMax       = 72;

    cfA.Flags          = CF_FORCEFONTEXIST      |
                         CF_SCREENFONTS         |
                         CF_EFFECTS             |
                         CF_INITTOLOGFONTSTRUCT |
                         (fFixed ? CF_FIXEDPITCHONLY : 0);

    if (ChooseFontA (&cfA))
        {
        _sprintfA (abWeight, "%ld", lfA.lfWeight);

        switch (lfA.lfWeight)
            {
            case FW_DONTCARE:   pbWeight = "FW_DONTCARE";   break;
            case FW_THIN:       pbWeight = "FW_THIN";       break;
            case FW_EXTRALIGHT: pbWeight = "FW_EXTRALIGHT"; break;
            case FW_LIGHT:      pbWeight = "FW_LIGHT";      break;
            case FW_NORMAL:     pbWeight = "FW_NORMAL";     break;
            case FW_MEDIUM:     pbWeight = "FW_MEDIUM";     break;
            case FW_SEMIBOLD:   pbWeight = "FW_SEMIBOLD";   break;
            case FW_BOLD:       pbWeight = "FW_BOLD";       break;
            case FW_EXTRABOLD:  pbWeight = "FW_EXTRABOLD";  break;
            case FW_HEAVY:      pbWeight = "FW_HEAVY";      break;
            default:            pbWeight =  abWeight;       break;
            }
        pbData = _bprintfA (NULL,

        "\r\n"
        "LOGFONTA lfA;\r\n"
        "\r\n"
        "lfA.lfHeight         = %ld;\r\n"
        "lfA.lfWidth          = %ld;\r\n"
        "lfA.lfEscapement     = %ld;\r\n"
        "lfA.lfOrientation    = %ld;\r\n"
        "lfA.lfWeight         = %s;\r\n"
        "lfA.lfItalic         = %s;\r\n"
        "lfA.lfUnderline      = %s;\r\n"
        "lfA.lfStrikeOut      = %s;\r\n"
        "lfA.lfCharSet        = ANSI_CHARSET;\r\n"
        "lfA.lfOutPrecision   = OUT_DEFAULT_PRECIS;\r\n"
        "lfA.lfClipPrecision  = CLIP_DEFAULT_PRECIS;\r\n"
        "lfA.lfQuality        = DRAFT_QUALITY;\r\n"
        "lfA.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;\r\n"
        "\r\n"
        "_tcopynA (lfA.lfFaceName, \"%s\", LF_FACESIZE);\r\n",

        lfA.lfHeight,
        lfA.lfWidth,
        lfA.lfEscapement,
        lfA.lfOrientation,
        pbWeight,
        lfA.lfItalic    ? gabTrue : gabFalse,
        lfA.lfUnderline ? gabTrue : gabFalse,
        lfA.lfStrikeOut ? gabTrue : gabFalse,
        lfA.lfFaceName);

        if (pbData != NULL)
            {
            fOk = SbsClipboardCopyTextA (hWnd, pbData);
            _mfree (pbData);
            }
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsFontCodeW (HWND hWnd,
                          BOOL fFixed)
    {
    LOGFONTW    lfW;
    CHOOSEFONTW cfW;
    WORD        awWeight [SBS_SIGNED32];
    PWORD       pwWeight;
    PWORD       pwData;
    BOOL        fOk = FALSE;

    SbsFontSystemW (&lfW, FALSE);

    cfW.lStructSize    = CHOOSEFONTW_;
    cfW.hInstance      = NULL;
    cfW.hwndOwner      = hWnd;
    cfW.hDC            = NULL;
    cfW.lpLogFont      = &lfW;
    cfW.rgbColors      = SBS_COLOR_BLACK;
    cfW.lCustData      = (LPARAM) NULL;
    cfW.lpfnHook       = NULL;
    cfW.lpTemplateName = NULL;
    cfW.lpszStyle      = NULL;
    cfW.nFontType      = SCREEN_FONTTYPE;
    cfW.nSizeMin       = 8;
    cfW.nSizeMax       = 72;

    cfW.Flags          = CF_FORCEFONTEXIST      |
                         CF_SCREENFONTS         |
                         CF_EFFECTS             |
                         CF_INITTOLOGFONTSTRUCT |
                         (fFixed ? CF_FIXEDPITCHONLY : 0);

    if (ChooseFontW (&cfW))
        {
        _sprintfW (awWeight, L"%ld", lfW.lfWeight);

        switch (lfW.lfWeight)
            {
            case FW_DONTCARE:   pwWeight = L"FW_DONTCARE";   break;
            case FW_THIN:       pwWeight = L"FW_THIN";       break;
            case FW_EXTRALIGHT: pwWeight = L"FW_EXTRALIGHT"; break;
            case FW_LIGHT:      pwWeight = L"FW_LIGHT";      break;
            case FW_NORMAL:     pwWeight = L"FW_NORMAL";     break;
            case FW_MEDIUM:     pwWeight = L"FW_MEDIUM";     break;
            case FW_SEMIBOLD:   pwWeight = L"FW_SEMIBOLD";   break;
            case FW_BOLD:       pwWeight = L"FW_BOLD";       break;
            case FW_EXTRABOLD:  pwWeight = L"FW_EXTRABOLD";  break;
            case FW_HEAVY:      pwWeight = L"FW_HEAVY";      break;
            default:            pwWeight =   awWeight;       break;
            }
        pwData = _bprintfW (NULL,

        L"\r\n"
        L"LOGFONTW lfW;\r\n"
        L"\r\n"
        L"lfW.lfHeight         = %ld;\r\n"
        L"lfW.lfWidth          = %ld;\r\n"
        L"lfW.lfEscapement     = %ld;\r\n"
        L"lfW.lfOrientation    = %ld;\r\n"
        L"lfW.lfWeight         = %s;\r\n"
        L"lfW.lfItalic         = %s;\r\n"
        L"lfW.lfUnderline      = %s;\r\n"
        L"lfW.lfStrikeOut      = %s;\r\n"
        L"lfW.lfCharSet        = ANSI_CHARSET;\r\n"
        L"lfW.lfOutPrecision   = OUT_DEFAULT_PRECIS;\r\n"
        L"lfW.lfClipPrecision  = CLIP_DEFAULT_PRECIS;\r\n"
        L"lfW.lfQuality        = DRAFT_QUALITY;\r\n"
        L"lfW.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;\r\n"
        L"\r\n"
        L"_tcopynW (lfW.lfFaceName, L\"%s\", LF_FACESIZE);\r\n",

        lfW.lfHeight,
        lfW.lfWidth,
        lfW.lfEscapement,
        lfW.lfOrientation,
        pwWeight,
        lfW.lfItalic    ? gawTrue : gawFalse,
        lfW.lfUnderline ? gawTrue : gawFalse,
        lfW.lfStrikeOut ? gawTrue : gawFalse,
        lfW.lfFaceName);

        if (pwData != NULL)
            {
            fOk = SbsClipboardCopyTextW (hWnd, pwData);
            _mfree (pwData);
            }
        }
    return fOk;
    }

// =================================================================
// DATE/TIME MANAGEMENT
// =================================================================

VOID WINAPI SbsTimeNow (PSBS_TIME pst)
    {
    SYSTEMTIME st;

    GetLocalTime (&st);

    pst->wYear      = (WORD) st.wYear;
    pst->bMonth     = (BYTE) st.wMonth;
    pst->bDay       = (BYTE) st.wDay;
    pst->bHour      = (BYTE) st.wHour;
    pst->bMinute    = (BYTE) st.wMinute;
    pst->bSecond    = (BYTE) st.wSecond;
    pst->bDayOfWeek =
        (BYTE) (((((SbsTimePack (pst) / 60) / 60) / 24) + 4) % 7);

    return;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsTimePack (PSBS_TIME pst)
    {
    DWORD dYear, dMonth, dDay, i;
    DWORD dTime = 0;

    if (pst->wYear >= 1970)
        {
        dMonth = (pst->bMonth ? pst->bMonth-1 : 0);

        dYear  = (DWORD) pst->wYear - 1600 + (dMonth / 12);
        dMonth = dMonth % 12;
        dDay   = (dYear / 400) * DAYS_PER_400_YEARS;

        for (i = 0; i < dMonth; i++)
            {
            dDay += adDaysPerMonth [i];
            }
        if ((dMonth > 1) && (!(dYear % 4)) &&
            ((dYear % 100) || (!(dYear % 400))))
            {
            dDay += 1;
            }
        if (pst->bDay)
            {
            dDay += pst->bDay - 1;
            }
        if (dYear = dYear % 400)
            {
            dDay += ((dYear / 100) * (DAYS_PER_100_YEARS-1)) + 1;

            if (dYear = dYear % 100)
                {
                dDay += ((dYear / 4) * DAYS_PER_4_YEARS) - 1;

                if (dYear = dYear % 4)
                    {
                    dDay += (dYear * DAYS_PER_YEAR) + 1;
                    }
                }
            }
        dDay -= DAYS_1600_TO_1970;

        dTime  = (dDay                 * 60 * 60 * 24) +
                 ((DWORD) pst->bHour   * 60 * 60     ) +
                 ((DWORD) pst->bMinute * 60          ) +
                 ((DWORD) pst->bSecond               );
        }
    return dTime;
    }

// -----------------------------------------------------------------

VOID WINAPI SbsTimeUnpack (PSBS_TIME pst,
                           DWORD     dTime)
    {
    DWORD dDaySince1600, dDayIn100Years, dDayIn4Years;
    DWORD d400Years, d100In400Years, d4In100Years, d1In4Years;
    DWORD dDay, dMonth, dYear, dDaysPerMonth;
    BOOL  fLeap;

    pst->bSecond   = (BYTE) ((dTime        ) % 60);
    pst->bMinute   = (BYTE) ((dTime /    60) % 60);
    pst->bHour     = (BYTE) ((dTime /  3600) % 24);
    dDaySince1600  =         (dTime / 86400) + DAYS_1600_TO_1970;

    dDayIn100Years = dDaySince1600 % DAYS_PER_400_YEARS;
    d400Years      = dDaySince1600 / DAYS_PER_400_YEARS;
    d100In400Years = 0;

    while (dDayIn100Years >= DAYS_PER_100_YEARS)
        {
        dDayIn100Years -= (DAYS_PER_100_YEARS-1);
        d100In400Years++;
        }
    dDayIn4Years = dDayIn100Years % DAYS_PER_4_YEARS;
    d4In100Years = dDayIn100Years / DAYS_PER_4_YEARS;

    if (dDayIn4Years)
        {
        dDay       = (dDayIn4Years-1) % DAYS_PER_YEAR;
        d1In4Years = (dDayIn4Years-1) / DAYS_PER_YEAR;
        if (!d1In4Years) dDay++;
        }
    else
        {
        dDay       = 0;
        d1In4Years = 0;
        }
    fLeap = (d1In4Years ? FALSE
                        : (d4In100Years ? TRUE
                                        : !d100In400Years));

    for (dMonth = 0; dMonth < 12; dMonth++)
        {
        dDaysPerMonth = adDaysPerMonth [dMonth];
        if ((dMonth == 2) && fLeap) dDaysPerMonth++;
        if (dDay < dDaysPerMonth) break;
        dDay -= dDaysPerMonth;
        }
    dYear = (d1In4Years          ) +
            (d4In100Years   *   4) +
            (d100In400Years * 100) +
            (d400Years      * 400) + 1600;

    pst->bDay       = (BYTE) (dDay + 1);
    pst->bMonth     = (BYTE) (dMonth + 1);
    pst->wYear      = (WORD) dYear;
    pst->bDayOfWeek = (BYTE) ((dDaySince1600 + 6) % 7);
    return;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsTimeDayOfYear (DWORD dDay,
                               DWORD dMonth,
                               DWORD dYear)
    {
    DWORD dYear1, dMonth1, dDay1, i;
    DWORD dDayOfYear = 0;

    dDay1   = (dDay   ? dDay   - 1 : 0);
    dMonth1 = (dMonth ? dMonth - 1 : 0);
    dYear1  = dYear + (dMonth1 / 12);
    dMonth1 = dMonth1 % 12;

    for (i = 0; i < dMonth1; i++)
        {
        dDayOfYear += adDaysPerMonth [i];
        }
    if ((dMonth1 > 1) && (!(dYear1 % 4)) &&
        ((dYear1 % 100) || (!(dYear1 % 400))))
        {
        dDayOfYear += 1;
        }
    dDayOfYear += dDay1;
    return dDayOfYear;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsTimeTextA (PSYSTEMTIME pst,
                           BOOL        fGerman)
    {
    PBYTE pbData = NULL;

    if (pst != NULL)
        {
        pbData = _bprintfA (NULL,
                            (fGerman ? gabTimeDE   : gabTimeUS),
                            (fGerman ? pst->wDay   : pst->wMonth),
                            (fGerman ? pst->wMonth : pst->wDay),
                            pst->wYear,
                            pst->wHour,
                            pst->wMinute,
                            pst->wSecond);
        }
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsTimeTextW (PSYSTEMTIME pst,
                           BOOL        fGerman)
    {
    PWORD pwData = NULL;

    if (pst != NULL)
        {
        pwData = _bprintfW (NULL,
                            (fGerman ? gawTimeDE   : gawTimeUS),
                            (fGerman ? pst->wDay   : pst->wMonth),
                            (fGerman ? pst->wMonth : pst->wDay),
                            pst->wYear,
                            pst->wHour,
                            pst->wMinute,
                            pst->wSecond);
        }
    return pwData;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsTimeTextUtcA (PFILETIME pft,
                              BOOL      fGerman)
    {
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToSystemTime (pft, &st))
        {
        pst = &st;
        }
    return SbsTimeTextA (pst, fGerman);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsTimeTextUtcW (PFILETIME pft,
                              BOOL      fGerman)
    {
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToSystemTime (pft, &st))
        {
        pst = &st;
        }
    return SbsTimeTextW (pst, fGerman);
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsTimeTextLocalA (PFILETIME pft,
                                BOOL      fGerman)
    {
    FILETIME    ft;
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToLocalFileTime (pft, &ft)
                      && FileTimeToSystemTime    (&ft, &st))
        {
        pst = &st;
        }
    return SbsTimeTextA (pst, fGerman);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsTimeTextLocalW (PFILETIME pft,
                                BOOL      fGerman)
    {
    FILETIME    ft;
    SYSTEMTIME  st;
    PSYSTEMTIME pst = NULL;

    if ((pft != NULL) && FileTimeToLocalFileTime (pft, &ft)
                      && FileTimeToSystemTime    (&ft, &st))
        {
        pst = &st;
        }
    return SbsTimeTextW (pst, fGerman);
    }

// -----------------------------------------------------------------

PVOID WINAPI SbsTimeTextEx (PVOID pTime,
                            BOOL  fGerman,
                            BOOL  fFileTime,
                            BOOL  fLocal,
                            BOOL  fUnicode)
    {
    return (fUnicode

            ? (PVOID) (fFileTime
                       ? (fLocal
                          ? SbsTimeTextLocalW (pTime, fGerman)
                          : SbsTimeTextUtcW   (pTime, fGerman))
                       : SbsTimeTextW (pTime, fGerman))

            : (PVOID) (fFileTime
                       ? (fLocal
                          ? SbsTimeTextLocalA (pTime, fGerman)
                          : SbsTimeTextUtcA   (pTime, fGerman))
                       : SbsTimeTextA (pTime, fGerman)));
    }

// =================================================================
// GUID MANAGEMENT
// =================================================================

PBYTE WINAPI SbsGuidTextA (PGUID pGuid)
    {
    GUID Guid = GUID_NULL;

    if (pGuid != NULL) Guid = *pGuid;

    return _bprintfA (NULL, gabGuid, (DWORD) Guid.Data1,
                                     (DWORD) Guid.Data2,
                                     (DWORD) Guid.Data3,
                                     (DWORD) Guid.Data4 [0],
                                     (DWORD) Guid.Data4 [1],
                                     (DWORD) Guid.Data4 [2],
                                     (DWORD) Guid.Data4 [3],
                                     (DWORD) Guid.Data4 [4],
                                     (DWORD) Guid.Data4 [5],
                                     (DWORD) Guid.Data4 [6],
                                     (DWORD) Guid.Data4 [7]);
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsGuidTextW (PGUID pGuid)
    {
    GUID Guid = GUID_NULL;

    if (pGuid != NULL) Guid = *pGuid;

    return _bprintfW (NULL, gawGuid, (DWORD) Guid.Data1,
                                     (DWORD) Guid.Data2,
                                     (DWORD) Guid.Data3,
                                     (DWORD) Guid.Data4 [0],
                                     (DWORD) Guid.Data4 [1],
                                     (DWORD) Guid.Data4 [2],
                                     (DWORD) Guid.Data4 [3],
                                     (DWORD) Guid.Data4 [4],
                                     (DWORD) Guid.Data4 [5],
                                     (DWORD) Guid.Data4 [6],
                                     (DWORD) Guid.Data4 [7]);
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsGuidTextExA (PGUID pGuid,
                             PBYTE pbFormat)
    {
    PBYTE pbGuid;
    PBYTE pbData = NULL;

    if ((pbGuid = SbsGuidTextA (pGuid)) != NULL)
        {
        pbData = _bprintfA (NULL, SAFEPTR (pbFormat, gabString),
                            pbGuid);
        _mfree (pbGuid);
        }
    return pbData;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsGuidTextExW (PGUID pGuid,
                             PWORD pwFormat)
    {
    PWORD pwGuid;
    PWORD pwData = NULL;

    if ((pwGuid = SbsGuidTextW (pGuid)) != NULL)
        {
        pwData = _bprintfW (NULL, SAFEPTR (pwFormat, gawString),
                            pwGuid);
        _mfree (pwGuid);
        }
    return pwData;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidBinaryA (PBYTE pbGuid,
                            PGUID pGuid)
    {
    DWORD i, n;
    GUID  Guid1;
    GUID  Guid = GUID_NULL;
    BOOL  fOk  = (pbGuid == NULL);

    if (!fOk)
        {
        for (i = 0; pbGuid [i] && (pbGuid [i] != '{'); i++);

        if (pbGuid [i++] == '{')
            {
            i += (n = SbsHexDwordA (pbGuid+i,
                                    &Guid1.Data1));

            if ((n == DWORD_ * 2) && (pbGuid [i++] == '-'))
                {
                i += (n = SbsHexWordA (pbGuid+i,
                                       &Guid1.Data2));

                if ((n == WORD_ * 2) && (pbGuid [i++] == '-'))
                    {
                    i += (n = SbsHexWordA (pbGuid+i,
                                           &Guid1.Data3));

                    if ((n == WORD_ * 2) && (pbGuid [i++] == '-'))
                        {
                        i += (n = SbsHexBytesA (pbGuid+i,
                                                Guid1.Data4,
                                                2));

                        if ((n == 2*2) && (pbGuid [i++] == '-'))
                            {
                            i += (n = SbsHexBytesA (pbGuid+i,
                                                    Guid1.Data4+2,
                                                    6));

                            if ((n == 6*2) && (pbGuid [i++] == '}'))
                                {
                                Guid = Guid1;
                                fOk  = TRUE;
                                }
                            }
                        }
                    }
                }
            }
        }
    if (pGuid != NULL) *pGuid = Guid;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidBinaryW (PWORD pwGuid,
                            PGUID pGuid)
    {
    DWORD i, n;
    GUID  Guid1;
    GUID  Guid = GUID_NULL;
    BOOL  fOk  = (pwGuid == NULL);

    if (!fOk)
        {
        for (i = 0; pwGuid [i] && (pwGuid [i] != '{'); i++);

        if (pwGuid [i++] == '{')
            {
            i += (n = SbsHexDwordW (pwGuid+i,
                                    &Guid1.Data1));

            if ((n == DWORD_ * 2) && (pwGuid [i++] == '-'))
                {
                i += (n = SbsHexWordW (pwGuid+i,
                                       &Guid1.Data2));

                if ((n == WORD_ * 2) && (pwGuid [i++] == '-'))
                    {
                    i += (n = SbsHexWordW (pwGuid+i,
                                           &Guid1.Data3));

                    if ((n == WORD_ * 2) && (pwGuid [i++] == '-'))
                        {
                        i += (n = SbsHexBytesW (pwGuid+i,
                                                Guid1.Data4,
                                                2));

                        if ((n == 2*2) && (pwGuid [i++] == '-'))
                            {
                            i += (n = SbsHexBytesW (pwGuid+i,
                                                    Guid1.Data4+2,
                                                    6));

                            if ((n == 6*2) && (pwGuid [i++] == '}'))
                                {
                                Guid = Guid1;
                                fOk  = TRUE;
                                }
                            }
                        }
                    }
                }
            }
        }
    if (pGuid != NULL) *pGuid = Guid;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidValueReadA (PGUID pGuid,
                               PBYTE pbFormat,
                               PBYTE pbItem)
    {
    PBYTE pbGuid;
    GUID  Guid = GUID_NULL;
    BOOL  fOk  = FALSE;

    if ((pbFormat != NULL)
        &&
        ((pbGuid = SbsSoftwareReadTextA (pbFormat, NULL, NULL, NULL,
                                         pbItem, NULL))
         != NULL))
        {
        fOk = SbsGuidBinaryA (pbGuid, &Guid);
        _mfree (pbGuid);
        }
    if (pGuid != NULL) *pGuid = Guid;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidValueReadW (PGUID pGuid,
                               PWORD pwFormat,
                               PWORD pwItem)
    {
    PWORD pwGuid;
    GUID  Guid = GUID_NULL;
    BOOL  fOk  = FALSE;

    if ((pwFormat != NULL)
        &&
        ((pwGuid = SbsSoftwareReadTextW (pwFormat, NULL, NULL, NULL,
                                         pwItem, NULL))
         != NULL))
        {
        fOk = SbsGuidBinaryW (pwGuid, &Guid);
        _mfree (pwGuid);
        }
    if (pGuid != NULL) *pGuid = Guid;
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidValueWriteA (PGUID pGuid,
                                PBYTE pbFormat,
                                PBYTE pbItem)
    {
    PBYTE pbGuid;
    BOOL  fOk = FALSE;

    if ((pbFormat != NULL) &&
        ((pbGuid = SbsGuidTextA (pGuid)) != NULL))
        {
        fOk = SbsSoftwareWriteTextA (pbFormat, NULL, NULL, NULL,
                                     pbItem, pbGuid);
        _mfree (pbGuid);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidValueWriteW (PGUID pGuid,
                                PWORD pwFormat,
                                PWORD pwItem)
    {
    PWORD pwGuid;
    BOOL  fOk = FALSE;

    if ((pwFormat != NULL) &&
        ((pwGuid = SbsGuidTextW (pGuid)) != NULL))
        {
        fOk = SbsSoftwareWriteTextW (pwFormat, NULL, NULL, NULL,
                                     pwItem, pwGuid);
        _mfree (pwGuid);
        }
    return fOk;
    }

// -----------------------------------------------------------------

PBYTE WINAPI SbsGuidReadTextA (PGUID pGuid,
                               PBYTE pbFormat)
    {
    PBYTE pbValue;
    PBYTE pbName = NULL;

    if ((pbFormat != NULL) &&
        ((pbValue = SbsGuidTextExA (pGuid, pbFormat)) != NULL))
        {
        pbName = SbsRegistryReadTextA (HKEY_NULL, pbValue, NULL);
        _mfree (pbValue);
        }
    return pbName;
    }

// -----------------------------------------------------------------

PWORD WINAPI SbsGuidReadTextW (PGUID pGuid,
                               PWORD pwFormat)
    {
    PWORD pwValue;
    PWORD pwName = NULL;

    if ((pwFormat != NULL) &&
        ((pwValue = SbsGuidTextExW (pGuid, pwFormat)) != NULL))
        {
        pwName = SbsRegistryReadTextW (HKEY_NULL, pwValue, NULL);
        _mfree (pwValue);
        }
    return pwName;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidWriteTextA (PGUID pGuid,
                               PBYTE pbFormat,
                               PBYTE pbData)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbFormat != NULL) && (pbData != NULL) &&
        ((pbValue = SbsGuidTextExA (pGuid, pbFormat)) != NULL))
        {
        fOk = SbsRegistryWriteTextA (HKEY_NULL, pbValue, pbData);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidWriteTextW (PGUID pGuid,
                               PWORD pwFormat,
                               PWORD pwData)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwFormat != NULL) && (pwData != NULL) &&
        ((pwValue = SbsGuidTextExW (pGuid, pwFormat)) != NULL))
        {
        fOk = SbsRegistryWriteTextW (HKEY_NULL, pwValue, pwData);
        _mfree (pwValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidDeleteKeyA (PGUID pGuid,
                               PBYTE pbFormat)
    {
    PBYTE pbPath;
    BOOL  fOk = FALSE;

    if ((pbFormat != NULL) &&
        ((pbPath = SbsGuidTextExA (pGuid, pbFormat)) != NULL))
        {
        fOk = SbsRegistryPathDeleteA (pbPath);
        _mfree (pbPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidDeleteKeyW (PGUID pGuid,
                               PWORD pwFormat)
    {
    PWORD pwPath;
    BOOL  fOk = FALSE;

    if ((pwFormat != NULL) &&
        ((pwPath = SbsGuidTextExW (pGuid, pwFormat)) != NULL))
        {
        fOk = SbsRegistryPathDeleteW (pwPath);
        _mfree (pwPath);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidDeleteValueA (PGUID pGuid,
                                 PBYTE pbFormat)
    {
    PBYTE pbValue;
    BOOL  fOk = FALSE;

    if ((pbFormat != NULL) &&
        ((pbValue = SbsGuidTextExA (pGuid, pbFormat)) != NULL))
        {
        fOk = SbsRegistryDeleteValueA (HKEY_NULL, pbValue);
        _mfree (pbValue);
        }
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsGuidDeleteValueW (PGUID pGuid,
                                 PWORD pwFormat)
    {
    PWORD pwValue;
    BOOL  fOk = FALSE;

    if ((pwFormat != NULL) &&
        ((pwValue = SbsGuidTextExW (pGuid, pwFormat)) != NULL))
        {
        fOk = SbsRegistryDeleteValueW (HKEY_NULL, pwValue);
        _mfree (pwValue);
        }
    return fOk;
    }

// =================================================================
// CRC32 COMPUTATION
// =================================================================

DWORD WINAPI SbsCrc32Start (PDWORD pdCrc32)
    {
    return _crc32_start (pdCrc32);
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsCrc32Stop (PDWORD pdCrc32)
    {
    return _crc32_stop (pdCrc32);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32Byte (PDWORD pdCrc32,
                           BYTE   bData)
    {
    return _crc32_08 (pdCrc32, bData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32Word (PDWORD pdCrc32,
                           WORD   wData)
    {
    return _crc32_16 (pdCrc32, wData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32Dword (PDWORD pdCrc32,
                            DWORD  dData)
    {
    return _crc32_32 (pdCrc32, dData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32CharA (PDWORD pdCrc32,
                            BYTE   bData)
    {
    return _crc32_16 (pdCrc32, bData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32CharW (PDWORD pdCrc32,
                            WORD   wData)
    {
    return _crc32_16 (pdCrc32, wData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32Multi (PDWORD pdCrc32,
                            PVOID  pData,
                            DWORD  dData)
    {
    return _crc32 (pdCrc32, pData, dData);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32TextA (PDWORD pdCrc32,
                            PBYTE  pbData,
                            DWORD  dData)
    {
    DWORD dCrc32, dData1, i;

    if (pdCrc32 != NULL) dCrc32 = *pdCrc32;
    else _crc32_start (&dCrc32);

    if (pbData != NULL)
        {
        dData1 = SbsTextDataA (pbData, dData);

        for (i = 0; i < dData1; i++)
            {
            _crc32_16 (&dCrc32, pbData [i]);
            }
        }
    if (pdCrc32 != NULL) *pdCrc32 = dCrc32;
    else _crc32_stop (&dCrc32);

    return dCrc32;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsCrc32TextW (PDWORD pdCrc32,
                            PWORD  pwData,
                            DWORD  dData)
    {
    return _crc32 (pdCrc32, pwData,
                   SbsTextDataW (pwData, dData) * WORD_);
    }

// =================================================================
// PSEUDO-RANDOM NUMBER GENERATOR
// =================================================================

DWORD WINAPI SbsRandomInitialize (PDWORD pdData,
                                  PDWORD pdAddr,
                                  DWORD  dSeed)
    {
    return _random_seed (pdData, pdAddr, dSeed);
    }

// -----------------------------------------------------------------

BYTE WINAPI SbsRandomByte (PDWORD pdData,
                           PDWORD pdAddr)
    {
    return _random_08 (pdData, pdAddr);
    }

// -----------------------------------------------------------------

WORD WINAPI SbsRandomWord (PDWORD pdData,
                           PDWORD pdAddr)
    {
    return _random_16 (pdData, pdAddr);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsRandomDword (PDWORD pdData,
                             PDWORD pdAddr)
    {
    return _random_32 (pdData, pdAddr);
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsRandomTextA (PDWORD pdData,
                             PDWORD pdAddr,
                             PBYTE  pbBuffer,
                             DWORD  dBuffer,
                             DWORD  dNumeric)
    {
    DWORD dText, dVowel, dVowels, dConsonants, i;
    DWORD n = 0;

    if ((pbBuffer != NULL) && dBuffer)
        {
        dText       = dBuffer-1;
        dText       = (dText > dNumeric ? dText - dNumeric : 0);
        dVowel      = (dText > 1 ? dText & 1 : 0);
        dVowels     = _tsizeA (gabVowels    );
        dConsonants = _tsizeA (gabConsonants);

        while (n < dBuffer-1)
            {
            i = _random_32 (pdData, pdAddr);

            pbBuffer [n++] = (n >= dText
                              ? (BYTE) ((i % 10) + '0')
                              : (dVowel
                                 ? gabVowels    [i % dVowels    ]
                                 : gabConsonants[i % dConsonants]));
            dVowel ^= 1;
            }
        pbBuffer [n] = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsRandomTextW (PDWORD pdData,
                             PDWORD pdAddr,
                             PWORD  pwBuffer,
                             DWORD  dBuffer,
                             DWORD  dNumeric)
    {
    DWORD dText, dVowel, dVowels, dConsonants, i;
    DWORD n = 0;

    if ((pwBuffer != NULL) && dBuffer)
        {
        dText       = dBuffer-1;
        dText       = (dText > dNumeric ? dText - dNumeric : 0);
        dVowel      = (dText > 1 ? dText & 1 : 0);
        dVowels     = _tsizeW (gawVowels    );
        dConsonants = _tsizeW (gawConsonants);

        while (n < dBuffer-1)
            {
            i = _random_32 (pdData, pdAddr);

            pwBuffer [n++] = (n >= dText
                              ? (WORD) ((i % 10) + '0')
                              : (dVowel
                                 ? gawVowels    [i % dVowels    ]
                                 : gawConsonants[i % dConsonants]));
            dVowel ^= 1;
            }
        pwBuffer [n] = 0;
        }
    return n;
    }

// =================================================================
// SERVICE CONTROL MANAGER
// =================================================================

HSCM WINAPI SbsServiceConnect (VOID)
    {
    HSCM hScm = NULL;

    if ((hScm = _mnew (SBS_SCM_)) != NULL)
        {
        if ((hScm->hModule = LoadLibraryW (L"ADVAPI32.dll"))
            != NULL)
            {
            if (((hScm->CloseServiceHandle =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "CloseServiceHandle"))
                 == NULL)
                ||
                ((hScm->ControlService =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "ControlService"))
                 == NULL)
                ||
                ((hScm->CreateService =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "CreateServiceW"))
                 == NULL)
                ||
                ((hScm->DeleteService =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "DeleteService"))
                 == NULL)
                ||
                ((hScm->EnumServicesStatus =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "EnumServicesStatusW"))
                 == NULL)
                ||
                ((hScm->OpenSCManager =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "OpenSCManagerW"))
                 == NULL)
                ||
                ((hScm->OpenService =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "OpenServiceW"))
                 == NULL)
                ||
                ((hScm->QueryServiceStatus =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "QueryServiceStatus"))
                 == NULL)
                ||
                ((hScm->StartService =
                    (PVOID) GetProcAddress (hScm->hModule,
                                            "StartServiceW"))
                 == NULL)
                ||
                ((hScm->hManager =
                    hScm->OpenSCManager (NULL, NULL,
                                         SC_MANAGER_ALL_ACCESS))
                 == NULL))
                {
                FreeLibrary (hScm->hModule);
                hScm->hModule = NULL;
                }                
            }
        if (hScm->hModule == NULL) hScm = _mfree (hScm);
        }
    return hScm;
    }

// -----------------------------------------------------------------

HSCM WINAPI SbsServiceDisconnect (HSCM hScm)
    {
    if (hScm != NULL)
        {
        hScm->CloseServiceHandle (hScm->hManager);
        FreeLibrary              (hScm->hModule);
        _mfree                   (hScm);
        }
    return NULL;
    }

// -----------------------------------------------------------------

HSCM WINAPI SbsServiceManager (HSCM hPermanent,
                               HSCM hTransient)
    {
    return (hTransient == NULL
            ? SAFEPTR (hPermanent, SbsServiceConnect ())
            : (hPermanent == NULL
               ? SbsServiceDisconnect (hTransient)
               : NULL));
    }

// -----------------------------------------------------------------

SC_HANDLE WINAPI SbsServiceOpen (HSCM  hScm,
                                 PWORD pwName)
    {
    HSCM      hScm1    = SbsServiceManager (hScm, NULL);
    SC_HANDLE hService = NULL;

    if ((hScm1 != NULL) && (pwName != NULL))
        {
        hService = hScm1->OpenService (hScm1->hManager, pwName,
                                       SERVICE_ALL_ACCESS);
        }
    SbsServiceManager (hScm, hScm1);
    return hService;
    }

// -----------------------------------------------------------------

SC_HANDLE WINAPI SbsServiceClose (HSCM      hScm,
                                  SC_HANDLE hService)
    {
    HSCM hScm1 = SbsServiceManager (hScm, NULL);

    if ((hScm1 != NULL) && (hService != NULL))
        {
        hScm1->CloseServiceHandle (hService);
        }
    SbsServiceManager (hScm, hScm1);
    return NULL;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceLoad (HSCM  hScm,
                            PWORD pwName,
                            PWORD pwInfo,
                            PWORD pwPath,
                            DWORD dType,
                            DWORD dStart,
                            BOOL  fStart)
    {
    SC_HANDLE hService;
    PWORD     pwFile, pwName1, pwInfo1;
    WORD      awPath [MAX_PATH];
    DWORD     n;
    HSCM      hScm1 = SbsServiceManager (hScm, NULL);
    BOOL      fOk   = FALSE;

    if ((hScm1 != NULL) && (pwPath != NULL) &&
        (n = GetFullPathNameW (pwPath, MAX_PATH, awPath, &pwFile))&&
        (n < MAX_PATH))
        {
        pwName1 = SAFESTR (pwName, _ffileW (awPath));
        pwInfo1 = SAFESTR (pwInfo, awPath);

        if ((hService = hScm1->CreateService (hScm1->hManager,
                                              pwName1, pwInfo1,
                                              SERVICE_ALL_ACCESS,
                                              dType, dStart,
                                              SERVICE_ERROR_NORMAL,
                                              awPath, NULL, NULL,
                                              NULL, NULL, NULL))
            != NULL)
            {
            hScm1->CloseServiceHandle (hService);
            fOk = TRUE;
            }
        else
            {
            fOk = (GetLastError() == ERROR_SERVICE_EXISTS);
            }
        if (fOk && fStart)
            {
            fOk =     SbsServiceStart  (hScm1, pwName1);
            if (!fOk) SbsServiceUnload (hScm1, pwName1);
            }
        }
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceUnload (HSCM  hScm,
                              PWORD pwName)
    {
    SC_HANDLE hService;
    HSCM      hScm1 = SbsServiceManager (hScm, NULL);
    BOOL      fOk   = FALSE;

    if ((hService = SbsServiceOpen (hScm1, pwName)) != NULL)
        {
        SbsServiceStop (hScm1, pwName);

        fOk = hScm1->DeleteService (hService) ||
              (GetLastError() == ERROR_SERVICE_MARKED_FOR_DELETE);

        hScm1->CloseServiceHandle (hService);
        }
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceStart (HSCM  hScm,
                             PWORD pwName)
    {
    SC_HANDLE hService;
    HSCM      hScm1 = SbsServiceManager (hScm, NULL);
    BOOL      fOk   = FALSE;

    if ((hService = SbsServiceOpen (hScm1, pwName)) != NULL)
        {
        fOk = hScm1->StartService (hService, 1, &pwName) ||
              (GetLastError () == ERROR_SERVICE_ALREADY_RUNNING);

        hScm1->CloseServiceHandle (hService);
        }
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceControl (HSCM  hScm,
                               PWORD pwName,
                               DWORD dControl)
    {
    SC_HANDLE      hService;
    SERVICE_STATUS ServiceStatus;
    HSCM           hScm1 = SbsServiceManager (hScm, NULL);
    BOOL           fOk   = FALSE;

    if ((hService = SbsServiceOpen (hScm1, pwName)) != NULL)
        {
        if (hScm1->QueryServiceStatus (hService, &ServiceStatus))
            {
            switch (ServiceStatus.dwCurrentState)
                {
                case SERVICE_STOP_PENDING:
                case SERVICE_STOPPED:
                    {
                    fOk = (dControl == SERVICE_CONTROL_STOP);
                    break;
                    }
                case SERVICE_PAUSE_PENDING:
                case SERVICE_PAUSED:
                    {
                    fOk = (dControl == SERVICE_CONTROL_PAUSE);
                    break;
                    }
                case SERVICE_START_PENDING:
                case SERVICE_CONTINUE_PENDING:
                case SERVICE_RUNNING:
                    {
                    fOk = (dControl == SERVICE_CONTROL_CONTINUE);
                    break;
                    }
                }
            }
        if (!fOk)
            {
            fOk = hScm1->ControlService (hService, dControl,
                                         &ServiceStatus);
            }
        hScm1->CloseServiceHandle (hService);
        }
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceStop (HSCM  hScm,
                            PWORD pwName)
    {
    HSCM hScm1 = SbsServiceManager (hScm, NULL);
    BOOL fOk   = SbsServiceControl (hScm1, pwName,
                                    SERVICE_CONTROL_STOP);
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServicePause (HSCM  hScm,
                             PWORD pwName)
    {
    HSCM hScm1 = SbsServiceManager (hScm, NULL);
    BOOL fOk   = SbsServiceControl (hScm1, pwName,
                                    SERVICE_CONTROL_PAUSE);
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// -----------------------------------------------------------------

BOOL WINAPI SbsServiceContinue (HSCM  hScm,
                                PWORD pwName)
    {
    HSCM hScm1 = SbsServiceManager (hScm, NULL);
    BOOL fOk   = SbsServiceControl (hScm1, pwName,
                                    SERVICE_CONTROL_CONTINUE);
    SbsServiceManager (hScm, hScm1);
    return fOk;
    }

// =================================================================
// VERSION TESTING
// =================================================================

DWORD WINAPI SbsRtlVersion (VOID)
    {
    return SBS_RTL_VERSION;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsRtlVersionTestA (DWORD        dRequired,
                                 PSBS_VERSION psv)
    {
    SBS_VERSION sv;

    sv.dRequired  = dRequired;
    sv.dInstalled = SbsRtlVersion ();
    sv.pbFilename = gabFilename;

    if (sv.dRequired <= sv.dInstalled)
        {
        sv.dRequired  = 0;
        sv.dInstalled = 0;
        sv.pbFilename = NULL;
        }
    if (psv != NULL) *psv = sv;
    return sv.dInstalled;
    }

// -----------------------------------------------------------------

DWORD WINAPI SbsRtlVersionTestW (DWORD        dRequired,
                                 PSBS_VERSION psv)
    {
    SBS_VERSION sv;

    sv.dRequired  = dRequired;
    sv.dInstalled = SbsRtlVersion ();
    sv.pwFilename = gawFilename;

    if (sv.dRequired <= sv.dInstalled)
        {
        sv.dRequired  = 0;
        sv.dInstalled = 0;
        sv.pwFilename = NULL;
        }
    if (psv != NULL) *psv = sv;
    return sv.dInstalled;
    }

// =================================================================
// DLL MANAGEMENT
// =================================================================

BOOL WINAPI DllInitialize (VOID)
    {
    WNDCLASSA wcA;
    WNDCLASSW wcW;

    _printf_init (gabCaption, gawCaption);

    InitCommonControls ();

    ghfDefault = GetStockObject (SYSTEM_FONT              );
    ghbDefault = GetStockObject (SBS_BRUSH_DEFAULT        );
    ghiDefault = LoadIconA      (ghInstance, SA(MAIN_ICON));
    ghcDefault = LoadCursorA    (NULL,       IDCA (ARROW) );
    ghcWait    = LoadCursorA    (NULL,       IDCA (WAIT ) );

    ghfLarge   = SbsFontObjectA (NULL, "MS Sans Serif",
                                 -13, TRUE,  FALSE, TRUE);

    ghfSmall   = SbsFontObjectA (NULL, "MS Sans Serif",
                                 -11, TRUE,  FALSE, TRUE);

    ghfLight   = SbsFontObjectA (NULL, "MS Sans Serif",
                                 -11, FALSE, FALSE, TRUE);

    wcA.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    wcA.cbClsExtra    = 0;
    wcA.cbWndExtra    = 0;
    wcA.hInstance     = ghInstance;
    wcA.hIcon         = ghiDefault;
    wcA.hCursor       = ghcDefault;
    wcA.hbrBackground = ghbDefault;
    wcA.lpszMenuName  = NULL;
    wcA.lpszClassName = gabSbsClass;
    wcA.lpfnWndProc   = SbsThreadWindow;

    RegisterClassA (&wcA);

    wcW.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    wcW.cbClsExtra    = 0;
    wcW.cbWndExtra    = 0;
    wcW.hInstance     = ghInstance;
    wcW.hIcon         = ghiDefault;
    wcW.hCursor       = ghcDefault;
    wcW.hbrBackground = ghbDefault;
    wcW.lpszMenuName  = NULL;
    wcW.lpszClassName = gawSbsClass;
    wcW.lpfnWndProc   = SbsThreadWindow;

    RegisterClassW (&wcW);

    _random_reset (NULL, NULL);

    return (ghfDefault != NULL) &&
           (ghbDefault != NULL) &&
           (ghiDefault != NULL) &&
           (ghcDefault != NULL) &&
           (ghcWait    != NULL) &&
           (ghfLarge   != NULL) &&
           (ghfSmall   != NULL) &&
           (ghfLight   != NULL);
    }

// -----------------------------------------------------------------

BOOL WINAPI DllCleanUp (VOID)
    {
    ghfLarge = SbsFontDelete (ghfLarge);
    ghfSmall = SbsFontDelete (ghfSmall);
    ghfLight = SbsFontDelete (ghfLight);

    UnregisterClassA (gabSbsClass, ghInstance);
    UnregisterClassW (gawSbsClass, ghInstance);

    _printf_exit ();
    return TRUE;
    }

// -----------------------------------------------------------------

BOOL WINAPI DllMain (HINSTANCE hInstance,
                     DWORD     dReason,
                     PVOID     pReserved)
    {
    BOOL fOk = TRUE;

    if (dReason == DLL_PROCESS_ATTACH)
        {
        ghInstance = hInstance;
        fOk = DllInitialize ();
        }
    if (dReason == DLL_PROCESS_DETACH)
        {
        fOk = DllCleanUp ();
        }
    return fOk;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
